async function loadPanelDataNoCOmpress(id, panelId, formId, objectKey) {
        // Show loading started toast
        toast(
            `⏳ Loading ${panelId}...`,
            { pos: "TR", slide: "rtl" },
            { background: "#2196F3" },
            1500,
        );

        const panel = document.getElementById(panelId);
    // First try to load from localStorage
    if (loadFromLocalStorage(id)) {
        console.log(`✅ Loaded ${panelId} data from localStorage`);
        processCachedDataFromLocalStorage(panelId, formId, objectKey);
        toast(
            `✅ ${panelId} loaded from cache`,
            { pos: "BR", slide: "rtl" },
            { background: "#4CAF50" },
            1500,
        );
        return; // Exit after successful localStorage load
    }
        if (!panel) {
            createPanelWrapper(panelId, formId);
            const targetEl = document.getElementById(formId);
            if (targetEl) {
                targetEl.innerHTML = `<div style="
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    margin: auto;
                    color: white;
                    min-height: 500px;" class="loader">.</div>`;
            }
        }

        if (!navigator.onLine) {
            const message = `⚠️ Offline — loading ${panelId} from cache if available`;
            console.warn(message);
            toast(
                message,
                { pos: "TR", slide: "rtl" },
                { background: "orange" },
                2000,
            );

            const cached = responseCache.get(id);
            if (cached && cached.data) {
                console.log(`🛠 Using cached data for ${panelId}`);
                processCachedData(cached.data, panelId, formId, objectKey);
                toast(
                    `✅ ${panelId} loaded from cache`,
                    { pos: "BR", slide: "rtl" },
                    { background: "#4CAF50" },
                    1500,
                );
                return;
            }
            toast(
                `❌ No cached data for ${panelId}`,
                { pos: "BR", slide: "rtl" },
                { background: "#F44336" },
                2000,
            );
            return;
        }

        if (!id) {
            const message = "⚠️ ID is required";
            console.warn(message);
            toast(
                message,
                { pos: "TR", slide: "rtl" },
                { background: "orange" },
                2000,
            );
            return;
        }

        const url = `https://j.kunok.com/api622.php?id=${encodeURIComponent(id)}`;
        console.log(`🚀 Start loading for ${panelId}`);

        try {
            const startTime = Date.now();
            const cached = responseCache.get(id);
            const headers = cached?.etag
                ? { "If-None-Match": cached.etag }
                : {};

            // Show fetching toast
            toast(
                `🔍 Fetching ${panelId} data...`,
                { pos: "TR", slide: "rtl" },
                { background: "#2196F3" },
                1000,
            );

            const response = await fetch(url, { headers });

            if (response.status === 304) {
                console.log(`✅ ${panelId} not modified, using cached data`);
                if (cached && cached.data) {
                    processCachedData(cached.data, panelId, formId, objectKey);
                    const loadTime = Date.now() - startTime;
                    console.log(
                        `✅ ${panelId} load completed in ${loadTime}ms`,
                    );
                    toast(
                        `✅ ${panelId} loaded (cached) in ${loadTime}ms`,
                        { pos: "TR", slide: "rtl" },
                        { background: "#4CAF50" },
                        1500,
                    );
                    return;
                }
                throw new Error("No cached data available for 304 response");
            }

            // Show parsing toast
            toast(
                `📦 Processing ${panelId} data...`,
                { pos: "CR", slide: "rtl" },
                { background: "#2196F3" },
                1000,
            );

            const text = await response.text();
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${text}`);
            }

            const json = JSON.parse(text);
            const data = json?.data;
            if (!data) {
                const message = `⚠️ No 'data' field in ${panelId} response`;
                console.warn(message);
                toast(
                    message,
                    { pos: "TR", slide: "rtl" },
                    { background: "orange" },
                    2000,
                );
                return;
            }

            // Store in cache
            const etag = response.headers.get("ETag");
            if (etag) {
                responseCache.set(id, { etag, data });
            }
// Save to localStorage
        saveToLocalStorage(id);
            processCachedData(data, panelId, formId, objectKey);
            const loadTime = Date.now() - startTime;
            console.log(`✅ ${panelId} load completed in ${loadTime}ms`);
            toast(
                `✅ ${panelId} loaded in ${loadTime}ms`,
                { pos: "CR", slide: "rtl" },
                { background: "#4CAF50" },
                1500,
            );
        } catch (e) {
            console.error(`❌ ${panelId} load failed:`, e.message);
            toast(
                `❌ ${panelId} load failed: ${e.message}`,
                { pos: "CR", slide: "rtl" },
                { background: "#F44336" },
                3000,
            );
        }
    }
function processCachedData(data, panelId, formId, objectKey) {
        const tryParse = (str) => {
            try {
                return JSON.parse(str);
            } catch {
                return {};
            }
        };

        bucket = tryParse(data.bucket || "{}");
        cup = tryParse(data.cup || "{}");
        board = tryParse(data.board || "{}");

        const objectMap = { bucket, cup };
        const targetObj = objectMap[objectKey];

        const panel = document.getElementById(panelId);
        if (!panel) {
            createPanelWrapper(panelId, formId);
        } else {
            genform(formId, targetObj);
        }
    }
        
// New helper function for localStorage data processing
function processCachedDataFromLocalStorage(panelId, formId, objectKey) {
    const objectMap = { bucket, cup };
    const targetObj = objectMap[objectKey];

    const panel = document.getElementById(panelId);
    if (!panel) {
        createPanelWrapper(panelId, formId);
    } else {
        genform(formId, targetObj);
    }
    
    // Trigger any necessary UI updates
    if (document.getElementById("summan")) {
        renderSchemaStructureToSumman(bucket, "gform", window.lang);
    }
    if (document.getElementById("jumperhelper")) {
        updateJumperHelperWithUnanswered(QSummary(bucket).summary, bucket);
    }
}    
// Helper function to process data (cached or fresh)
    function processCachedData(data, panelId, formId, objectKey) {
        const tryParse = (str) => {
            try {
                return JSON.parse(str);
            } catch {
                return {};
            }
        };

        bucket = tryParse(data.bucket || "{}");
        cup = tryParse(data.cup || "{}");
        board = tryParse(data.board || "{}");

        const objectMap = { bucket, cup };
        const targetObj = objectMap[objectKey];

        const panel = document.getElementById(panelId);
        if (!panel) {
            createPanelWrapper(panelId, formId);
        } else {
            genform(formId, targetObj);
        }
    }
    
    async function saveToServerM(id) {
        // Early exit checks
        if (!navigator.onLine) {
            console.warn("⚠️ Offline — cannot save to server");
            toast(
                "⚠️ Offline - cannot save",
                { pos: "TR", slide: "ttb" },
                { background: "orange" },
            );
            return false;
        }

        if (!id) {
            console.warn("⚠️ ID is required");
            toast(
                "⚠️ ID is required",
                { pos: "TR", slide: "ttb" },
                { background: "orange" },
            );
            return false;
        }

        // Check for actual data content
        const hasValidData = () => {
            const hasBucket = bucket && Object.keys(bucket).length > 0;
            const hasCup = cup && Object.keys(cup).length > 0;
            const hasBoard = board && Object.keys(board).length > 0;
            return hasBucket || hasCup || hasBoard;
        };

        if (!hasValidData()) {
            toast(
                "⚠️ No data to save",
                { pos: "TR", slide: "rtl" },
                { background: "red" },
            );
            return false;
        }

        // Create progress panel
        createPanel("verticalProgressPanel2", "TR", "ctc", 60, 220, {
            title: "SAVE",
            verticalProgressBar: 100,
            noHeader: true,
            progressBar: 100,
            noFooter: true,
            bodyContent: `
            <div class="progress-container" style="display: flex; flex-direction: column; align-items: center;">
                <span class="progress-label">Saving...</span>
                <div class="progress-bar vertical">
                    <div class="progress-fill vertical" style="height: 0%"></div>
                </div>
            </div>`,
        });
        setTimeout(() => {
            toast(
                " Start saving ...",
                { pos: "TR", slide: "rtl" },
                { background: "red", width: 200 },
                1000,
            );
            moveXY("verticalProgressPanel2", window.innerWidth - 100, 80);
        }, 50);

        const panel = document.getElementById("verticalProgressPanel2");
        panel.style.left = window.innerWidth - 70 + "px";
        panel.style.top = "10px";

        try {
            const startTime = Date.now();

            // Prepare payload with all data
            const payload = {
                data: {
                    id,
                    name: id,
                    bucket: JSON.stringify(bucket || {}),
                    cup: JSON.stringify(cup || {}),
                    board: JSON.stringify(board || {}),
                    timestamp: new Date().toISOString(),
                },
            };

            // Get cached ETag if available
            const getUrl = `https://j.kunok.com/api622.php?id=${encodeURIComponent(id)}`;
            const cachedETag = sessionStorage.getItem(`etag:${getUrl}`);
            const headers = {
                "Content-Type": "application/json",
                ...(cachedETag && { "If-Match": cachedETag }),
            };

            // Show saving progress
            updateProgress(panel, 30, "Uploading data...");

            const response = await fetch("https://j.kunok.com/api622.php", {
                method: "POST",
                headers: headers,
                body: JSON.stringify(payload),
            });

            if (!response.ok) {
                const text = await response.text();
                if (response.status === 412) {
                    throw new Error(
                        "Data was modified by another user. Please reload and try again.",
                    );
                }
                throw new Error(`HTTP ${response.status}: ${text}`);
            }

            // Update progress
            updateProgress(panel, 70, "Verifying save...");

            // Update ETag if available
            const etag = response.headers.get("ETag");
            if (etag) {
                sessionStorage.setItem(`etag:${getUrl}`, etag);
            }

            // Verify the save by fetching the data back
            const verifyResponse = await fetch(getUrl);
            if (!verifyResponse.ok) {
                throw new Error("Failed to verify saved data");
            }

            const verifiedData = await verifyResponse.json();
            if (!verifiedData.data) {
                throw new Error("No data in verification response");
            }

            // Compare the saved data with what we sent
            const isDataValid = compareSavedData(
                payload.data,
                verifiedData.data,
            );
            if (!isDataValid) {
                throw new Error("Saved data verification failed");
            }

            // Update progress to complete
            updateProgress(panel, 100, "Save verified!");

            console.log(`✅ Save completed in ${Date.now() - startTime}ms`);
            toast(
                "✅ Saved successfully",
                { pos: "TR", slide: "ttb" },
                { background: "green" },
            );

            // Also save to local storage as backup
            saveToLocalStorage(id);

            return true;
        } catch (e) {
            console.error("❌ Save failed:", e.message);
            toast(
                `❌ Save failed: ${e.message}`,
                { pos: "TR", slide: "ttb" },
                { background: "red" },
            );
            return false;
        } finally {
            // Close progress panel after a delay
            setTimeout(() => closePanel("verticalProgressPanel2"), 1500);
        }
    }
function loadFromLocalStorage(userId) {
        console.log("loadFromLocalStorage");
        try {
            const savedData = localStorage.getItem(`userData_${userId}`);
            if (!savedData) {
                console.log("ℹ️ No local data found for user:", userId);
                return false;
            }

            const data = JSON.parse(savedData);

            // Update the global objects
            bucket = data.bucket || {};
            cup = data.cup || {};
            board = data.board || {};

            console.log("✅ Loaded from LocalStorage for user:", userId);
            return true;
        } catch (e) {
            console.error("❌ LocalStorage load failed:", e);
            return false;
        }
    }
    function saveToLocalStorage(userId) {
        console.log("saveToLocalStorage");
        try {
            const data = {
                bucket: bucket || {},
                cup: cup || {},
                board: board || {},
                lastSaved: new Date().toISOString(),
            };

            localStorage.setItem(`userData_${userId}`, JSON.stringify(data));
            console.log("✅ Saved to LocalStorage for user:", userId);
            toast(
                "✅ Saved to LocalStorage",
                { pos: "TR", slide: "ttb" },
                { background: "green", color: "white" },
                1000,
            );
        } catch (e) {
            console.error("❌ LocalStorage save failed:", e);
            toast(
                "❌ LocalStorage save failed",
                { pos: "TR", slide: "ttb" },
                { background: "red", color: "white" },
                1000,
            );
        }
    }    