function addStyle() {
    const styleId = 'timeline-styles';
    let styleElement = document.getElementById(styleId);
    if (!styleElement) {
        styleElement = document.createElement('style');
        styleElement.id = styleId;
        document.head.appendChild(styleElement);
    }

    styleElement.textContent = `
        body {
            font-family: sans-serif;
            padding: 40px;
            display: flex;
            gap: 80px;
            justify-content: center;
        }

        .timeline-label {
            font-size: 12px;
            //color: #333;
        }

        .timeline-node-circle {
            width: 14px;
            height: 14px;
            border-radius: 50%;
            border: 3px solid white;
            box-shadow: 0 0 0 1px #999;
            background: #ccc;
            margin: 0 auto;
        }

        #timeline-horizontal {
            position: relative;
            width: 400px;
            height: 90px;
        }

        #timeline-horizontal .line {
            position: absolute;
            top: 30px;
            left: 0;
            right: 0;
            height: 2px;
            background: #ccc;
            z-index: 1;
        }

        #timeline-horizontal .progress {
            position: absolute;
            top: 30px;
            left: 0;
            height: 2px;
            background: #4caf50;
            width: 0%;
            transition: width 1.5s ease;
            z-index: 2;
        }

        #timeline-horizontal .node {
            position: absolute;
            top: 21px;
            display: flex;
            flex-direction: column;
            align-items: center;
            z-index: 3;
            width: 14px;
        }

        #timeline-horizontal .label {
            margin-top: 4px;
            text-align: center;
            min-width: 60px;
            max-width: 100px;
        }

        #timeline-horizontal .tooltip {
            position: absolute;
            top: 21px;
            display: inline-block;
            z-index: 4;
        }

        #timeline-horizontal .tooltiptext {
            width: 120px;
            color: #fff;
            text-align: center;
            border-radius: 6px;
            padding: 5px 0;
            position: absolute;
            z-index: 1;
            top: 125%;
            left: calc(50% + 10px); /* Shifted 10px right */
            transform: translateX(-50%);
            background-color: black; /* Default */
            margin-top:60px;
        }

        #timeline-horizontal .tooltiptext.red { background-color: #ff4444; }
        #timeline-horizontal .tooltiptext.blue { background-color: #4444ff; }
        #timeline-horizontal .tooltiptext.green { background-color: #44aa44; }
        #timeline-horizontal .tooltiptext.purple { background-color: #aa44aa; }
        #timeline-horizontal .tooltiptext.orange { background-color: #ffaa44; }
        #timeline-horizontal .tooltiptext.gray { background-color: #666666; }

        #timeline-horizontal .tooltiptext.bottom::after {
            content: "";
            position: absolute;
            bottom: 100%;
            left: 50%;
            margin-left: -5px;
            border-width: 5px;
            border-style: solid;
            
        }
        #timeline-horizontal .tooltiptext.bottom.red::after { border-color: transparent transparent #ff4444 transparent; }
        #timeline-horizontal .tooltiptext.bottom.blue::after { border-color: transparent transparent #4444ff transparent; }
        #timeline-horizontal .tooltiptext.bottom.green::after { border-color: transparent transparent #44aa44 transparent; }
        #timeline-horizontal .tooltiptext.bottom.purple::after { border-color: transparent transparent #aa44aa transparent; }
        #timeline-horizontal .tooltiptext.bottom.orange::after { border-color: transparent transparent #ffaa44 transparent; }
        #timeline-horizontal .tooltiptext.bottom.gray::after { border-color: transparent transparent #666666 transparent; }
        #timeline-horizontal .tooltiptext.bottom::after { border-color: transparent transparent black transparent; } /* Default */

        #timeline-vertical {
            position: relative;
            height: 300px;
            width: 200px;
        }

        #timeline-vertical .line {
            position: absolute;
            top: 0;
            bottom: 0;
            left: 9px;
            width: 2px;
            background: #ccc;
            z-index: 1;
        }

        #timeline-vertical .progress {
            position: absolute;
            top: 0;
            left: 9px;
            width: 2px;
            height: 0%;
            background: #4caf50;
            transition: height 1.5s ease;
            z-index: 2;
        }

        #timeline-vertical .node {
            position: absolute;
            left: 0;
            display: flex;
            align-items: center;
            gap: 8px;
            transform: translateY(-50%);
            z-index: 3;
        }

        #timeline-vertical .label {
            margin-left: 6px;
        }

        #timeline-vertical .tooltip {
            position: relative;
            display: inline-block;
        }

        #timeline-vertical .tooltiptext {
            width: 120px;
            color: #fff;
            text-align: center;
            border-radius: 6px;
            padding: 5px 0;
            position: absolute;
            z-index: 1;
            top: -10px; 
            left: 110%;
            transform: translateY(-50%);
            background-color: black; /* Default */
            margin-left:100px;
        }

        #timeline-vertical .tooltiptext.red { background-color: #ff4444; }
        #timeline-vertical .tooltiptext.blue { background-color: #4444ff; }
        #timeline-vertical .tooltiptext.green { background-color: #44aa44; }
        #timeline-vertical .tooltiptext.purple { background-color: #aa44aa; }
        #timeline-vertical .tooltiptext.orange { background-color: #ffaa44; }
        #timeline-vertical .tooltiptext.gray { background-color: #666666; }

        #timeline-vertical .tooltiptext.right::after {
            content: "";
            position: absolute;
            top: 50%;
            right: 100%;
            margin-top: -5px;
            border-width: 5px;
            border-style: solid;
        }
        #timeline-vertical .tooltiptext.right.red::after { border-color: transparent #ff4444 transparent transparent; }
        #timeline-vertical .tooltiptext.right.blue::after { border-color: transparent #4444ff transparent transparent; }
        #timeline-vertical .tooltiptext.right.green::after { border-color: transparent #44aa44 transparent transparent; }
        #timeline-vertical .tooltiptext.right.purple::after { border-color: transparent #aa44aa transparent transparent; }
        #timeline-vertical .tooltiptext.right.orange::after { border-color: transparent #ffaa44 transparent transparent; }
        #timeline-vertical .tooltiptext.right.gray::after { border-color: transparent #666666 transparent transparent; }
        #timeline-vertical .tooltiptext.right::after { border-color: transparent black transparent transparent; } /* Default */
    `;
}

function toast(message, { pos = "TR", slide = "ttb" }, { background = "red", color = "white" }, duration = 3000) {
    const toast = document.createElement("div");
    toast.textContent = message;
    toast.style.cssText = `
        position: fixed;
        ${pos === "TR" ? "top: 10px; right: 10px;" : "top: 10px; left: 10px;"}
        max-width: 300px;
        padding: 8px 12px;
        font-size: 14px;
        font-family: sans-serif;
        background: ${background};
        color: ${color};
        border-radius: 4px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        z-index: 1000;
        opacity: 0;
        transition: opacity 0.3s ease, transform 0.3s ease;
        ${slide === "ttb" ? "transform: translateY(-20px);" : ""}
    `;
    document.body.appendChild(toast);
    setTimeout(() => {
        toast.style.opacity = "1";
        toast.style.transform = slide === "ttb" ? "translateY(0)" : "";
    }, 10);
    setTimeout(() => {
        toast.style.opacity = "0";
        setTimeout(() => toast.remove(), 300);
    }, duration);
}

function renderTimelines() {
    const horizontalContainer = document.getElementById('timeline-horizontal');
    const verticalContainer = document.getElementById('timeline-vertical');

    if (!horizontalContainer || !verticalContainer) {
        toast('❌ Timeline containers not found', { pos: 'TR', slide: 'ttb' }, { background: 'red', color: 'white' }, 3000);
        return;
    }

    const nodes = [
        { percent: 0, label: '0%<br>Start', color: 'green' },
        { percent: 25, label: '25%<br>Design', color: '#ccc' },
        { percent: 50, label: '50%<br>Build', color: '#ccc' },
        { percent: 75, label: '75%<br>Test', color: '#ccc' },
        { percent: 100, label: '100%<br>Done', color: 'red' }
    ];

    horizontalContainer.innerHTML = `
        <div class="line"></div>
        <div class="progress" id="h-progress"></div>
        ${nodes.map((node, index) => `
            <div class="node" style="left: ${node.percent}%">
                <div class="timeline-node-circle" style="background: ${node.color}"></div>
                <div class="timeline-label label">${node.label}</div>
            </div>
        `).join('')}
    `;

    verticalContainer.innerHTML = `
        <div class="line"></div>
        <div class="progress" id="v-progress"></div>
        ${nodes.map((node, index) => `
            <div class="node" style="top: ${node.percent}%">
                <div class="timeline-node-circle" style="background: ${node.color}"></div>
                <div class="timeline-label">${node.label.replace('<br>', ' ')}</div>
            </div>
        `).join('')}
    `;

    console.log(`renderTimelines: horizontal nodes=${horizontalContainer.querySelectorAll('.node').length}, vertical nodes=${verticalContainer.querySelectorAll('.node').length}`);
    setTimeout(() => {
        const hProgress = document.getElementById('h-progress');
        const vProgress = document.getElementById('v-progress');
        if (hProgress) hProgress.style.width = '75%';
        if (vProgress) vProgress.style.height = '75%';
    }, 300);
}

function updateNode(timelineId, nodeIndex, percent, label, color = null) {
    const container = document.getElementById(timelineId);
    if (!container) {
        toast(`❌ Timeline "${timelineId}" not found`, { pos: 'TR', slide: 'ttb' }, { background: 'red', color: 'white' }, 3000);
        return;
    }
    const nodes = container.querySelectorAll('.node');
    console.log(`updateNode: timelineId=${timelineId}, nodeIndex=${nodeIndex}, nodes.length=${nodes.length}`);
    if (nodeIndex < 0 || nodeIndex >= nodes.length) {
        toast(`❌ Invalid node index: ${nodeIndex} (available: 0-${nodes.length - 1})`, { pos: 'TR', slide: 'ttb' }, { background: 'red', color: 'white' }, 3000);
        return;
    }
    const node = nodes[nodeIndex];
    const isHorizontal = timelineId === 'timeline-horizontal';
    node.style[isHorizontal ? 'left' : 'top'] = `${percent}%`;
    const labelElement = node.querySelector('.timeline-label');
    if (labelElement) {
        labelElement.innerHTML = isHorizontal ? label : label.replace('<br>', ' ');
    }
    if (color) {
        const circle = node.querySelector('.timeline-node-circle');
        if (circle) circle.style.background = color;
    }
    // Clear existing tooltips
    container.querySelectorAll('.tooltip').forEach(el => el.remove());
    // Add tooltip
    const tooltip = document.createElement('div');
    tooltip.className = 'tooltip';
    const colorClass = ['red', 'blue', 'green', 'purple', 'orange', 'gray'].includes(color) ? color : '';
    tooltip.innerHTML = `<div class="tooltiptext ${isHorizontal ? 'bottom' : 'right'} ${colorClass}">You are here!</div>`;
    if (isHorizontal) {
        node.insertAdjacentElement('beforebegin', tooltip);
        tooltip.style.left = `${percent}%`;
    } else {
        const circle = node.querySelector('.timeline-node-circle');
        if (circle) circle.appendChild(tooltip);
    }
}

function initialize() {
    addStyle();
    renderTimelines();
}

document.addEventListener('DOMContentLoaded', initialize);
//window.updateNode = updateNode;