const host = "https://i.kunok.com/";

function displayMessage(message, isError = false, isInfo = false) {
    let msgDiv = document.getElementById("message");
    if (!msgDiv) {
        msgDiv = document.createElement("div");
        msgDiv.id = "message";
        document.body.insertBefore(msgDiv, document.body.firstChild);
    }
    msgDiv.textContent = message;
    msgDiv.className = "message " + (isError ? "error" : isInfo ? "info" : "success");
    if (isInfo) {
        setTimeout(() => {
            if (msgDiv.className.includes("info")) {
                msgDiv.textContent = "";
                msgDiv.className = "message";
            }
        }, 3000);
    }
}

async function postto(apiUrl = "", payload = {}) {
    try {
        const response = await fetch(apiUrl, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: payload }),
        });
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error("Request failed:", error);
        displayMessage(error.message, true);
        throw error;
    }
}

function closePanel(id) {
    const panel = document.getElementById(id);
    if (panel) panel.remove();
}

async function listItemsPanel() {
    try {
        displayMessage("Loading items...", false, true);
        const response = await fetch(host + "api62.php?action=list_all&fields=id,name,flag,timestamp,board");
        const result = await response.json();
        const items = result.data || [];

        if (items.length === 0) {
            displayMessage("No items found", false, true);
            return;
        }

        let tableHTML = `
            <style>
                table { width: 100%; border-collapse: collapse; }
                th, td { padding: 8px; border: 1px solid #ddd; text-align: left; }
                th { background-color: #f2f2f2; }
                .action-btn { padding: 4px 8px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; }
                .edit-btn { background-color: #4CAF50; color: white; }
                .delete-btn { background-color: #f44336; color: white; }
            </style>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Flag</th>
                        <th>Timestamp</th>
                        <th>Board</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
        `;

        items.forEach(item => {
            const flagText = item.flag == 1 ? 'Active' : item.flag == 0 ? 'Suspended' : item.flag == 3 ? 'Wait' : item.flag;
            tableHTML += `
                <tr>
                    <td>${item.id}</td>
                    <td>${item.name || ''}</td>
                    <td>${flagText}</td>
                    <td>${item.timestamp}</td>
                    <td>${item.board || ''}</td>
                    <td>
                        <button class="action-btn edit-btn" onclick="editItemPanel('${item.id}')">Edit</button>
                        <button class="action-btn delete-btn" onclick="deleteItem('${item.id}')">Delete</button>
                    </td>
                </tr>
            `;
        });

        tableHTML += `
                </tbody>
            </table>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("listItemsPanel", "CC", "ctc", 800, 600, {
            bodyContent: tableHTML,
            noFooter: true,
            title: 'List Items'
        });

    } catch (error) {
        displayMessage(error.message, true);
    }
}

async function addItemPanel() {
    const formHTML = `
        <style>
            .form-row { display: flex; gap: 10px; }
            .form-row > div { flex: 1; }
            textarea { width: 100%; height: 100px; resize: vertical; }
        </style>
        <form id="addItemForm">
            <div><label for="addName">Name:</label> <textarea id="addName"></textarea></div>
            <div><label for="addAge">Age:</label> <input type="number" id="addAge" value=""></div>
            <div>
                <label for="addFlag">Flag:</label>
                <select id="addFlag">
                    <option value="1">1 (Active)</option>
                    <option value="0">0 (Suspended)</option>
                    <option value="3">3 (Wait)</option>
                </select>
            </div>
            <div><label for="addBucket">Bucket:</label> <textarea id="addBucket"></textarea></div>
            <div><label for="addCup">Cup:</label> <textarea id="addCup"></textarea></div>
            <div><label for="addBoard">Board:</label> <textarea id="addBoard"></textarea></div>
            <div><label for="addKeeper">Keeper:</label> <textarea id="addKeeper"></textarea></div>
            <button type="submit">Add Item</button>
        </form>
        <div id="message" class="message"></div>
    `;

    if (typeof createPanel !== "function") {
        console.error("Missing createPanel(). Please include panel system.");
        return;
    }

    createPanel("addItemPanel", "CC", "ctc", 400, 600, {
        bodyContent: formHTML,
        noFooter: true,
        title: 'Add Item'
    });

    const form = document.getElementById("addItemForm");
    if (form) {
        form.addEventListener("submit", async (e) => {
            e.preventDefault();
            const payload = {
                name: document.getElementById("addName").value,
                age: parseInt(document.getElementById("addAge").value) || 0,
                flag: parseInt(document.getElementById("addFlag").value),
                bucket: document.getElementById("addBucket").value,
                cup: document.getElementById("addCup").value,
                board: document.getElementById("addBoard").value,
                keeper: document.getElementById("addKeeper").value
            };

            try {
                const response = await postto(host + "aui.php?action=add", payload);
                displayMessage(response.message || "Item added successfully");
                form.reset();
                closePanel("addItemPanel");
                listItemsPanel();
            } catch (error) {
                console.error("Add item error:", error);
                displayMessage(`Error adding item: ${error.message}`, true);
            }
        });
    }
}

async function editItemPanel(itemId) {
    try {
        displayMessage("Loading item data...", false, true);
        const response = await fetch(host + `api622.php?id=${encodeURIComponent(itemId)}`);
        const result = await response.json();
        // Handle different possible response structures
        const item = result.data && result.data.length > 0 ? result.data[0] : result.item || result;

        if (!item || !item.id) {
            displayMessage("Item not found", true);
            return;
        }

        const formHTML = `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                textarea { width: 100%; height: 100px; resize: vertical; }
            </style>
            <form id="editItemForm">
                <div><label for="editId">ID:</label> <textarea id="editId" readonly>${item.id}</textarea></div>
                <div><label for="editName">Name:</label> <textarea id="editName">${item.name || ''}</textarea></div>
                <div><label for="editAge">Age:</label> <input type="number" id="editAge" value="${item.age || ''}"></div>
                <div>
                    <label for="editFlag">Flag:</label>
                    <select id="editFlag">
                        <option value="1" ${item.flag == 1 ? 'selected' : ''}>1 (Active)</option>
                        <option value="0" ${item.flag == 0 ? 'selected' : ''}>0 (Suspended)</option>
                        <option value="3" ${item.flag == 3 ? 'selected' : ''}>3 (Wait)</option>
                    </select>
                </div>
                <div><label for="editBucket">Bucket:</label> <textarea id="editBucket">${item.bucket || ''}</textarea></div>
                <div><label for="editCup">Cup:</label> <textarea id="editCup">${item.cup || ''}</textarea></div>
                <div><label for="editBoard">Board:</label> <textarea id="editBoard">${item.board || ''}</textarea></div>
                <div><label for="editKeeper">Keeper:</label> <textarea id="editKeeper">${item.keeper || ''}</textarea></div>
                <button type="submit">Update Item</button>
            </form>
            <div id="message" class="message"></div>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("editItemPanel", "CC", "ctc", 400, 600, {
            bodyContent: formHTML,
            noFooter: true,
            title: 'Edit Item'
        });

        const form = document.getElementById("editItemForm");
        if (form) {
            form.addEventListener("submit", async (e) => {
                e.preventDefault();
                const payload = {
                    id: itemId,
                    name: document.getElementById("editName").value,
                    age: parseInt(document.getElementById("editAge").value) || 0,
                    flag: parseInt(document.getElementById("editFlag").value),
                    bucket: document.getElementById("editBucket").value,
                    cup: document.getElementById("editCup").value,
                    board: document.getElementById("editBoard").value,
                    keeper: document.getElementById("editKeeper").value
                };

                try {
                    const response = await postto(host + "aui.php?action=update", payload);
                    displayMessage(response.message || `Item ${itemId} updated successfully`);
                    closePanel("editItemPanel");
                    listItemsPanel();
                } catch (error) {
                    console.error("Update item error:", error);
                    displayMessage(`Error updating item ${itemId}: ${error.message}`, true);
                }
            });
        }

    } catch (error) {
        displayMessage(error.message, true);
    }
}

async function deleteItem(itemId) {
    if (!confirm(`Are you sure you want to delete item ${itemId}?`)) {
        return;
    }

    try {
        const response = await postto(host + "aui.php?action=del", { id: itemId });
        displayMessage(response.message || `Item ${itemId} deleted successfully`);
        closePanel("listItemsPanel");
        listItemsPanel();
    } catch (error) {
        console.error("Delete item error:", error);
        displayMessage(`Error deleting item ${itemId}: ${error.message}`, true);
    }
}