window.debug = false;

// Utility Functions
function displayMessage(message, isError = false, isInfo = false) {
  const msgDiv = document.getElementById("message");
  msgDiv.textContent = message;
  msgDiv.className =
    "message " + (isError ? "error" : isInfo ? "info" : "success");
  if (isInfo) {
    setTimeout(() => {
      if (msgDiv.className.includes("info")) {
        msgDiv.textContent = "";
        msgDiv.className = "message";
        domuserupdate = document.getElementById("userupdate");
        //domuserupdate.style.display = "none";
      }
    }, 3000);
  }
}

function logoset(targetId, previewId = "updateImgPreview") {
  const stringText = window.imgsrc;
  const targetInput = document.getElementById(targetId);
  const previewDiv = document.getElementById(previewId);
  const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';
  if (targetInput && typeof stringText !== "undefined") {
    if (/^https?:\/\//.test(stringText)) {
      targetInput.value = stringText;
      if (previewDiv) {
        previewDiv.innerHTML = `<img src="${stringText}" alt="preview" style="max-width: 100px; max-height: 100px; object-fit: cover;">`;
        const img = previewDiv.querySelector('img');
        img.onerror = () => {
          console.warn("Failed to load preview image, reverting to default.");
          img.src = defaultAvatar;
        };
      }
    } else {
      console.warn("Invalid image URL in logoset, using default:", stringText);
      targetInput.value = defaultAvatar;
      if (previewDiv) {
        previewDiv.innerHTML = `<img src="${defaultAvatar}" alt="preview" style="max-width: 100px; max-height: 100px; object-fit: cover;">`;
      }
    }
  }
}

function togglePassword(inputId, buttonId) {
  const input = document.getElementById(inputId);
  if (input) {
    input.type = input.type === "password" ? "text" : "password";
    const button = document.getElementById(buttonId);
    if (button)
      button.textContent = input.type === "password" ? "Show" : "Hide";
  }
}

async function postto(apiUrl = "", payload = {}) {
  try {
    const response = await fetch(apiUrl, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ data: payload }),
    });
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(
        errorData.error || `HTTP error! status: ${response.status}`,
      );
    }
    return await response.json();
  } catch (error) {
    console.error("Request failed:", error);
    displayMessage(error.message, true);
    throw error;
  }
}

async function getData(url) {
  try {
    const response = await fetch(url);
    if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
    return await response.json();
  } catch (error) {
    console.error("Get request failed:", error);
    throw error;
  }
}

// Check User Function
async function checkUser(userId) {
  if (!userId) {
    displayMessage("No user ID provided in URL", true);
    return null;
  }
  try {
    displayMessage(`Checking user ${userId}...`, false, true);
    const response = await getData(
      `https://i.kunok.com/aui.php?action=list&id=${encodeURIComponent(userId)}`,
    );
    const user =
      response.data && response.data.length ? response.data[0] : null;
    if (!user) {
      displayMessage(`User ${userId} not found`, true);
      return null;
    }
    displayMessage(
      `User ${userId} found, loading update panel...`,
      false,
      true,
    );
    return userId;
  } catch (error) {
    console.error("Check user error:", error);
    displayMessage(`Error checking user ${userId}: ${error.message}`, true);
    return null;
  }
}

// Update User Panel Function
async function userUpdatePanel(userId) {
  // Add spinner styles if not already present
  addStyle(`
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .loading-spinner {
            border: 4px solid rgba(0, 0, 0, 0.1);
            border-radius: 50%;
            border-top: 4px solid #3498db;
            width: 30px;
            height: 30px;
            animation: spin 1s linear infinite;
        }
    `);

  setTimeout(async () => {
    try {
      displayMessage("Loading user data...", false, true);
      const response = await getData(
        `https://i.kunok.com/aui.php?action=list&id=${encodeURIComponent(userId)}`,
      );
      if (response.error || !response.data || response.data.length === 0) {
        throw new Error(response.error || "User not found");
      }

      const user = response.data[0];
      const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';
      let userImg = defaultAvatar;
      if (user.img && /^https?:\/\//.test(user.img)) {
        userImg = user.img;
      } else if (user.img) {
        console.warn("Invalid image URL from API, using default:", user.img);
      }

      localStorage.setItem("userid", user.id);
      localStorage.setItem("user_name", user.name);
      localStorage.setItem("flag", user.flag);
      localStorage.setItem("level", user.level);
      localStorage.setItem("img", userImg);

      const formHTML = `
                <div class="logo-controls">
                    <div id="updateImgPreview">${userImg ? `<img src="${userImg}" alt="preview" style="max-width: 100px; max-height: 100px; object-fit: cover;">` : ""}</div>
                    <div class="button-group">
                        <button id="updateUploadLogoBtn">Upload logo</button>
                        <button id="updateApplyLogoBtn" hidden>Apply Logo</button>
                    </div>
                </div>
                <form id="updateUserForm">
                    <div><label for="updateId">ID:</label> <input type="text" id="updateId" value="${user.id || ""}" disabled></div>
                    <div><label for="updateName">Name:</label> <input type="text" id="updateName" value="${user.name || ""}"></div>
                    <div><label for="updateUsername">Username:</label> <input type="text" id="updateUsername" value="${user.username || ""}" disabled></div>
                    <div>
                        <label for="updatePassword">Password:</label>
                        <input type="password" id="updatePassword" value="${user.password || ""}">
                        <button type="button" id="updateTogglePasswordBtn">Show</button>
                    </div>
                    <input type="hidden" id="updateFlag" value="${user.flag ?? ""}">
                    <input type="hidden" id="updateLevel" value="${user.level ?? ""}">
                    <input type="hidden" id="updateImg" value="${userImg}">
                    <button type="submit">Update User</button>
                </form><br>
                <div id="message" class="message"></div>
                `;

      const panelBody = document.querySelector("#updateUserPanel .panel-body");
      if (panelBody) {
        panelBody.innerHTML = formHTML;

        // Add onerror handler to preview image
        const previewImg = panelBody.querySelector("#updateImgPreview img");
        if (previewImg) {
          previewImg.onerror = () => {
            console.warn("Failed to load preview image, reverting to default.");
            previewImg.src = defaultAvatar;
          };
        }

        const form = document.getElementById("updateUserForm");
        if (form) {
          form.addEventListener("submit", async (e) => {
            e.preventDefault();
            displayMessage(`Starting to update user ${userId}...`, false, true);
            const payload = {
              id: userId,
              name: document.getElementById("updateName").value || undefined,
              username:
                document.getElementById("updateUsername").value || undefined,
              password:
                document.getElementById("updatePassword").value || undefined,
              flag:
                parseInt(document.getElementById("updateFlag").value) ||
                undefined,
              level:
                parseInt(document.getElementById("updateLevel").value) ||
                undefined,
            };
            const img = document.getElementById("updateImg").value;
            if (img && /^https?:\/\//.test(img)) {
              payload.img = img;
            }
            try {
              const response = await postto(
                "https://i.kunok.com/aui.php?action=update",
                payload,
              );
              displayMessage(
                response.message || `User ${userId} updated successfully`,
              );
              window.user_name = document.getElementById("updateUsername").value;
              localStorage.setItem("img", img || defaultAvatar);
              window.updateAvatar?.(); // Update draggable pill avatar
              form.reset();
              document.getElementById("updateImgPreview").innerHTML = "";
              closePanel("updateUserPanel");
            } catch (error) {
              console.error("Update user error:", error);
              displayMessage(
                `Error updating user ${userId}: ${error.message}`,
                true,
              );
            }
          });

          document
            .getElementById("updateTogglePasswordBtn")
            .addEventListener("click", () =>
              togglePassword("updatePassword", "updateTogglePasswordBtn"),
            );
          document
            .getElementById("updateUploadLogoBtn")
            .addEventListener("click", function () {
              window.logomode = true;
              toggleUploadPanel();
            });

          document
            .getElementById("updateApplyLogoBtn")
            .addEventListener("click", () =>
              logoset("updateImg", "updateImgPreview"),
            );
        }
      }
    } catch (error) {
      console.error("Fetch user error:", error);
      const panelBody = document.querySelector("#updateUserPanel .panel-body");
      if (panelBody) {
        panelBody.innerHTML = '<p class="error">Failed to load user data</p>';
      }
      displayMessage(`Error loading user ${userId}: ${error.message}`, true);
    }
  }, 100);
}

// Main function to show user update panel
async function showUserUpdatePanel() {
  createPanel("updateUserPanel", "T1", "ctc", 400, window.innerHeight - 100, {
    bodyContent: `  <div style="
                display: flex;
                justify-content: center;
                align-items: center;
                height: 300px;
                color: #666;
                font-size: 18px;">                
                <div class="loading-spinner"></div>
                <span style="margin-left: 10px;">Loading user data...</span>
            </div>`,
    noFooter: true,
    radius: "8px",
    title: "Update User",
  });
  const params = new URLSearchParams(window.location.search);
  userId = userid; // Or get from URL params if needed: params.get('userid');

  const validUserId = await checkUser(userId);
  if (validUserId) {
    userUpdatePanel(validUserId);
  }
}
addStyle(`
    #updateUserPanel .message {
            margin-bottom: 20px;
            padding: 10px;
            border-radius: 5px;
        }
        #updateUserPanel form {
            margin-bottom: 10px;
            padding: 15px;
            border: 1px solid #ccc;
            border-radius: 5px;
        }
        #updateUserPanel form div {
            margin-bottom: 10px;
        }
        #updateUserPanel label {
            display: inline-block;
            width: 120px;
            font-weight: bold;
        }
        #updateUserPanel input[type="text"],
        #updateUserPanel input[type="password"] {
            width: 200px;
            padding: 5px;
        }
        #updateUserPanel button {
            padding: 5px 10px;
            cursor: pointer;
        }
        .logo-controls {
            text-align: center;
            margin: 10px 0;
        }
        #updateUserPanel .button-group {
            display: flex;
            gap: 10px;
            justify-content: center;
        }
        #updateImgPreview img {
            max-width: 100px;
            max-height: 100px;
            object-fit: cover;
        }
        .success {
            background-color: #e0f7e9;
            color: #2e7d32;
        }
        .error {
            background-color: #fdecea;
            color: #c62828;
        }
        .info {
            background-color: #e3f2fd;
            color: #1565c0;
        }
    `);