// analog-clock.js
(function() {
  // Track all clocks
  const clocks = [];
  
  // Main function to create a clock
  window.makeAnalogClock = function(clockId, size = 100) {
    const clockEl = document.getElementById(clockId);
    if (!clockEl) {
      console.error('Clock element not found:', clockId);
      return;
    }
    
    // Clear existing content
    clockEl.innerHTML = '';
    
    // Set clock styles
    clockEl.style.position = 'relative';
    clockEl.style.width = size + 'px';
    clockEl.style.height = size + 'px';
    clockEl.style.border = '1px solid #333';
    clockEl.style.borderRadius = '50%';
    clockEl.style.backgroundColor = '#fff';
    clockEl.style.display = 'inline-block';
    
    // Create clock face
    const face = document.createElement('div');
    face.style.position = 'relative';
    face.style.width = '100%';
    face.style.height = '100%';
    clockEl.appendChild(face);
    
    // Create hands
    const hourHand = createHand('hour', size);
    const minuteHand = createHand('minute', size);
    const secondHand = createHand('second', size);
    
    face.appendChild(hourHand);
    face.appendChild(minuteHand);
    face.appendChild(secondHand);
    
    // Add center dot only if clock is larger than 32px
    if (size > 32) {
      const centerDot = document.createElement('div');
      centerDot.style.position = 'absolute';
      centerDot.style.top = '50%';
      centerDot.style.left = '50%';
      centerDot.style.width = Math.max(4, size/20) + 'px';
      centerDot.style.height = Math.max(4, size/20) + 'px';
      centerDot.style.backgroundColor = '#333';
      centerDot.style.borderRadius = '50%';
      centerDot.style.margin = `-${Math.max(2, size/40)}px 0 0 -${Math.max(2, size/40)}px`;
      centerDot.style.zIndex = '10';
      face.appendChild(centerDot);
    }
    
    // Store clock reference
    const clock = {
      id: clockId,
      element: clockEl,
      size: size,
      interval: null
    };
    clocks.push(clock);
    
    // Initial update
    updateClock(clock);
    
    // Set up regular updates
    clock.interval = setInterval(() => {
      updateClock(clock);
    }, 1000);
  };
  
  // Create a clock hand element
  function createHand(type, size) {
    const hand = document.createElement('div');
    hand.style.position = 'absolute';
    hand.style.left = '50%';
    hand.style.bottom = '50%';
    hand.style.transformOrigin = '50% 100%';
    
    switch(type) {
      case 'hour':
        hand.style.width = Math.max(2, size/30) + 'px';
        hand.style.height = (size * 0.3) + 'px';
        hand.style.backgroundColor = '#333';
        hand.style.marginLeft = `-${Math.max(1, size/60)}px`;
        hand.className = 'hour-hand';
        break;
      case 'minute':
        hand.style.width = Math.max(1, size/60) + 'px';
        hand.style.height = (size * 0.45) + 'px';
        hand.style.backgroundColor = '#666';
        hand.style.marginLeft = `-${Math.max(0.5, size/120)}px`;
        hand.className = 'minute-hand';
        break;
      case 'second':
        hand.style.width = Math.max(0.5, size/120) + 'px';
        hand.style.height = (size * 0.5) + 'px';
        hand.style.backgroundColor = 'red';
        hand.style.marginLeft = `-${Math.max(0.25, size/240)}px`;
        hand.className = 'second-hand';
        break;
    }
    
    return hand;
  }
  
  // Update clock hands
  function updateClock(clock) {
    const now = new Date();
    const hours = now.getHours() % 12;
    const minutes = now.getMinutes();
    const seconds = now.getSeconds();
    
    // Calculate angles with smooth movement
    const hourAngle = (hours * 30) + (minutes * 0.5);
    const minuteAngle = (minutes * 6) + (seconds * 0.1);
    const secondAngle = seconds * 6;
    
    // Update hands
    const face = clock.element.querySelector('div');
    face.querySelector('.hour-hand').style.transform = `rotate(${hourAngle}deg)`;
    face.querySelector('.minute-hand').style.transform = `rotate(${minuteAngle}deg)`;
    face.querySelector('.second-hand').style.transform = `rotate(${secondAngle}deg)`;
  }
  
  // Cleanup function
  window.removeAnalogClock = function(clockId) {
    const index = clocks.findIndex(c => c.id === clockId);
    if (index >= 0) {
      clearInterval(clocks[index].interval);
      clocks.splice(index, 1);
    }
  };
})();