(function () {
  function createDraggablePanel(options = {}) {
    const {
      bottom = 100,
      left = 10,
      width = 200,
      collapsedHeight = 70,
      expandedHeight = 200,
      expandedWidth = width,
      avatars = [],
      showExternalAvatar = true,
      externalAvatarURL = 'https://i.pravatar.cc/41',
      expandedContent = '',
        expandFrom = 'bottom',
      toggleOnClick = false
    } = options;

    const panelId = 'miniPanel_' + Math.random().toString(36).substr(2, 5);
    const panel = document.createElement('div');
    panel.id = panelId;

    panel.style.cssText = `
      position: absolute;
  ${expandFrom === 'bottom' ? `bottom: ${bottom}px;` : `top: ${bottom}px;`}
  left: ${left}px;      left: ${left}px;
      width: ${width}px;
      height: ${collapsedHeight}px;
      background: #f0f0f0;
      border: 1px solid #aaa;
      border-radius: 8px 8px 0 0;
      padding: 0;
      transition: height 0.3s, width 0.3s;
      display: flex;
      flex-direction: column;
      box-sizing: border-box;
      z-index: 1000;
      overflow: hidden;
      font-family: sans-serif;
    `;

    // Expanded content area
    const expandArea = document.createElement('div');
    expandArea.className = 'panel-expanded-content';
    expandArea.style.cssText = `
      display: none;
      flex: 1;
      padding: 10px;
      overflow: auto;
      background: #fff;
    `;
    expandArea.innerHTML = expandedContent;

    // Bottom row (avatars + button)
    const controlRow = document.createElement('div');
    controlRow.style.cssText = `
      display: flex;
      flex-direction: row;
      align-items: center;
      gap: 10px;
      padding: 10px;
      background: #f0f0f0;
      border-top: 1px solid #ccc;
    `;

    // Avatars
    avatars.forEach((url, i) => {
      const avatar = document.createElement('div');
      avatar.className = 'avatar';
      avatar.title = `Avatar ${i + 1}`;
      avatar.style.cssText = `
        width: 40px;
        height: 40px;
        flex-shrink: 0;
        background: url('${url}') no-repeat center/cover;
        border-radius: 50%;
        cursor: grab;
      `;
      avatar.addEventListener('mousedown', (e) => {
        panel.dispatchEvent(new MouseEvent('mousedown', e));
      });
      controlRow.appendChild(avatar);
    });

    // Expand button
    if (!toggleOnClick) {
      const btn = document.createElement('button');
      btn.innerText = 'Expand';
      btn.className = 'toggle-btn';
      btn.style.cssText = `
        margin-left: auto;
        cursor: pointer;
        background: #ccc;
        padding: 5px 10px;
        border: none;
        border-radius: 5px;
      `;
      btn.onclick = togglePanel;
      controlRow.appendChild(btn);
    }

    // Combine elements
    panel.appendChild(expandArea);
    panel.appendChild(controlRow);
    document.body.appendChild(panel);

    // Toggle expand/collapse
    function togglePanel() {
        
      const isExpanded = panel.classList.toggle('expanded');
        
    const limitexpanded = panel.offsetTop - (expandedHeight-panel.offsetHeight);
    
if (limitexpanded<0)      alert("Will over screen edge!");

    console.log(`Panel limit: ${limitexpanded}`);
        

      panel.style.height = isExpanded ? expandedHeight + 'px' : collapsedHeight + 'px';
      panel.style.width = isExpanded ? expandedWidth + 'px' : width + 'px';
      expandArea.style.display = isExpanded ? 'block' : 'none';
    }

    if (toggleOnClick) {
      panel.addEventListener('click', (e) => {
        if (e.target.tagName !== 'BUTTON') togglePanel();
      });
    }

    // Drag handling
    makeDraggableBottomAnchor(panel);

    // Optional external draggable avatar
    if (showExternalAvatar) {
      const ext = document.createElement('div');
      ext.className = 'external-avatar';
      ext.style.cssText = `
        position: absolute;
        left: ${left + width + 10}px;
        bottom: ${bottom}px;
        width: 40px;
        height: 40px;
        background: url('${externalAvatarURL}') no-repeat center/cover;
        border-radius: 50%;
        cursor: pointer;
      `;
      ext.title = 'Double-click me!';
      ext.ondblclick = () => alert('hi');
      makeDraggableBottomAnchor(ext);
      document.body.appendChild(ext);
    }

    return panelId;
  }

  // Make draggable anchored from bottom
  function makeDraggableBottomAnchor(el) {
    let startX = 0, startY = 0, startLeft = 0, startBottom = 0, isDragging = false;

    el.addEventListener('mousedown', (e) => {
      if (e.target.tagName === 'BUTTON') return;
      isDragging = true;
      startX = e.clientX;
      startY = e.clientY;
      const style = window.getComputedStyle(el);
      startLeft = parseInt(style.left, 10);
      startBottom = parseInt(style.bottom, 10);
      el.style.cursor = 'grabbing';
      e.preventDefault();
    });

    document.addEventListener('mousemove', (e) => {
      if (isDragging) {
        const dx = e.clientX - startX;
        const dy = e.clientY - startY;
        el.style.left = (startLeft + dx) + 'px';
        el.style.bottom = (startBottom - dy) + 'px';
      }
    });

    document.addEventListener('mouseup', () => {
      isDragging = false;
      el.style.cursor = 'grab';
    });
  }

  // Expose globally
  window.createDraggablePanel = createDraggablePanel;
})();
