/* tooltip.js  – drop-in smart tooltip */
(function () {
  const gap       = 8;
  const arrowSize = 6;
  const tooltip   = document.createElement('div');
  tooltip.id      = 'tooltip';
  tooltip.role    = 'tooltip';
  Object.assign(tooltip.style, {
    position: 'fixed',
    background: '#222',
    color: '#fff',
    padding: '6px 10px',
    borderRadius: '4px',
    fontSize: '13px',
    whiteSpace: 'nowrap',
    pointerEvents: 'none',
    zIndex: '9999',
    visibility: 'hidden',
    transition: 'opacity .15s'
  });
  document.body.appendChild(tooltip);

  /* arrow CSS once */
  const style = document.createElement('style');
  style.textContent = `
    #tooltip[data-arrow="top"]::after,
    #tooltip[data-arrow="bottom"]::after,
    #tooltip[data-arrow="left"]::after,
    #tooltip[data-arrow="right"]::after{
      content:'';
      position:absolute;
      width:0;height:0;
      border:6px solid transparent;
    }
    #tooltip[data-arrow="top"]::after{
      top:100%;left:50%;transform:translateX(-50%);
      border-top-color:#222;
    }
    #tooltip[data-arrow="bottom"]::after{
      bottom:100%;left:50%;transform:translateX(-50%);
      border-bottom-color:#222;
    }
    #tooltip[data-arrow="left"]::after{
      top:50%;left:-12px;transform:translateY(-50%);
      border-right-color:#222;
    }
    #tooltip[data-arrow="right"]::after{
      top:50%;right:-12px;transform:translateY(-50%);
      border-left-color:#222;
    }
  `;
  document.head.appendChild(style);

  /* ---------- show / hide ---------- */
  function show(e){
    const rect = e.target.getBoundingClientRect();
    const pos  = e.target.dataset.pos || 'TL';   // default fallback

    tooltip.textContent = e.target.dataset.tooltip || 'Tooltip';
    tooltip.style.visibility = 'visible';
    tooltip.style.transform = '';

    /* quick render to read real size */
    tooltip.style.left = '-9999px';
    tooltip.style.top  = '-9999px';
    const tipW = tooltip.offsetWidth  || 60;
    const tipH = tooltip.offsetHeight || 30;

    let left, top, arrowDir;

    /* horizontal (TTL / TTR) */
    if (pos === 'TTL' || pos === 'TTR') {
      const vw = window.innerWidth;
      const elCenter = rect.left + rect.width / 2;
      const useRight = elCenter < vw / 2;   // closer to left → use right side
      if (useRight) {
        left     = rect.right + gap;
        arrowDir = 'left';
      } else {
        left     = rect.left - tipW - gap;
        arrowDir = 'right';
      }
      top = rect.top + rect.height / 2;
      tooltip.style.transform = 'translateY(-50%)';
    }

    /* TB vertical */
    else if (pos === 'TB') {
      const vh = window.innerHeight;
      const elCenterY = rect.top + rect.height / 2;
      const useTop = elCenterY < vh / 2;

      left = rect.left + rect.width / 2;
      top  = useTop ? rect.bottom + arrowSize
                    : rect.top - arrowSize - rect.height / 2;
      arrowDir = useTop ? 'bottom' : 'top';
      tooltip.style.transform = 'translateX(-50%)';
    }

    /* CL/CC/CR smart */
    else if (pos === 'CL' || pos === 'CC' || pos === 'CR') {
      const vh = window.innerHeight;
      const elCenterY = rect.top + rect.height / 2;
      const useTop = elCenterY < vh / 2;

      left = rect.left + rect.width / 2;
      top  = useTop ? rect.bottom + arrowSize
                    : rect.top - arrowSize - rect.height / 2;
      arrowDir = useTop ? 'bottom' : 'top';
      tooltip.style.transform = 'translateX(-50%)';
    }

    /* classic TL/TC/TR */
    else if (pos === 'TL' || pos === 'TC' || pos === 'TR') {
      left = rect.left + rect.width / 2;
      top  = rect.bottom + arrowSize;
      arrowDir = 'bottom';
      tooltip.style.transform = 'translateX(-50%)';
    }

    /* classic BL/BC/BR */
    else {
      left = rect.left + rect.width / 2;
      top  = rect.top - arrowSize - rect.height / 2;
      arrowDir = 'top';
      tooltip.style.transform = 'translateX(-50%)';
    }

    /* viewport clamp */
    const vw = window.innerWidth;
    const vh = window.innerHeight;
    if (arrowDir === 'left' || arrowDir === 'right') {
      top  = Math.max(8, Math.min(top  - tipH/2, vh - tipH - 8));
      left = Math.max(8, Math.min(left, vw - tipW - 8));
    } else {
      left = Math.max(8, Math.min(left - tipW/2, vw - tipW - 8));
      top  = Math.max(8, Math.min(top,  vh - tipH - 8));
    }

    tooltip.style.left = left + 'px';
    tooltip.style.top  = top  + 'px';
    tooltip.setAttribute('data-arrow', arrowDir);
  }

  function hide() {
    tooltip.style.visibility = 'hidden';
  }

  /* auto-attach on any element with data-pos */
  document.addEventListener('mouseover', e=>{
    if (e.target.dataset.pos) show(e);
  });
  document.addEventListener('mouseout', e=>{
    if (e.target.dataset.pos) hide();
  });
})();