// ===== Dynamic API Config =====
let API_BASE_URL = localStorage.getItem("apiBaseUrl") || "https://j.kunok.com/aui.php";
const API_ENDPOINTS = {
    list: (id = "") => id ? `${API_BASE_URL}?action=list&id=${encodeURIComponent(id)}` : `${API_BASE_URL}?action=list`,
    update: () => `${API_BASE_URL}?action=update`,
    delete: () => `${API_BASE_URL}?action=del`
};
// ==============================

function setApiBaseUrl(url) {
    API_BASE_URL = url.trim();
    localStorage.setItem("apiBaseUrl", API_BASE_URL);
    displayMessage(`API base URL updated: ${API_BASE_URL}`, false, true);
}

// Grey circle avatar generator
function greyAvatar(size = 40) {
    return `<div style="width:${size}px;height:${size}px;border-radius:50%;background:#ccc;display:inline-block;"></div>`;
}

function displayMessage(message, isError = false, isInfo = false) {
    let msgDiv = document.getElementById("message");
    if (!msgDiv) {
        msgDiv = document.createElement("div");
        msgDiv.id = "message";
        document.body.insertBefore(msgDiv, document.body.firstChild);
    }
    msgDiv.textContent = message;
    msgDiv.className = "message " + (isError ? "error" : isInfo ? "info" : "success");
    if (isInfo) {
        setTimeout(() => {
            if (msgDiv.className.includes("info")) {
                msgDiv.textContent = "";
                msgDiv.className = "message";
            }
        }, 3000);
    }
}

async function postto(apiUrl = "", payload = {}) {
    try {
        const response = await fetch(apiUrl, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: payload }),
        });
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error("Request failed:", error);
        displayMessage(error.message, true);
        throw error;
    }
}

async function checkUser(userId) {
    const response = await fetch(API_ENDPOINTS.list(userId));
    const result = await response.json();
    if (result.data && result.data.length > 0) {
        return result.data[0];
    } else {
        throw new Error('User not found');
    }
}

async function updateUserPanel(userId = "") {
    if (!userId) {
        displayMessage("No user ID provided", true);
        return;
    }

    try {
        displayMessage("Loading user data...", false, true);
        const user = await checkUser(userId);

        const formHTML = `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                .avatar-circle { width:80px; height:80px; border-radius:50%; background:#ccc; margin:auto; }
                #updateImgPreview img {
                    max-width: 80px;
                    max-height: 80px;
                    object-fit: cover;
                    border-radius: 50%;
                }
            </style>
            <div class="logo-controls">
                <div id="updateImgPreview"><img src="${user.img ? (user.img) : defaultAvatar}" alt="preview"></div>
                <div class="button-group" style="display: flex; gap: 10px; justify-content: center; margin-top: 10px;">
                    <button type="button" id="updateUploadLogoBtn">Upload logo</button>
                    <button type="button" id="updateApplyLogoBtn">Apply Logo</button>
                </div>
            </div>
            <form id="updateUserForm">
                <div><label>Name:</label> <input type="text" id="updateName" value="${user.name || ''}"></div>
                <div><label>Username:</label> <input type="text" id="updateUsername" value="${user.username || ''}"></div>
                <div><label>Password:</label> <input type="password" id="updatePassword" value="${user.password || ''}"></div>
                <div class="form-row">
                    <div>
                        <label>Flag:</label>
                        <select id="updateFlag">
                            <option value="1" ${user.flag == 1 ? 'selected' : ''}>Active</option>
                            <option value="0" ${user.flag == 0 ? 'selected' : ''}>Suspended</option>
                            <option value="3" ${user.flag == 3 ? 'selected' : ''}>Wait</option>
                        </select>
                    </div>
                    <div>
                        <label>Level:</label>
                        <select id="updateLevel">
                            <option value="1" ${user.level == 1 ? 'selected' : ''}>User</option>
                            <option value="2" ${user.level == 2 ? 'selected' : ''}>Moderator</option>
                            <option value="3" ${user.level == 3 ? 'selected' : ''}>Admin</option>
                        </select>
                    </div>
                </div>
                <input type="hidden" id="updateImg" value="${user.img || ''}">
                <button type="submit">Update User</button>
            </form>
            <div id="message" class="message"></div>
        `;

        createPanel("updateUserPanel", "CC", "ctc", 400, 460, {
            bodyContent: formHTML,
            noFooter: true,
            title: 'Update User'
        });

        // Form submit handler
        document.getElementById("updateUserForm").addEventListener("submit", async (e) => {
            e.preventDefault();
            const payload = {
                id: userId,
                name: document.getElementById("updateName").value,
                username: document.getElementById("updateUsername").value,
                password: document.getElementById("updatePassword").value,
                flag: parseInt(document.getElementById("updateFlag").value),
                level: parseInt(document.getElementById("updateLevel").value),
                img: document.getElementById("updateImg").value
            };
            try {
                const response = await postto(API_ENDPOINTS.update(), payload);
                displayMessage(response.message || `User ${userId} updated successfully`);
            } catch (error) {
                displayMessage(`Error updating user ${userId}: ${error.message}`, true);
            }
        });

        // Restore Upload Logo / Apply Logo buttons
        document.getElementById("updateUploadLogoBtn").addEventListener("click", () => {
            window.logomode = true;
            toggleUploadPanel(); // from upl3.js
        });

        document.getElementById("updateApplyLogoBtn").addEventListener("click", () => {
            logoset("updateImg", "updateImgPreview");
        });

    } catch (error) {
        displayMessage(error.message, true);
    }
}


async function userListPanel() {
    try {
        const res = await fetch(API_ENDPOINTS.list());
        const json = await res.json();
        const users = json.data || [];

        const serverInput = `
            <div style="margin-bottom:10px;">
                <label>Server API:</label>
                <input type="text" id="serverUrlInput" value="${API_BASE_URL}" style="width:70%">
                <button onclick="setApiBaseUrl(document.getElementById('serverUrlInput').value)">Set</button>
            </div>
        `;

        let tableHTML = `
        <style>
            table.user-table { width:100%; border-collapse: collapse; font-size:14px; }
            table.user-table th, table.user-table td { padding:6px; border-bottom:1px solid #ddd; text-align:left; }
            table.user-table th { background:#f0f0f0; }
            table.user-table td.actions button { margin-right:4px; }
        </style>
        ${serverInput}
        <table class="user-table">
            <thead>
                <tr>
                    <th>Avatar</th>
                    <th>Name</th>
                    <th>ID</th>
                    <th>Username</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
        `;

        users.forEach(u => {
            tableHTML += `
            <tr>
                <td>${greyAvatar(32)}</td>
                <td>${u.name || ""}</td>
                <td>${u.id || ""}</td>
                <td>${u.username || ""}</td>
                <td class="actions">
                    <button onclick="viewUser('${u.id}')">View</button>
                    <button onclick="updateUserPanel('${u.id}')">Edit</button>
                    <button onclick="deleteUser('${u.id}')">Delete</button>
                </td>
            </tr>`;
        });

        tableHTML += `</tbody></table>`;

        createPanel("userListPanel", "CC", "ctc", 650, 450, {
            bodyContent: tableHTML,
            noFooter: true,
            title: "User List"
        });

    } catch (err) {
        console.error("Error loading user list:", err);
    }
}

function viewUser(id) {
    alert("View user " + id);
}

async function deleteUser(id) {
    if (!confirm("Delete this user?")) return;
    try {
        const payload = { id };
        const res = await fetch(API_ENDPOINTS.delete(), {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: payload })
        });
        const json = await res.json();
        alert(json.message || "Deleted");
        closePanel("userListPanel");
        userListPanel(); // refresh
    } catch (err) {
        console.error("Delete error:", err);
        alert("Error deleting user");
    }
}
