// Enhanced Tooltip System
function updateTooltip(elementId, text, options = {}) {
  const element = document.getElementById(elementId);
  if (!element) return;

  // Default options
  const defaults = {
    position: 'top',    // top, right, bottom, left
    color: 'gray',      // red, blue, green, purple, orange, gray
    hover: true,        // show on hover (false for manual trigger)
    width: 'auto',      // width in px or 'auto'
    container: null     // ID of container to escape from
  };
  const settings = { ...defaults, ...options };

  // Remove existing tooltip if re-rendering
  let oldTooltip = element._tooltipElement;
  if (oldTooltip) {
    oldTooltip.remove();
    if (element._tooltipHandlers) {
      element.removeEventListener('mouseenter', element._tooltipHandlers.enter);
      element.removeEventListener('mouseleave', element._tooltipHandlers.leave);
    }
  }

  // Create tooltip element (hidden by default)
  const tooltip = document.createElement('div');
  tooltip.className = `dynamic-tooltip ${settings.position} ${settings.color}`;
  if (settings.hover) tooltip.classList.add('hover');
  tooltip.style.visibility = 'hidden'; // Start hidden
  tooltip.style.opacity = '0';

  // Create arrow
  const arrow = document.createElement('div');
  arrow.className = 'tooltip-arrow';
  tooltip.appendChild(arrow);

  // Create content
  const content = document.createElement('div');
  content.className = 'tooltip-content';
  content.textContent = text;
  tooltip.appendChild(content);

  document.body.appendChild(tooltip);

  // Store tooltip reference
  element._tooltipElement = tooltip;
  element._tooltipSettings = settings;

  // Position tooltip (but keep hidden)
  const positionTooltip = (target) => {
    const rect = target.getBoundingClientRect();
    const tooltipRect = tooltip.getBoundingClientRect();
    const scrollY = window.scrollY || window.pageYOffset;
    const scrollX = window.scrollX || window.pageXOffset;

    let top, left;

    switch (settings.position) {
      case 'top':
        top = rect.top + scrollY - tooltipRect.height - 10;
        left = rect.left + scrollX + (rect.width / 2) - (tooltipRect.width / 2);
        break;
      case 'right':
        top = rect.top + scrollY + (rect.height / 2) - (tooltipRect.height / 2);
        left = rect.right + scrollX + 10;
        break;
      case 'bottom':
        top = rect.bottom + scrollY + 10;
        left = rect.left + scrollX + (rect.width / 2) - (tooltipRect.width / 2);
        break;
      case 'left':
        top = rect.top + scrollY + (rect.height / 2) - (tooltipRect.height / 2);
        left = rect.left + scrollX - tooltipRect.width - 10;
        break;
    }

    // Adjust for container bounds if defined
    if (settings.container) {
      const container = document.getElementById(settings.container);
      if (container) {
        const containerRect = container.getBoundingClientRect();

        if (settings.position === 'top' && top < containerRect.top + scrollY) {
          settings.position = 'bottom';
          tooltip.className = `dynamic-tooltip bottom ${settings.color} ${settings.hover ? 'hover' : ''}`;
          return positionTooltip(target);
        }
        if (settings.position === 'bottom' && top + tooltipRect.height > containerRect.bottom + scrollY) {
          settings.position = 'top';
          tooltip.className = `dynamic-tooltip top ${settings.color} ${settings.hover ? 'hover' : ''}`;
          return positionTooltip(target);
        }
        if (settings.position === 'left' && left < containerRect.left + scrollX) {
          settings.position = 'right';
          tooltip.className = `dynamic-tooltip right ${settings.color} ${settings.hover ? 'hover' : ''}`;
          return positionTooltip(target);
        }
        if (settings.position === 'right' && left + tooltipRect.width > containerRect.right + scrollX) {
          settings.position = 'left';
          tooltip.className = `dynamic-tooltip left ${settings.color} ${settings.hover ? 'hover' : ''}`;
          return positionTooltip(target);
        }
      }
    }

    // Ensure tooltip stays inside viewport
    left = Math.max(10, Math.min(left, window.innerWidth - tooltipRect.width - 10));
    top = Math.max(10, Math.min(top, window.innerHeight - tooltipRect.height - 10));

    tooltip.style.left = `${left}px`;
    tooltip.style.top = `${top}px`;
  };

  // Event listeners for hover
  const showTooltip = () => {
    positionTooltip(element);
    tooltip.style.visibility = 'visible';
    tooltip.style.opacity = '1';
  };

  const hideTooltip = () => {
    tooltip.style.visibility = 'hidden';
    tooltip.style.opacity = '0';
  };

  if (settings.hover) {
    element.addEventListener('mouseenter', showTooltip);
    element.addEventListener('mouseleave', hideTooltip);

    element._tooltipHandlers = {
      enter: showTooltip,
      leave: hideTooltip
    };
  }

  // Manual trigger function
  element.showTooltip = () => {
    positionTooltip(element);
    tooltip.style.visibility = 'visible';
    tooltip.style.opacity = '1';
  };

  element.hideTooltip = () => {
    tooltip.style.visibility = 'hidden';
    tooltip.style.opacity = '0';
  };

  // Initial positioning (still hidden)
  requestAnimationFrame(() => {
    positionTooltip(element);
  });
}

// Initialize tooltip styles
function initTooltipStyles() {
  const style = document.createElement('style');
  style.textContent = `
    /* Dynamic Tooltip System */
    .dynamic-tooltip {
      position: absolute;
      visibility: hidden;
      opacity: 0;
      z-index: 10000;
      transition: opacity 0.3s ease, visibility 0.3s ease;
      pointer-events: none;
      font-size: 12px;
      line-height: 1.4;
      max-width: 300px;
    }
    
    .dynamic-tooltip:not(.hover) {
      visibility: visible;
      opacity: 1;
    }
    
    .tooltip-content {
      padding: 8px 12px;
      border-radius: 4px;
      position: relative;
      box-shadow: 0 2px 8px rgba(0,0,0,0.15);
    }
    
    .tooltip-arrow {
      position: absolute;
      width: 0;
      height: 0;
      border-style: solid;
    }
    
    /* Position classes */
    .dynamic-tooltip.top .tooltip-arrow {
      bottom: -5px;
      left: 50%;
      transform: translateX(-50%);
      border-width: 5px 5px 0 5px;
    }
    
    .dynamic-tooltip.right .tooltip-arrow {
      left: -5px;
      top: 50%;
      transform: translateY(-50%);
      border-width: 5px 5px 5px 0;
    }
    
    .dynamic-tooltip.bottom .tooltip-arrow {
      top: -5px;
      left: 50%;
      transform: translateX(-50%);
      border-width: 0 5px 5px 5px;
    }
    
    .dynamic-tooltip.left .tooltip-arrow {
      right: -5px;
      top: 50%;
      transform: translateY(-50%);
      border-width: 5px 0 5px 5px;
    }
    
    /* Color classes */
    .dynamic-tooltip.red .tooltip-content {
      background-color: #ff4444;
      color: white;
    }
    .dynamic-tooltip.red .tooltip-arrow {
      border-color: #ff4444 transparent transparent transparent;
    }
    .dynamic-tooltip.top.red .tooltip-arrow {
      border-color: #ff4444 transparent transparent transparent;
    }
    .dynamic-tooltip.right.red .tooltip-arrow {
      border-color: transparent #ff4444 transparent transparent;
    }
    .dynamic-tooltip.bottom.red .tooltip-arrow {
      border-color: transparent transparent #ff4444 transparent;
    }
    .dynamic-tooltip.left.red .tooltip-arrow {
      border-color: transparent transparent transparent #ff4444;
    }
    
    .dynamic-tooltip.blue .tooltip-content {
      background-color: #4444ff;
      color: white;
    }
    .dynamic-tooltip.blue .tooltip-arrow {
      border-color: #4444ff transparent transparent transparent;
    }
    
    .dynamic-tooltip.green .tooltip-content {
      background-color: #44aa44;
      color: white;
    }
    .dynamic-tooltip.green .tooltip-arrow {
      border-color: #44aa44 transparent transparent transparent;
    }
    
    .dynamic-tooltip.purple .tooltip-content {
      background-color: #aa44aa;
      color: white;
    }
    .dynamic-tooltip.purple .tooltip-arrow {
      border-color: #aa44aa transparent transparent transparent;
    }
    
    .dynamic-tooltip.orange .tooltip-content {
      background-color: #ffaa44;
      color: black;
    }
    .dynamic-tooltip.orange .tooltip-arrow {
      border-color: #ffaa44 transparent transparent transparent;
    }
    
    .dynamic-tooltip.gray .tooltip-content {
      background-color: #666666;
      color: white;
    }
    .dynamic-tooltip.gray .tooltip-arrow {
      border-color: #666666 transparent transparent transparent;
    }
  `;
  document.head.appendChild(style);
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
  initTooltipStyles();
});