function genform(divId, data) {
  const container = document.getElementById(divId);
  if (!container) {
    console.error(`Div with ID ${divId} not found`);
    return;
  }

  // Store reference to original data (no deep copy)
  container._formData = data;
  const lang = window.lang || "en";
  // Normalize radiotext values on load
  for (const key in data) {
    if (key === "grouping" || data[key].type === "section") continue;
    const field = data[key];
    if (field.data_type === "radiotext" && field.value) {
      if (
        field.value.radio === "" ||
        field.value.text === "Others (please specify)/อื่น ๆ (โปรดระบุ)"
      ) {
        field.value = { radio: null, text: "" };
      }
    }
  }
  // Initialize container HTML
  container.innerHTML = `
        <div style="font-family: Arial, sans-serif;">
            <div style="display: none; justify-content: space-between; align-items: center;">
                <h3 style="font-size: 0.8em;">${getLabel({ label: { en: "Form in", th: "FORM" } }, lang)} ${divId}</h3>
            </div>
            <div id="${divId}-fields"></div>
            <div style="display:none;">
                <textarea id="${divId}-input" placeholder="Enter JSON schema"></textarea>
                <button id="${divId}-copy-btn">Copy to Schema</button>
                <textarea id="${divId}-output" readonly></textarea>
            </div>
        </div>
    `;

  const fieldsDiv = document.getElementById(`${divId}-fields`);
  const schema = container._formData;
  const grouping = getGroupingData(schema);

  // Organize segments and sections
  const segments = new Map();

  if (grouping) {
    const names = Array.isArray(grouping.name) ? grouping.name : [];
    const types = Array.isArray(grouping.type) ? grouping.type : [];
    const labels = {
      en: Array.isArray(grouping.label?.en) ? grouping.label.en : [],
      th: Array.isArray(grouping.label?.th) ? grouping.label.th : [],
    };
    console.log("Grouping:", grouping);
    console.log("Names:", names, "Types:", types, "Labels:", labels);

    names.forEach((name, i) => {
      if (types[i] === "segment") {
        const segmentLabel = {
          en: labels.en[i] || name,
          th: labels.th[i] || name,
        };
        console.log(`Creating segment: ${name}, Label:`, segmentLabel);
        segments.set(name, {
          label: segmentLabel,
          sections: new Map(),
        });
      }
    });
  } else {
    console.warn("No valid grouping found in schema. Using default segment.");
    segments.set("default", {
      label: {
        en: "Default",
        th: "ค่าเริ่มต้น",
      },
      sections: new Map(),
    });
  }

  // Map fields to segments and sections
  for (const key in schema) {
    if (key === "grouping" || key === "data_type") continue;
    const field = schema[key];

    // Determine segment - first try field.segment, then first grouping name, then default
    const segment =
      field.segment?.[0] ||
      (grouping?.name
        ? Array.isArray(grouping.name)
          ? grouping.name[0]
          : grouping.name
        : null) ||
      "default";

    if (!segments.has(segment)) {
      // Auto-create missing segments
      segments.set(segment, {
        label: {
          en: segment,
          th: segment,
        },
        sections: new Map(),
      });
    }

    if (field.type === "section") {
      segments.get(segment).sections.set(key, getLabel(field, lang) || key);
    } else if (field.section) {
      if (!segments.get(segment).sections.has(field.section)) {
        if (schema[field.section]) {
          segments
            .get(segment)
            .sections.set(
              field.section,
              getLabel({ label: schema[field.section]?.label }, lang) ||
                field.section,
            );
        } else {
          console.warn(
            `Section ${field.section} for field ${key} not found in schema`,
          );
        }
      }
    } else if (field.data_type !== "grouping") {
      console.warn(`Field ${key} has no section assigned`);
    }
  }

  // Generate form structure
  segments.forEach((segmentData, segmentId) => {
    const segDetails = document.createElement("details");
    segDetails.className = "segment";
    segDetails.style.cssText =
      "border-radius: 5px;background-color: rgba(255,255,255,.3);backdrop-filter: blur(15px);border: 1px solid #4a90e2; border-radius: 6px; margin-bottom: 10px; padding: 10px;";
    segDetails.id = `${divId}-segment-${encodeFieldKey(segmentId)}`;
    segDetails.open = true;
    console.log(
      "Segment ID:",
      segmentId,
      "Label:",
      segmentData.label,
      "Lang:",
      lang,
    );
    console.log("getLabel output:", getLabel(segmentData.label, lang));
    segDetails.innerHTML = `<summary style="font-weight: bold; cursor: pointer;">${getLabel({ label: { en: "Segment", th: "ส่วน" } }, lang)}: ${getLabel(segmentData.label, lang) || segmentId}</summary>`;

    const segDiv = document.createElement("div");
    segDiv.style.padding = "10px";

    segmentData.sections.forEach((sectionLabel, sectionId) => {
      const secDetails = document.createElement("details");
      secDetails.className = "section";
      secDetails.id = `${divId}-section-${encodeFieldKey(sectionId)}`;
      secDetails.style.cssText =
        "box-shadow: 0 14px 12px rgba(0, 0, 0, 0.2);border-radius: 5px;background-color: rgba(255,255,255,.1);backdrop-filter: blur(20px);border: 1px solid green; border-radius: 5px; margin: 8px 0; padding: 4px 8px;";
      secDetails.open = true;

      // Calculate section totals for progress bar
      let sectionTotals = {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredFields: [],
      };

      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (field.section !== sectionId || !field.segment?.includes(segmentId))
          continue;
        if (!field.data_type && !field.type) {
          console.warn(`Field ${key} has no data_type or type defined`);
          continue;
        }

        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const isAnswered = isFieldAnswered(field);
        const isFile = field.data_type === "attachment";
        const fileCount = isFile ? countFiles(field) : 0;

        sectionTotals.allFields++;
        if (field.required) sectionTotals.requiredFields++;
        if (isAnswered) sectionTotals.answeredFields++;
        if (field.required && isAnswered)
          sectionTotals.answeredRequiredFields++;
        sectionTotals.totalScore += isAnswered ? field.score || 1 : 0;
        sectionTotals.totalScoreWeighted += currentScore;
        sectionTotals.maxPossibleScore += field.score || 1;
        sectionTotals.maxPossibleScoreWeighted += maxScore;
        sectionTotals.totalFiles += fileCount;
        if (field.required && !isFieldAnswered(field)) {
          if (
            document.getElementById(`${divId}-field-${encodeFieldKey(key)}`)
          ) {
            sectionTotals.unansweredFields.push({
              key,
              label: getLabel(field, lang) || key,
            });
          } else {
            console.warn(
              `Field ${key} not rendered in DOM, skipping from unanswered list`,
            );
          }
        }
      }

      // Section title with progress bar, styled like section-summary
      const progressPercent =
        sectionTotals.requiredFields > 0
          ? Math.round(
              (sectionTotals.answeredRequiredFields /
                sectionTotals.requiredFields) *
                100,
            )
          : 100;
      secDetails.innerHTML = `
        <summary style="cursor: pointer; margin-top: 10px; padding: 8px; background-color: rgba(200,190,200,.1); border-radius: 4px; font-size: 0.9em;">
          <span style="font-weight: bold;">${sectionLabel}</span>
        </summary>
        <div id="${divId}-section-header-${encodeFieldKey(sectionId)}" style="margin-bottom: 8px; padding: 8px; background-color:rgba(200,200,200,.1);backdrop-filter: blur(15px); ">
          <div style="font-weight: bold;"></div>
          <div style="margin-top: 5px;  background-color:rgb(163, 61, 61);border-radius: 4px;">
            [Progress: ${progressPercent}% (${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields})]
            <div style="width: 100%;  height: 15px; border-radius: 5px; overflow: hidden;">
              <div style="width: ${progressPercent}%; background-color: #4caf50; height: 100%;"></div>
            </div>
          </div>
        </div>
      `;

      const secDiv = document.createElement("div");
      secDiv.style.padding = "18px 0px";

      let hasField = false;
      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (field.section !== sectionId || !field.segment?.includes(segmentId))
          continue;

        hasField = true;
        const fieldDiv = document.createElement("div");
        fieldDiv.className = "form-field";
        fieldDiv.id = `${divId}-field-${encodeFieldKey(key)}`;
        fieldDiv.style.cssText =
          "border-radius: 5px;background-color: rgba(255,255,255,.3);backdrop-filter: blur(10px); padding: 10px; margin-bottom: 10px; border-radius: 4px;";
        if (field.required && !isFieldAnswered(field)) {
          fieldDiv.style.border = "2px solid red";
        }

        const label = document.createElement("label");

        label.style.cssText =
          "margin:10px 0px;box-shadow: 0 14px 12px rgba(0, 0, 0, 0.1);padding:10px 5px;border-radius: 5px;background-color: rgba(255,255,255,.1);backdrop-filter: blur(5px);font-weight: 500; margin-right: 6px; display: block; padding-bottom: 5px;";
        if (field.required) label.classList.add("required-field");
        label.innerHTML =
          (getLabel(field, lang) || key) + (field.required ? "" : ""); // *required field
        label.htmlFor = `${divId}-${encodeFieldKey(key)}`;
        if (field.score !== undefined || field.weights) {
          const scoreDisplay = document.createElement("span");
          scoreDisplay.className = "score-display";
          const maxScore = calculateMaxScore(field);
          const currentScore = calculateCurrentScore(field);
          if (typeof window.score !== "undefined" && window.score) {
            scoreDisplay.textContent = `(Score: ${currentScore}/${maxScore})`;
          }
          label.appendChild(scoreDisplay);
        }
        fieldDiv.appendChild(label);

        const input = createInputElement(divId, key, field, lang, schema);
        fieldDiv.appendChild(input);
        fieldDiv.appendChild(
          createResetButton(divId, key, field, input, lang, schema, fieldDiv),
        );

        // Add score bar area
        const scoreBar = document.createElement("div");
        scoreBar.className = "score-bar";
        scoreBar.style.cssText =
          "margin-top: 5px; font-size: 0.9em; color: #555;";
        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const weight = field.weights ? field.weights.join(", ") : "1";
        const fileCount =
          field.data_type === "attachment" ? countFiles(field) : 0;
        //const value = field.value || "";
        //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
        let valueText = "";
        if (typeof field.value === "object" && field.value !== null) {
          try {
            valueText = JSON.stringify(field.value);
          } catch (e) {
            valueText = "[object]";
          }
        } else {
          valueText = field.value || "";
        }
        if (window.debug)
          scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
        else
          scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
        fieldDiv.appendChild(scoreBar);

        secDiv.appendChild(fieldDiv);
      }

      if (hasField) {
        // Add section summary
        const sectionSummary = document.createElement("details");
        sectionSummary.className = "section-summary";
        sectionSummary.id = `${divId}-summary-${encodeFieldKey(segmentId)}-${encodeFieldKey(sectionId)}`;
        sectionSummary.style.cssText =
          "margin-top: 10px; padding: 8px; background-color: rgba(200,200,200,.7); border-radius: 4px; font-size: 0.9em;";
        sectionSummary.open = true;
        const unansweredCount =
          sectionTotals.requiredFields - sectionTotals.answeredRequiredFields;
        sectionSummary.innerHTML = `
          <summary style="cursor: pointer;">UnAnswered/Total/TotalRequired (${unansweredCount}/${sectionTotals.allFields}/${sectionTotals.requiredFields})</summary>
          <div style="padding: 8px;">
            [ field total: ${sectionTotals.allFields}, score: ${sectionTotals.totalScore}, achieved score: ${sectionTotals.totalScoreWeighted}, maxscore: ${sectionTotals.maxPossibleScoreWeighted}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles} ]
            ${
              sectionTotals.unansweredFields.length > 0
                ? `
              <div style="margin-top: 10px; color: red;">
                <strong>Unanswered Required Fields:</strong>
                <ul style="margin: 5px 0; padding-left: 20px;">
                  ${sectionTotals.unansweredFields
                    .filter((field) =>
                      document.getElementById(
                        `${divId}-field-${encodeFieldKey(field.key)}`,
                      ),
                    )
                    .map(
                      (field) => `
                      <li style="max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                        <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${divId}')"
                           style="text-decoration: none; color: red;"
                           title="${field.label}">
                          ${field.label}
                        </a>
                      </li>
                    `,
                    )
                    .join("")}
                </ul>
              </div>
            `
                : ""
            }
          </div>
        `;
        secDiv.appendChild(sectionSummary);
        segDiv.appendChild(secDetails);
        secDetails.appendChild(secDiv);
      }
    });

    segDetails.appendChild(segDiv);
    fieldsDiv.appendChild(segDetails);
  });

  // Handle schema copy button
  const copyBtn = document.getElementById(`${divId}-copy-btn`);
  copyBtn.addEventListener("click", () => {
    const inputEl = document.getElementById(`${divId}-input`);
    try {
      const newSchema = JSON.parse(inputEl.value);
      genform(divId, newSchema);
    } catch (e) {
      console.error(`Invalid JSON in ${divId}-input: ${e.message}`);
      alert(
        getLabel(
          {
            label: {
              en: `Invalid JSON: ${e.message}`,
              th: `JSON ไม่ถูกต้อง: ${e.message}`,
            },
          },
          lang,
        ),
      );
    }
  });

  listenform(divId);
}
//

// Helper function to check if a field is answered
function XXXisFieldAnswered(field) {
  const inputType = field.data_type || field.type;
  let result = false;

  if (inputType === "checkbox") {
    result = Array.isArray(field.value) && field.value.length > 0;
  } else if (inputType === "radio" || inputType === "select") {
    result = field.value !== null && field.value !== "";
  } else if (inputType === "attachment" || inputType === "file") {
    result = field.value && countFiles(field) > 0;
  } else if (inputType === "radiotext") {
    const hasValidRadio =
      field.value?.radio !== null &&
      field.value?.radio !== undefined &&
      field.value?.radio !== "";
    const hasValidText =
      field.value?.text &&
      field.value.text.trim() !== "" &&
      field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    result = hasValidRadio || hasValidText;
  } else if (inputType === "radiotextatt") {
    const hasValidRadio =
      field.value?.radio !== null &&
      field.value?.radio !== undefined &&
      field.value?.radio !== "";
    const hasValidText =
      field.value?.text &&
      field.value.text.trim() !== "" &&
      field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    const hasValidAttachment = countFiles(field) > 0;
    result = hasValidRadio || hasValidText || hasValidAttachment;
  } else if (inputType === "checktext" || inputType === "checktextatt") {
    const hasValidCheck =
      Array.isArray(field.value?.check) && field.value.check.length > 0;
    const hasValidText =
      field.value?.text &&
      field.value.text.trim() !== "" &&
      field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    const hasValidAttachment =
      inputType === "checktextatt" ? countFiles(field) > 0 : false;
    return hasValidCheck || hasValidText || hasValidAttachment;
  } else {
    result = field.value !== null && field.value !== "";
  }

  console.log(
    `[isFieldAnswered] field=${field.key || "?"}, type=${inputType}, value=`,
    field.value,
    `→ answered=${result}`,
  );

  return result;
}

function XXisFieldAnsweredAA(field) {
  const inputType = field.data_type || field.type;
  const fieldKey = field.key || "unknown";

  if (inputType === "checkbox") {
    return Array.isArray(field.value) && field.value.length > 0;
  } else if (inputType === "radio" || inputType === "select") {
    return field.value !== null && field.value !== "";
  } else if (inputType === "attachment" || inputType === "file") {
    return field.value && countFiles(field) > 0;
  } else if (inputType === "radiotext") {
    const isValidRadio =
      field.value?.radio !== null &&
      field.value?.radio !== undefined &&
      field.value?.radio !== "";
    const isValidText =
      field.value?.text &&
      field.value.text.trim() !== "" &&
      field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    return isValidRadio && isValidText;
  } else {
    return field.value !== null && field.value !== "";
  }
}
function isFieldAnswered(field) {
  const inputType = field.data_type || field.type;

  if (inputType === "checkbox") {
    return Array.isArray(field.value) && field.value.length > 0;
  } else if (inputType === "radio" || inputType === "select") {
    return field.value !== null && field.value !== "";
  } else if (inputType === "attachment" || inputType === "file") {
    return field.value && countFiles(field) > 0;
  } else if (inputType === "radiotext") {
    const hasValidRadio =
      field.value?.radio !== null &&
      field.value?.radio !== undefined &&
      field.value?.radio !== "";
    const hasValidText =
      field.value?.text &&
      field.value.text.trim() !== "" &&
      field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    return hasValidRadio || hasValidText;
  } else if (inputType === "radiotextatt" || inputType === "radioatt") {
    const hasValidRadio =
      field.value?.radio !== null &&
      field.value?.radio !== undefined &&
      field.value?.radio !== "";
    const hasValidAttachment = countFiles(field) > 0;
    return hasValidRadio || hasValidAttachment;
  } else if (inputType === "checktext" || inputType === "checktextatt") {
    const hasValidCheck =
      Array.isArray(field.value?.check) && field.value.check.length > 0;
    const hasValidText =
      field.value?.text &&
      field.value.text.trim() !== "" &&
      field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    const hasValidAttachment =
      inputType === "checktextatt" ? countFiles(field) > 0 : false;
    return hasValidCheck || hasValidText || hasValidAttachment;
  }
  if (inputType === "radioatt") {
    const hasValidRadio =
      field.value?.radio !== null &&
      field.value?.radio !== undefined &&
      field.value?.radio !== "";
    const hasValidAttachment = countFiles(field) > 0;
    return hasValidRadio || hasValidAttachment;
  } else {
    return field.value !== null && field.value !== "";
  }
}

// Helper function to count files

function countFiles(field) {
  if (!field || !field.value) return 0;

  // Handle attachment type
  if (field.data_type === "attachment" && typeof field.value === "string") {
    return field.value.split(",").filter((f) => f.trim()).length;
  }

  // Handle radiotextatt and radioatt types
  if (
    (field.data_type === "radiotextatt" || field.data_type === "radioatt") &&
    field.value?.att &&
    typeof field.value.att === "string"
  ) {
    return field.value.att.split(",").filter((f) => f.trim()).length;
  }
  // Handle checktextatt type
  if (
    field.data_type === "checktextatt" &&
    field.value?.att &&
    typeof field.value.att === "string"
  ) {
    return field.value.att.split(",").filter((f) => f.trim()).length;
  }
  return 0;
}

// Helper function to create input elements
function createInputElement(divId, key, field, lang, schema) {
  const inputType = field.data_type || field.type;
  let input;

  if (inputType === "textarea") {
    input = document.createElement("textarea");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.value = field.value || "";
    input.style.width = "260px";
  } else if (inputType === "radioatt") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";

    // Create radio buttons container
    const radioContainer = document.createElement("div");
    radioContainer.style.cssText =
      "display: flex; flex-direction: column; gap: 5px;";

    // Add radio options
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}-radio" value="${val}" ${
        field.value && field.value.radio === val ? "checked" : ""
      }> ${opt}`;
      radioContainer.appendChild(radioLabel);
    });

    input.appendChild(radioContainer);

    // Add attachment input
    const attDiv = document.createElement("div");
    attDiv.style.marginTop = "5px";

    const attInput = document.createElement("input");
    attInput.type = "text";
    attInput.id = `${divId}-${encodeFieldKey(key)}-att`;
    attInput.value = (field.value && field.value.att) || "";
    attInput.placeholder = getLabel(
      { label: { en: "Enter attachment", th: "ป้อนไฟล์แนบ" } },
      lang,
    );
    attInput.style.width = "260px";
    attDiv.appendChild(attInput);

    const fileCount = document.createElement("div");
    fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
    fileCount.style.marginTop = "5px";
    fileCount.textContent = `Files: ${countFiles(field)}`;
    input.appendChild(fileCount);

    const addFileBtn = document.createElement("button"); //Add file เพิ่มไฟล์
    addFileBtn.textContent = getLabel(
      { label: { en: "Add file", th: "Add file" } },
      lang,
    );
    addFileBtn.style.cssText =
      "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
    addFileBtn.onclick = () => {
      window.logomode = false;
      toggleUploadPanel();
      updateFileListFromInput(`${divId}-${encodeFieldKey(key)}-att`);
      const count = countFiles(schema[key]);
      document.getElementById(
        `${divId}-filecount-${encodeFieldKey(key)}`,
      ).textContent = `Files: ${count}`;
      schema[key].value.att = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-att`,
      ).value;
      updateFieldDisplay(fieldDiv, field);
      updateSectionSummaries(divId, schema, lang);
    };
    attDiv.appendChild(addFileBtn);

    const saveAttBtn = document.createElement("button"); //บันทึกไฟล์แนบ
    saveAttBtn.textContent = getLabel(
      { label: { en: "Save Attachment", th: "Save Attachment" } },
      lang,
    );
    saveAttBtn.style.cssText =
      "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
    saveAttBtn.onclick = () => {
      schema[key].value.att = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-att`,
      ).value;
      document.getElementById(
        `${divId}-filecount-${encodeFieldKey(key)}`,
      ).textContent = `Files: ${countFiles(schema[key])}`;
      updateFieldDisplay(fieldDiv, field);
      updateSectionSummaries(divId, schema, lang);
    };
    attDiv.appendChild(saveAttBtn);
    input.appendChild(attDiv);
  } else if (inputType === "radio") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}" value="${val}" ${field.value === val ? "checked" : ""}> ${opt}`;
      input.appendChild(radioLabel);
    });
  } else if (inputType === "checkbox") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const checkLabel = document.createElement("label");
      checkLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      checkLabel.innerHTML = `<input type="checkbox" name="${divId}-${encodeFieldKey(key)}[]" value="${val}" ${field.value && field.value.includes(val) ? "checked" : ""}> ${opt}`;
      input.appendChild(checkLabel);
    });
  } else if (inputType === "select") {
    input = document.createElement("select");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.style.width = "260px";
    getOptions(field, lang).forEach((opt, idx) => {
      const option = document.createElement("option");
      const value = idx === 0 ? "" : opt;
      option.value = value;
      option.textContent = opt;
      option.selected = field.value === value;
      input.appendChild(option);
    });
  } else if (inputType === "attachment") {
    input = document.createElement("div");
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.id = `${divId}-${encodeFieldKey(key)}`;
    textInput.value = field.value || "";
    textInput.placeholder = getLabel(
      {
        label: {
          en: "Enter file names (comma-separated)",
          th: "ป้อนชื่อไฟล์ (คั่นด้วยเครื่องหมายจุลภาค)",
        },
      },
      lang,
    );
    textInput.style.width = "260px";
    input.appendChild(textInput);
    const fileCount = document.createElement("div");
    fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
    fileCount.style.marginTop = "5px";
    fileCount.textContent = `Files: ${countFiles(field)}`;
    input.appendChild(fileCount);
    const buttonDiv = document.createElement("div");
    buttonDiv.style.marginTop = "5px";
    const addFileBtn = document.createElement("button"); //เพิ่มไฟล์
    addFileBtn.textContent = getLabel(
      { label: { en: "Add file", th: "Add file" } },
      lang,
    );
    addFileBtn.style.cssText =
      "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
    addFileBtn.onclick = () => {
      window.logomode = false;
      toggleUploadPanel();
      updateFileListFromInput(`${divId}-${encodeFieldKey(key)}`);
      const count = countFiles(schema[key]);
      fileCount.textContent = `Files: ${count}`;
      schema[key].value = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      ).value;
      const fieldDiv = document.getElementById(
        `${divId}-field-${encodeFieldKey(key)}`,
      );
      if (fieldDiv) {
        fieldDiv.style.border =
          field.required && !isFieldAnswered(field) ? "2px solid red" : "";
      }
      const scoreBar = fieldDiv?.querySelector(".score-bar");
      const currentScore = calculateCurrentScore(field);
      const maxScore = calculateMaxScore(field);
      const weight = field.weights ? field.weights.join(", ") : "1";
      const fileCountVal = countFiles(field);
      const value = field.value || "";
      let valueText = "";
      if (typeof field.value === "object" && field.value !== null) {
        try {
          valueText = JSON.stringify(field.value);
        } catch (e) {
          valueText = "[object]";
        }
      } else {
        valueText = field.value || "";
      }
      if (scoreBar) {
        //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCountVal}, value: ${value}]`;
        if (window.debug)
          scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
        else
          scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
      }
      if (document.getElementById("jumperhelper")) {
        const summary = QSummary(schema).summary;
        updateJumperHelperWithUnanswered(summary, schema);
      }
      updateSectionSummaries(divId, schema, lang);
    };
    buttonDiv.appendChild(addFileBtn);
    const countFilesBtn = document.createElement("button");
    countFilesBtn.textContent = getLabel(
      { label: { en: "Save Attachment", th: "นับไฟล์" } },
      lang,
    );
    countFilesBtn.style.cssText =
      "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
    countFilesBtn.onclick = () => {
      schema[key].value = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      ).value;
      fileCount.textContent = `Files: ${countFiles(schema[key])}`;
      const fieldDiv = document.getElementById(
        `${divId}-field-${encodeFieldKey(key)}`,
      );
      if (fieldDiv) {
        fieldDiv.style.border =
          field.required && !isFieldAnswered(field) ? "2px solid red" : "";
      }
      const scoreBar = fieldDiv?.querySelector(".score-bar");
      const currentScore = calculateCurrentScore(field);
      const maxScore = calculateMaxScore(field);
      const weight = field.weights ? field.weights.join(", ") : "1";
      const fileCountVal = countFiles(field);
      const value = field.value || "";
      if (scoreBar) {
        //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCountVal}, value: ${value}]`;
        if (window.debug)
          scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
        else
          scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
      }
      if (document.getElementById("jumperhelper")) {
        const summary = QSummary(schema).summary;
        updateJumperHelperWithUnanswered(summary, schema);
      }
      updateSectionSummaries(divId, schema, lang);
    };
    buttonDiv.appendChild(countFilesBtn);
    input.appendChild(buttonDiv);
  } else if (inputType === "mixed") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";

    // Create a container for the mixed inputs
    const mixedContainer = document.createElement("div");
    mixedContainer.style.display = "flex";
    mixedContainer.style.gap = "10px";
    mixedContainer.style.alignItems = "center";

    // Add radio/select/checkbox options
    if (field.options) {
      const optionType = field.optionType || "radio"; // default to radio

      if (optionType === "radio") {
        getOptions(field, lang).forEach((opt, idx) => {
          const radioLabel = document.createElement("label");
          radioLabel.style.cssText =
            "margin-right: 8px; display: flex; align-items: center;";
          const val = idx;
          radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}-option" value="${val}" ${field.value?.option === val ? "checked" : ""}> ${opt}`;
          mixedContainer.appendChild(radioLabel);
        });
      } else if (optionType === "select") {
        const select = document.createElement("select");
        select.name = `${divId}-${encodeFieldKey(key)}-option`;
        getOptions(field, lang).forEach((opt, idx) => {
          const option = document.createElement("option");
          option.value = idx;
          option.textContent = opt;
          option.selected = field.value?.option === idx;
          select.appendChild(option);
        });
        mixedContainer.appendChild(select);
      } else if (optionType === "checkbox") {
        getOptions(field, lang).forEach((opt, idx) => {
          const checkLabel = document.createElement("label");
          checkLabel.style.cssText =
            "margin-right: 8px; display: flex; align-items: center;";
          const val = idx;
          checkLabel.innerHTML = `<input type="checkbox" name="${divId}-${encodeFieldKey(key)}-option[]" value="${val}" ${field.value?.options?.includes(val) ? "checked" : ""}> ${opt}`;
          mixedContainer.appendChild(checkLabel);
        });
      }
    }

    // Add text input
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.name = `${divId}-${encodeFieldKey(key)}-text`;
    textInput.value = field.value?.text || "";
    textInput.placeholder = field.textPlaceholder || "Enter text";
    textInput.style.flexGrow = "1";

    mixedContainer.appendChild(textInput);
    input.appendChild(mixedContainer);

    // Add file input if needed
    if (field.allowFile) {
      const fileInput = document.createElement("input");
      fileInput.type = "file";
      fileInput.name = `${divId}-${encodeFieldKey(key)}-file`;
      fileInput.style.marginTop = "5px";
      input.appendChild(fileInput);
    }

    return input;
  } else if (inputType === "radiotext" || inputType === "radiotextatt") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";

    // Create radio buttons container
    const radioContainer = document.createElement("div");
    radioContainer.style.cssText =
      "display: flex; flex-direction: column; gap: 5px;";

    // Add radio options
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}-radio" value="${val}" ${
        field.value && field.value.radio === val ? "checked" : ""
      }> ${opt}`;
      radioContainer.appendChild(radioLabel);
    });

    input.appendChild(radioContainer);

    // Add text input
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.id = `${divId}-${encodeFieldKey(key)}-text`;
    textInput.value = (field.value && field.value.text) || "";
    textInput.placeholder = getLabel(
      { label: { en: "Enter text", th: "ป้อนข้อความ" } },
      lang,
    );
    textInput.style.width = "260px";
    textInput.style.marginTop = "5px";
    input.appendChild(textInput);
    // Add attachment input for radiotextatt
    if (inputType === "radiotextatt") {
      const attDiv = document.createElement("div");
      attDiv.style.marginTop = "5px";

      const attInput = document.createElement("input");
      attInput.type = "text";
      attInput.id = `${divId}-${encodeFieldKey(key)}-att`;
      attInput.value = (field.value && field.value.att) || "";
      attInput.placeholder = getLabel(
        { label: { en: "Enter attachment", th: "ป้อนไฟล์แนบ" } },
        lang,
      );
      attInput.style.width = "260px";
      attDiv.appendChild(attInput);

      const fileCount = document.createElement("div");
      fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
      fileCount.style.marginTop = "5px";
      fileCount.textContent = `Files: ${countFiles(field)}`;
      input.appendChild(fileCount);

      const addFileBtn = document.createElement("button"); //เพิ่มไฟล์
      addFileBtn.textContent = getLabel(
        { label: { en: "Add file", th: "Add file" } },
        lang,
      );
      addFileBtn.style.cssText =
        "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
      addFileBtn.onclick = () => {
        window.logomode = false;
        toggleUploadPanel();
        updateFileListFromInput(`${divId}-${encodeFieldKey(key)}-att`);
        const count = countFiles(schema[key]);
        document.getElementById(
          `${divId}-filecount-${encodeFieldKey(key)}`,
        ).textContent = `Files: ${count}`;
        schema[key].value.att = document.getElementById(
          `${divId}-${encodeFieldKey(key)}-att`,
        ).value;
        //if (window.logomode = false)
        let fieldDiv = null; // Initialize with default value

        // Try to get the element if it should exist
        if (typeof field.elementId !== "undefined") {
          fieldDiv =
            document.getElementById(field.elementId) ||
            document.querySelector(`[data-field-id="${field.id}"]`);
        }

        // Then use it safely
        if (fieldDiv) {
          updateFieldDisplay(fieldDiv, field);
        } else {
          console.warn("fieldDiv not found for field:", field.id);
          // Optional fallback behavior
        }

        updateFieldDisplay(fieldDiv, field);
        updateSectionSummaries(divId, schema, lang);

        //        fileCount.textContent = `Files: ${count}`;
      };
      attDiv.appendChild(addFileBtn);
      const saveAttBtn = document.createElement("button");
      saveAttBtn.textContent = getLabel(
        { label: { en: "Save Attachment", th: "Save Attachment" } },
        lang,
      );
      saveAttBtn.style.cssText =
        "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
      saveAttBtn.onclick = () => {
        schema[key].value.att = document.getElementById(
          `${divId}-${encodeFieldKey(key)}-att`,
        ).value;
        document.getElementById(
          `${divId}-filecount-${encodeFieldKey(key)}`,
        ).textContent = `Files: ${countFiles(schema[key])}`;
        updateFieldDisplay(fieldDiv, field);
        updateSectionSummaries(divId, schema, lang);
      };
      attDiv.appendChild(saveAttBtn);
      input.appendChild(attDiv);
    }
  } else if (inputType === "checktext" || inputType === "checktextatt") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";

    // Create checkbox container
    const checkContainer = document.createElement("div");
    checkContainer.style.cssText =
      "display: flex; flex-direction: column; gap: 5px;";

    // Add checkbox options
    getOptions(field, lang).forEach((opt, idx) => {
      const checkLabel = document.createElement("label");
      checkLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      const isChecked = field.value?.check?.includes(val) || false;
      checkLabel.innerHTML = `<input type="checkbox" name="${divId}-${encodeFieldKey(key)}-check[]" value="${val}" ${isChecked ? "checked" : ""}> ${opt}`;
      checkContainer.appendChild(checkLabel);
    });

    input.appendChild(checkContainer);

    // Add text input
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.id = `${divId}-${encodeFieldKey(key)}-text`;
    textInput.value = (field.value && field.value.text) || "";
    textInput.placeholder = getLabel(
      { label: { en: "Enter text", th: "ป้อนข้อความ" } },
      lang,
    );
    textInput.style.width = "260px";
    textInput.style.marginTop = "5px";
    input.appendChild(textInput);

    // Add attachment input for checktextatt
    if (inputType === "checktextatt") {
      const attDiv = document.createElement("div");
      attDiv.style.marginTop = "5px";

      const attInput = document.createElement("input");
      attInput.type = "text";
      attInput.id = `${divId}-${encodeFieldKey(key)}-att`;
      attInput.value = (field.value && field.value.att) || "";
      attInput.placeholder = getLabel(
        { label: { en: "Enter attachment", th: "ป้อนไฟล์แนบ" } },
        lang,
      );
      attInput.style.width = "260px";
      attDiv.appendChild(attInput);

      const fileCount = document.createElement("div");
      fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
      fileCount.style.marginTop = "5px";
      fileCount.textContent = `Files: ${countFiles(field)}`;
      input.appendChild(fileCount);

      const addFileBtn = document.createElement("button"); //เพิ่มไฟล์
      addFileBtn.textContent = getLabel(
        { label: { en: "Add file", th: "Add file" } },
        lang,
      );
      addFileBtn.style.cssText =
        "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
      addFileBtn.onclick = () => {
        window.logomode = false;
        toggleUploadPanel();
        updateFileListFromInput(`${divId}-${encodeFieldKey(key)}-att`);
        const count = countFiles(schema[key]);
        document.getElementById(
          `${divId}-filecount-${encodeFieldKey(key)}`,
        ).textContent = `Files: ${count}`;
        schema[key].value.att = document.getElementById(
          `${divId}-${encodeFieldKey(key)}-att`,
        ).value;
        updateFieldDisplay(fieldDiv, field);
        updateSectionSummaries(divId, schema, lang);
      };
      attDiv.appendChild(addFileBtn);

      const saveAttBtn = document.createElement("button");
      saveAttBtn.textContent = getLabel(
        { label: { en: "Save Attachment", th: "Save Attachment" } },
        lang,
      );
      saveAttBtn.style.cssText =
        "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
      saveAttBtn.onclick = () => {
        schema[key].value.att = document.getElementById(
          `${divId}-${encodeFieldKey(key)}-att`,
        ).value;
        document.getElementById(
          `${divId}-filecount-${encodeFieldKey(key)}`,
        ).textContent = `Files: ${countFiles(schema[key])}`;
        updateFieldDisplay(fieldDiv, field);
        updateSectionSummaries(divId, schema, lang);
      };
      attDiv.appendChild(saveAttBtn);
      input.appendChild(attDiv);
    }
  } else {
    input = document.createElement("input");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.type = inputType === "string" ? "text" : inputType || "text";
    input.value = field.value || (inputType === "number" ? 0 : "");
    input.style.width = "260px";
  }

  return input;
}
// Helper function to get options (assumed to exist in original code)
function getOptions(field, lang) {
  return field.options?.[lang] || field.options?.en || [];
}
function getOptions(field, lang) {
  if (Array.isArray(field.options)) {
    return field.options; // Handle non-bilingual options
  }
  return field.options?.[lang] || field.options?.en || [];
}

// Helper function to get label (assumed to exist in original code)
function getLabel(field, lang) {
  return field.label?.[lang] || field.label?.en || "";
}

// Helper function to encode field key (assumed to exist in original code)
function encodeFieldKey(key) {
  return encodeURIComponent(key).replace(/'/g, "&#39;");
}
// Helper function to create reset button
function createResetButton(divId, key, field, input, lang, schema, fieldDiv) {
  const inputType = field.data_type || field.type;
  const resetBtn = document.createElement("button");
  resetBtn.textContent = getLabel(
    { label: { en: "reset", th: "reset" } },
    lang,
  );
  resetBtn.style.cssText = ` margin-left: 10px; padding: 4px 8px; cursor: pointer;
    background: #f44336;
           color: #ddd;
           border: none;
           border-radius: 10px;
           width: 38px;
           height: 15px;
           font-size: 12px;
           line-height: 1;
           cursor: pointer;
           margin-left: 5px;
           padding: 0px 15px;
           display: inline-flex;
           align-items: center;
           justify-content: center;
           transition: all 0.2s ease;
    `;
  resetBtn.onmouseover = function () {
    this.style.background = "#d32f2f";
    this.style.transform = "scale(1.1)";
  };
  resetBtn.onmouseout = function () {
    this.style.background = "#f44336";
    this.style.transform = "scale(1)";
  };
  resetBtn.onclick = () => {
    if (inputType === "radio") {
      schema[key].value = null;
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}"]`)
        .forEach((el) => {
          el.checked = false;
          fieldDiv.style.border = field.required ? "2px solid red" : "";
        });
    } else if (inputType === "radioatt") {
      // Reset radio and att components
      schema[key].value = { radio: null, att: "" };
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`)
        .forEach((el) => {
          el.checked = false;
        });
      const attInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-att`,
      );
      if (attInput) {
        attInput.value = "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    } else if (inputType === "checkbox") {
      schema[key].value = [];
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`)
        .forEach((el) => {
          el.checked = false;
          fieldDiv.style.border = field.required ? "2px solid red" : "";
        });
    } else if (inputType === "select") {
      schema[key].value = "";
      const inputEl = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      inputEl.value = "";
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    } else if (inputType === "attachment") {
      schema[key].value = "";
      const textInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      textInput.value = "";
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      document.getElementById(
        `${divId}-filecount-${encodeFieldKey(key)}`,
      ).textContent = `Files: 0`;
    } else if (inputType === "radiotext" || inputType === "radiotextatt") {
      // Reset both radio, text, and att components
      schema[key].value = { radio: null, text: "", att: "" };
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`)
        .forEach((el) => {
          el.checked = false;
        });
      const textInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-text`,
      );
      if (textInput) {
        textInput.value = "";
      }
      const attInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-att`,
      );
      if (attInput) {
        attInput.value = "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      //end radiotext case
    } else if (inputType === "checktext" || inputType === "checktextatt") {
      // Reset checkboxes, text, and att components
      schema[key].value = { check: [], text: "", att: "" };
      document
        .querySelectorAll(
          `input[name="${divId}-${encodeFieldKey(key)}-check[]"]`,
        )
        .forEach((el) => {
          el.checked = false;
        });
      const textInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-text`,
      );
      if (textInput) {
        textInput.value = "";
      }
      const attInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-att`,
      );
      if (attInput) {
        attInput.value = "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      //end checktext
    } else {
      // Handle string, number, textarea types
      schema[key].value = inputType === "number" ? 0 : "";
      const inputEl = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      if (inputEl) {
        inputEl.value = inputType === "number" ? "0" : "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    }

    // Update score bar
    const scoreBar = fieldDiv.querySelector(".score-bar");
    const currentScore = calculateCurrentScore(field);
    const maxScore = calculateMaxScore(field);
    const weight = field.weights ? field.weights.join(", ") : "1";
    const fileCount =
      field.data_type === "attachment" ||
      field.data_type === "radiotextatt" ||
      field.data_type === "checktextatt"
        ? countFiles(field)
        : 0;
    let valueText = "";
    if (typeof field.value === "object" && field.value !== null) {
      try {
        valueText = JSON.stringify(field.value);
      } catch (e) {
        valueText = "[object]";
      }
    } else {
      valueText = field.value || "";
    }
    if (scoreBar) {
      if (window.debug)
        scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
      else
        scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }

    // Update jumper helper and section summaries
    if (document.getElementById("jumperhelper")) {
      const summary = QSummary(schema).summary;
      updateJumperHelperWithUnanswered(summary, schema);
    }
    updateSectionSummaries(divId, schema, lang);
  };

  return resetBtn;
}

// Form input event listener
function listenform(divId) {
  const schema = document.getElementById(divId)._formData;
  const lang = window.lang || "en";

  for (const key in schema) {
    if (key === "grouping" || schema[key].type === "section") continue;
    const field = schema[key];
    const inputType = field.data_type || field.type;
    const inputId = `${divId}-${encodeFieldKey(key)}`;
    const fieldDiv = document.getElementById(
      `${divId}-field-${encodeFieldKey(key)}`,
    );

    if (["string", "textarea", "text", "number"].includes(inputType)) {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("input", () => {
          schema[key].value =
            inputType === "number"
              ? parseFloat(inputEl.value) || 0
              : inputEl.value.trim();
          fieldDiv.style.border =
            field.required && !isFieldAnswered(field) ? "2px solid red" : "";

          // Define valueText properly
          let valueText = "";
          if (typeof field.value === "object" && field.value !== null) {
            try {
              valueText = JSON.stringify(field.value);
            } catch (e) {
              valueText = "[object]";
            }
          } else {
            valueText = field.value || "";
          }
          // Update score bar
          const scoreBar = fieldDiv.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCount =
            field.data_type === "attachment" ? countFiles(field) : 0;
          const value = field.value || "";
          //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          if (window.debug)
            scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
          else
            scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
          // Update jumper helper and section summaries
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    }
    if (inputType === "radioatt") {
      // Handle radio button changes
      document
        .querySelectorAll(`input[name="${inputId}-radio"]`)
        .forEach((radio) => {
          radio.addEventListener("change", () => {
            if (!schema[key].value)
              schema[key].value = { radio: null, att: "" };
            schema[key].value.radio = parseInt(radio.value);
            updateFieldDisplay(fieldDiv, field);
            if (document.getElementById("jumperhelper")) {
              const summary = QSummary(schema).summary;
              updateJumperHelperWithUnanswered(summary, schema);
            }
            updateSectionSummaries(divId, schema, lang);
          });
        });

      // Handle attachment input changes
      const attInput = document.getElementById(`${inputId}-att`);
      if (attInput) {
        attInput.addEventListener("input", () => {
          if (!schema[key].value) schema[key].value = { radio: null, att: "" };
          schema[key].value.att = attInput.value.trim();
          document.getElementById(
            `${divId}-filecount-${encodeFieldKey(key)}`,
          ).textContent = `Files: ${countFiles(schema[key])}`;
          updateFieldDisplay(fieldDiv, field);
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "radio") {
      document.querySelectorAll(`input[name="${inputId}"]`).forEach((radio) => {
        radio.addEventListener("change", () => {
          schema[key].value = parseInt(radio.value);
          fieldDiv.style.border =
            field.required && !isFieldAnswered(field) ? "2px solid red" : "";
          // Update score bar
          const scoreBar = fieldDiv.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCount =
            field.data_type === "attachment" ? countFiles(field) : 0;
          //const value = field.value || "";
          // Define valueText properly
          let valueText = "";
          if (typeof field.value === "object" && field.value !== null) {
            try {
              valueText = JSON.stringify(field.value);
            } catch (e) {
              valueText = "[object]";
            }
          } else {
            valueText = field.value || "";
          }
          // scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          if (window.debug)
            scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
          else
            scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          // Update jumper helper and section summaries
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      });
    } else if (inputType === "checkbox") {
      document
        .querySelectorAll(`input[name="${inputId}[]"]`)
        .forEach((checkbox) => {
          checkbox.addEventListener("change", () => {
            const checked = Array.from(
              document.querySelectorAll(`input[name="${inputId}[]"]:checked`),
            ).map((el) => parseInt(el.value));
            schema[key].value = checked;
            fieldDiv.style.border =
              field.required && !isFieldAnswered(field) ? "2px solid red" : "";
            // Update score bar
            const scoreBar = fieldDiv.querySelector(".score-bar");
            const currentScore = calculateCurrentScore(field);
            const maxScore = calculateMaxScore(field);
            const weight = field.weights ? field.weights.join(", ") : "1";
            const fileCount =
              field.data_type === "attachment" ? countFiles(field) : 0;
            const value = field.value || "";
            if (window.debug)
              scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
            else
              scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;

            //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
            // Update jumper helper and section summaries
            if (document.getElementById("jumperhelper")) {
              const summary = QSummary(schema).summary;
              updateJumperHelperWithUnanswered(summary, schema);
            }
            updateSectionSummaries(divId, schema, lang);
          });
        });
    } else if (inputType === "select") {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("change", () => {
          schema[key].value = inputEl.value;
          fieldDiv.style.border =
            field.required && !isFieldAnswered(field) ? "2px solid red" : "";
          // Update score bar
          const scoreBar = fieldDiv.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCount =
            field.data_type === "attachment" ? countFiles(field) : 0;
          const value = field.value || "";
          scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          // Update jumper helper and section summaries
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "attachment") {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("input", () => {
          schema[key].value = inputEl.value;
          const fileCount = document.getElementById(
            `${divId}-filecount-${encodeFieldKey(key)}`,
          );
          fileCount.textContent = `Files: ${countFiles(schema[key])}`;
          const fieldDiv = document.getElementById(
            `${divId}-field-${encodeFieldKey(key)}`,
          );
          if (fieldDiv) {
            fieldDiv.style.border =
              field.required && !isFieldAnswered(field) ? "2px solid red" : "";
          }
          const scoreBar = fieldDiv?.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCountVal = countFiles(field);
          const value = field.value || "";
          if (scoreBar) {
            // scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCountVal}, value: ${value}]`;
            if (window.debug)
              scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
            else
              scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          }
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "radiotext" || inputType === "radiotextatt") {
      // Handle radio button changes
      document
        .querySelectorAll(`input[name="${inputId}-radio"]`)
        .forEach((radio) => {
          radio.addEventListener("change", () => {
            if (!schema[key].value)
              schema[key].value = { radio: null, text: "", att: "" };
            schema[key].value.radio = parseInt(radio.value);
            updateFieldDisplay(fieldDiv, field);
            if (document.getElementById("jumperhelper")) {
              const summary = QSummary(schema).summary;
              updateJumperHelperWithUnanswered(summary, schema);
            }
            updateSectionSummaries(divId, schema, lang);
          });
        });

      // Handle text input changes
      const textInput = document.getElementById(`${inputId}-text`);
      if (textInput) {
        textInput.addEventListener("input", () => {
          if (!schema[key].value) schema[key].value = { radio: null, text: "" };
          schema[key].value.text = textInput.value.trim();
          updateFieldDisplay(fieldDiv, field);
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }

      if (inputType === "radiotextatt") {
        const attInput = document.getElementById(`${inputId}-att`);
        if (attInput) {
          attInput.addEventListener("input", () => {
            if (!schema[key].value)
              schema[key].value = { radio: null, text: "", att: "" };
            schema[key].value.att = attInput.value.trim();
            updateFieldDisplay(fieldDiv, field);
            if (document.getElementById("jumperhelper")) {
              const summary = QSummary(schema).summary;
              updateJumperHelperWithUnanswered(summary, schema);
            }
            updateSectionSummaries(divId, schema, lang);
          });
        }
      } else if (inputType === "checktext" || inputType === "checktextatt") {
        // Handle checkbox changes
        document
          .querySelectorAll(`input[name="${inputId}-check[]"]`)
          .forEach((checkbox) => {
            checkbox.addEventListener("change", () => {
              if (!schema[key].value)
                schema[key].value = { check: [], text: "", att: "" };
              const checked = Array.from(
                document.querySelectorAll(
                  `input[name="${inputId}-check[]"]:checked`,
                ),
              ).map((el) => parseInt(el.value));
              schema[key].value.check = checked;
              updateFieldDisplay(fieldDiv, field);
              if (document.getElementById("jumperhelper")) {
                const summary = QSummary(schema).summary;
                updateJumperHelperWithUnanswered(summary, schema);
              }
              updateSectionSummaries(divId, schema, lang);
            });
          });

        // Handle text input changes
        const textInput = document.getElementById(`${inputId}-text`);
        if (textInput) {
          textInput.addEventListener("input", () => {
            if (!schema[key].value)
              schema[key].value = { check: [], text: "", att: "" };
            schema[key].value.text = textInput.value.trim();
            updateFieldDisplay(fieldDiv, field);
            if (document.getElementById("jumperhelper")) {
              const summary = QSummary(schema).summary;
              updateJumperHelperWithUnanswered(summary, schema);
            }
            updateSectionSummaries(divId, schema, lang);
          });
        }

        if (inputType === "checktextatt") {
          const attInput = document.getElementById(`${inputId}-att`);
          if (attInput) {
            attInput.addEventListener("input", () => {
              if (!schema[key].value)
                schema[key].value = { check: [], text: "", att: "" };
              schema[key].value.att = attInput.value.trim();
              document.getElementById(
                `${divId}-filecount-${encodeFieldKey(key)}`,
              ).textContent = `Files: ${countFiles(schema[key])}`;
              updateFieldDisplay(fieldDiv, field);
              if (document.getElementById("jumperhelper")) {
                const summary = QSummary(schema).summary;
                updateJumperHelperWithUnanswered(summary, schema);
              }
              updateSectionSummaries(divId, schema, lang);
            });
          }
        }
      }
      //end radiotext case
    }
  }

  // Initial update of jumper helper and section summaries
  if (document.getElementById("jumperhelper")) {
    const summary = QSummary(schema).summary;
    updateJumperHelperWithUnanswered(summary, schema);
  }
  updateSectionSummaries(divId, schema, lang);
}


function renderSchemaStructureToSumman(schema, divId, lang = "en") {
  const segments = new Map();
  let globalTotalFields = 0;
  let globalRequiredFields = 0;
  let globalAnsweredFields = 0;
  let globalAnsweredRequiredFields = 0;
  let globalunAnsweredRequiredFields = 0;
  let globalTotalFiles = 0; // Add global file counter

  // 1. Prepare segment structure
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  } else {
    segments.set("default", { label: "Default", sections: new Map() });
  }

  // 2. Organize fields into segments/sections and count
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];

    const segment =
      field.segment?.[0] || schema.grouping?.name?.[0] || "default";
    if (!segments.has(segment)) {
      segments.set(segment, { label: segment, sections: new Map() });
    }

    if (field.type === "section") {
      segments.get(segment).sections.set(key, {
        label: getLabel(field, lang) || key,
        fields: [],
      });
    } else if (field.section) {
      const section = field.section;
      if (!segments.get(segment).sections.has(section)) {
        if (schema[section]) {
          segments.get(segment).sections.set(section, {
            label: getLabel({ label: schema[section].label }, lang) || section,
            fields: [],
          });
        }
      }

      const sectionObj = segments.get(segment).sections.get(section);
      if (sectionObj) {
        sectionObj.fields.push({
          key,
          label: getLabel(field, lang) || key,
        });
      }

      // Count for global progress
      globalTotalFields++;
      const answered = isFieldAnswered(field);
      if (answered) globalAnsweredFields++;
      if (field.required) {
        globalRequiredFields++;
        if (answered) globalAnsweredRequiredFields++; else globalunAnsweredRequiredFields++
      }
      // Count files for all field types that can have them
      // Add this file counting block:
      if (field.data_type === "attachment") {
        globalTotalFiles += countFiles(field);
      } else if (field.data_type === "radiotextatt" && field.value?.att) {
        globalTotalFiles += countFiles({
          value: field.value.att,
          data_type: "attachment",
        });
      } else if (field.data_type === "radioatt" && field.value?.att) {
        // Add this case
        globalTotalFiles += countFiles({
          value: field.value.att,
          data_type: "attachment",
        });
      }
    }
  }

  // 3. Calculate percentages and set window.allowsubmit
  const percentAnsweredAll =
    globalTotalFields > 0
      ? Math.round((globalAnsweredFields / globalTotalFields) * 100)
      : 100;

  const percentAnsweredRequired =
    globalRequiredFields > 0
      ? Math.round((globalAnsweredRequiredFields / globalRequiredFields) * 100)
      : 100;

  let status = "Not Active";
  window.allowsubmit = false;

  if (globalAnsweredRequiredFields === 0) {
    status = "⛔ Not Active";
  } else if (globalAnsweredRequiredFields < globalRequiredFields) {
    status = "🔄 In Progress";
  } else {
    status = "✅ READY";
    window.allowsubmit = true;
  }

  // 4. Build HTML 📊
  let html = `<div style="font-size: 0.95em; font-family: sans-serif;">
    <div style="margin-bottom: 15px;">
      <strong>Summary ${status}</strong><br><hr>
      Total fields: <b>${globalTotalFields}</b><br>
      Total required: <b>${globalRequiredFields}</b><br>
      Answered fields: <b>${globalAnsweredFields}</b><br>
      Answered required: <b>${globalAnsweredRequiredFields}</b><br>
        UnAnswered required: <b style="color: red">${globalunAnsweredRequiredFields}</b><br>

      Total files: <b>${globalTotalFiles}</b><br><br>  <!-- Add files count -->

      <div style="margin-bottom: 10px;">
        <div>[Answered: ${percentAnsweredAll}%]</div>
        <div style="background: #eee; height: 6px; border-radius: 5px;">
          <div style="width: ${percentAnsweredAll}%; background: #2196f3; height: 6px; border-radius: 5px;"></div>
        </div>
      </div>

      <div>
        <div>[Answered Required: ${percentAnsweredRequired}%]</div>
        <div style="background: #eee; height: 6px; border-radius: 5px;">
          <div style="width: ${percentAnsweredRequired}%; background: #4caf50; height: 6px; border-radius: 5px;"></div>
        </div>
      </div>
    </div>`;

  // 5. Segment → Section → Fields  📁 color: #2a5; 📂 color: #06c; 🔑 color: #c00;  ${field.key}:
  // Helper function to count required and answered fields in a section
  const countSectionStats = (fields) => {
    let required = 0;
    let answeredRequired = 0;
    let unAnsweredRequired = 0;
    
    fields.forEach(field => {
      const fieldObj = schema[field.key];
      if (fieldObj?.required) {
        required++;
        if (isFieldAnswered(fieldObj)) {
          answeredRequired++;
        } else unAnsweredRequired++;
      }
    });
    
    return { required, answeredRequired,unAnsweredRequired };
  };

  // 5. Segment → Section → Fields with expand/collapse
  segments.forEach((seg, segId) => {
    // Count segment stats
    let segRequired = 0;
    let segAnsweredRequired = 0;
    let segTotalFields = 0;
    let segAnsweredFields = 0;
      let segUnanswered = 0;

    
    seg.sections.forEach(section => {
      section.fields.forEach(field => {
        const fieldObj = schema[field.key];
        segTotalFields++;
        if (isFieldAnswered(fieldObj)) segAnsweredFields++;
        if (fieldObj?.required) {
          segRequired++;
          if (isFieldAnswered(fieldObj)) segAnsweredRequired++; else  segUnanswered++;
        }
        
      });
    });
    
    const segPercent = segRequired > 0 ? Math.round((segAnsweredRequired / segRequired) * 100) : 100;
    scrollToField('${encodeFieldKey(field.key)}', '${divId}')
    //jumpsg=divId+'-segment_'+segId;
    //const jumpsg = `${(segId)}`;
    const jumpsg = `${divId}-segment-${encodeFieldKey(segId)}`;
    
    html += `<div style="margin-bottom: 10px;">
      <div style="text-decoration: none; font-size: 14px; cursor: pointer;" onclick="toggleSection(this, 'seg_${segId}')"><b>${seg.label}</b> (${segAnsweredRequired}/${segRequired}, ${segPercent}%,Unanswered: <b style="color: red">${segUnanswered}</b>)
      </div> <a href="javascript:scrollTo('${jumpsg}', '${divId}')" 
       style="margin-left: 10px; font-size: 0.8em; text-decoration: none;">
       [Jump ${jumpsg} ]
    </a>
      <div id="seg_${segId}" style=" margin-left: 5px;">`;
    
    seg.sections.forEach((section, secId) => {
      // Count section stats
      const stats = countSectionStats(section.fields);
      const secPercent = stats.required > 0 ? Math.round((stats.answeredRequired / stats.required) * 100) : 100;
      //jumpsc=divId+'-section_'+secId;
      const jumpsc = `${divId}-section-${encodeFieldKey(secId)}`;
      html += `<div style="margin-left: 0px;">
        <div style="font-size: 13px; cursor: pointer;" onclick="toggleSection(this, 'sec_${segId}_${secId}')">
          <b>${section.label}</b> (${stats.answeredRequired}/${stats.required}, ${secPercent}%,: UnAnswered:<b style="color: red">${stats.unAnsweredRequired}</b>)
        </div><a href="javascript:scrollTo('${jumpsc}', '${divId}')" 
       style="margin-left: 10px; font-size: 0.8em; text-decoration: none;">
       [Jump ${jumpsc} ]
    </a>
        <div id="sec_${segId}_${secId}" style=" margin-left: 0px;">`;
      
      section.fields.forEach((field) => {
        const fieldObj = schema[field.key];
        const isAnswered = fieldObj ? isFieldAnswered(fieldObj) : false;
        const dotColor = isAnswered ? 'green' : 'red';
        const isRequired = fieldObj?.required ? ' (Required)' : '';
        
        html += `<div style="margin-left: 0px; display: flex; align-items: center; min-height: 20px;">
        <span style="
            display: inline-block; 
            min-width: 8px; 
            min-height: 8px; 
            width: 8px; 
            height: 8px; 
            border-radius: 50%; 
            background: ${dotColor}; 
            margin-right: 4px;
            flex-shrink: 0;
        "></span>
          <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${divId}')"
             style="
             font-size: 14px;
               text-decoration: none; 
               color: inherit;
               white-space: nowrap;
               overflow: hidden;
               text-overflow: ellipsis;
               max-width: 90%;
           "
             title="${field.label}">
              ${field.label}${isRequired}
          </a>
        </div>`;
      });
      
      html += `</div></div>`;
    });
    
    html += `</div></div>`;
  });

  html += `</div>`;

  // Add the toggle function to the window object
  window.toggleSection = function(element, id) {
    const content = document.getElementById(id);
    if (content.style.display === "none") {
      content.style.display = "block";
      element.innerHTML = element.innerHTML.replace("▶", "▼");
    } else {
      content.style.display = "none";
      element.innerHTML = element.innerHTML.replace("▼", "▶");
    }
  };

  // 6. Output to #summan
  const summaryDiv = document.getElementById("summan");
  if (summaryDiv) {
    summaryDiv.innerHTML = html;
  } else {
    console.warn("Div with id 'summan' not found");
  }
}
// Add these functions to your window object
window.scrollToSegment = function(segmentId) {
  const element = document.getElementById(segmentId);
  if (element) {
    // Use your existing scrollToField logic but for segments
    scrollToField(segmentId, 'summan');
    // Highlight temporarily
    element.style.backgroundColor = '#f5f5f5';
    setTimeout(() => element.style.backgroundColor = '', 1000);
  }
};

window.scrollToSection = function(sectionId) {
  const element = document.getElementById(sectionId);
  if (element) {
    // Use your existing scrollToField logic but for sections
    scrollToField(sectionId, 'summan');
    // Highlight temporarily
    element.style.backgroundColor = '#f0f0f0';
    setTimeout(() => element.style.backgroundColor = '', 1000);
  }
};
function updateFieldDisplay(fieldDiv, field) {
  // Update border based on required status and whether the field is answered
  fieldDiv.style.border =
    field.required && !isFieldAnswered(field) ? "2px solid red" : "";

  // Update score bar
  const scoreBar = fieldDiv.querySelector(".score-bar");
  if (scoreBar) {
    const currentScore = calculateCurrentScore(field);
    const maxScore = calculateMaxScore(field);
    const weight = field.weights ? field.weights.join(", ") : "1";

    // Get file count based on type
    //let fileCount = 0;
    fileCount = countFiles(field);
    let valueDisplay = "";

    if (field.data_type === "attachment") {
      fileCount = countFiles(field);
    } else if (field.data_type === "radiotextatt" && field.value?.att) {
      fileCount = countFiles(field);
    }

    let valueText = "";
    if (typeof field.value === "object" && field.value !== null) {
      try {
        valueText = JSON.stringify(field.value);
      } catch (e) {
        valueText = "[object]";
      }
    } else {
      valueText = field.value || "";
    }
    if (field.data_type === "radioatt") {
      valueDisplay = `Selected: ${field.value?.radio ?? "none"}`;
      if (fileCount > 0) {
        valueDisplay += `, Files: ${fileCount}`;
      }
    }
    if (window.debug) {
      scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    } else {
      scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }
  }
}

function updateSumman(schema) {
  if (!schema) return;
  let totalFields = 0;
  let answeredFields = 0;

  for (const key in schema) {
    if (key === "grouping" || schema[key].type === "section") continue;
    totalFields++;
    if (isFieldAnswered(schema[key])) {
      answeredFields++;
    }
  }

  const summanDiv = document.getElementById("summan");
  if (summanDiv) {
    summanDiv.textContent = `Total fields: ${totalFields} | Answered: ${answeredFields}`;
  }
}
//
function calculateCurrentScore(field) {
  if (!isFieldAnswered(field)) return 0;

  if (field.weights) {
    // Handle radio/radioatt types
    if (field.data_type === "radio" || field.data_type === "radioatt") {
      const radioValue =
        field.data_type === "radioatt" ? field.value?.radio : field.value;
      return field.weights[radioValue] || 0;
    }
    // Handle checkbox types
    if (field.data_type === "checkbox") {
      return field.value.reduce(
        (sum, idx) => sum + (field.weights[idx] || 0),
        0,
      );
    }
    // Handle radiotext/checktext types
    if (field.data_type === "radiotext" || field.data_type === "radiotextatt") {
      return field.weights[field.value.radio] || 0;
    }
    if (field.data_type === "checktext" || field.data_type === "checktextatt") {
      return field.value.check.reduce(
        (sum, idx) => sum + (field.weights[idx] || 0),
        0,
      );
    }
  }
  return field.score || (isFieldAnswered(field) ? 1 : 0);
}

function calculateMaxScore(field) {
  if (field.weights) {
    // For radio/select types - max weight
    if (
      ["radio", "radioatt", "radiotext", "radiotextatt"].includes(
        field.data_type,
      )
    ) {
      return Math.max(...field.weights);
    }
    // For checkbox types - sum of all positive weights
    if (["checkbox", "checktext", "checktextatt"].includes(field.data_type)) {
      return field.weights.filter((w) => w > 0).reduce((sum, w) => sum + w, 0);
    }
  }
  return field.score || 1;
}
function calculateScoreAchieved(field) {
  const currentScore = calculateCurrentScore(field);
  return (field.score || 1) * currentScore;
}
// Helper function to update section summaries
function updateSectionSummaries(divId, schema, lang) {
  // Accumulate global totals from all segments and sections
  let globalTotalFields = 0;
  let globalAnsweredFields = 0;
  globalRequiredFields = 0;
  let globalAnsweredRequiredFields = 0;
  const segments = new Map();
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  } else {
    console.warn("No grouping defined in schema, using default segment");
    segments.set("default", { label: "Default", sections: new Map() });
  }

  // Map fields to segments and sections
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];
    const segment = field.segment?.[0] || schema.grouping?.name[0] || "default";
    if (!segments.has(segment)) {
      console.log(`Creating default segment for ${segment}`);
      segments.set(segment, { label: segment, sections: new Map() });
    }
    if (field.type === "section") {
      segments.get(segment).sections.set(key, getLabel(field, lang) || key);
    } else if (field.section) {
      if (!segments.get(segment).sections.has(field.section)) {
        if (schema[field.section]) {
          segments
            .get(segment)
            .sections.set(
              field.section,
              getLabel({ label: schema[field.section]?.label }, lang) ||
                field.section,
            );
        } else {
          console.warn(
            `Section ${field.section} for field ${key} not found in schema`,
          );
        }
      }
    } else {
      console.warn(`Field ${key} has no section assigned`);
    }
  }

  segments.forEach((segmentData, segmentId) => {
    console.log(`Processing segment: ${segmentId}`);
    segmentData.sections.forEach((sectionLabel, sectionId) => {
      console.log(`Processing section: ${sectionId} in segment: ${segmentId}`);
      let sectionTotals = {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredFields: [],
      };

      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (
          field.section !== sectionId ||
          !field.segment?.includes(segmentId)
        ) {
          console.log(
            `Skipping field ${key} in render: section=${field.section}, segment=${field.segment}, expected section=${sectionId}, segment=${segmentId}`,
          );
          continue;
        }

        console.log(
          `Rendering field: ${key}, type: ${field.data_type || field.type}`,
        );
        hasField = true;
        const fieldDiv = document.createElement("div");
        fieldDiv.className = "form-field";
        const encodedKey = encodeFieldKey(key);
        fieldDiv.id = `${divId}-field-${encodedKey}`;
        fieldDiv.style.cssText =
          "background-color: #f9f9f9; padding: 10px; margin-bottom: 10px; border-radius: 4px;";
        if (field.required && !isFieldAnswered(field)) {
          fieldDiv.style.border = "2px solid red";
        }
        if (!field.data_type && !field.type) {
          console.warn(`Field ${key} has no data_type or type defined`);
          continue;
        }
        globalTotalFields += sectionTotals.allFields;

        console.log(
          `Processing field: ${key}, type: ${field.data_type || field.type}`,
        );
        let fileCount = 0;

        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const scoreAchieved = calculateScoreAchieved(field);

        const isAnswered = isFieldAnswered(field);
        const isFile =
          field.data_type === "attachment" || field.type === "attachment";
        fileCount = isFile ? countFiles(field) : 0;
        if (field.data_type === "attachment") {
          fileCount = countFiles(field);
        } else if (field.data_type === "radiotextatt" && field.value?.att) {
          fileCount = countFiles({
            value: field.value.att,
            data_type: "attachment", // Treat it as attachment for counting
          });
        } else if (field.data_type === "radioatt" && field.value?.att) {
          // ADDED THIS CASE
          fileCount = countFiles({
            value: field.value.att,
            data_type: "attachment",
          });
        }
        sectionTotals.allFields++;
        if (field.required) sectionTotals.requiredFields++;
        if (isAnswered) sectionTotals.answeredFields++;
        if (field.required && isAnswered)
          sectionTotals.answeredRequiredFields++;
        sectionTotals.totalScore += isAnswered ? field.score || 1 : 0;
        sectionTotals.totalScoreWeighted += currentScore;
        sectionTotals.maxPossibleScore += field.score || 1;
        sectionTotals.maxPossibleScoreWeighted += maxScore;
        sectionTotals.totalFiles += fileCount;

        if (isFile) {
          console.log(
            `Attachment field ${key}: value="${field.value}", files=${fileCount}`,
          );
        }

        if (field.required && !isFieldAnswered(field)) {
          const encodedKey = encodeFieldKey(key);
          const fieldElement = document.getElementById(
            `${divId}-field-${encodedKey}`,
          );
          if (fieldElement) {
            sectionTotals.unansweredFields.push({
              key,
              label: getLabel(field, lang) || key,
              elementId: `${divId}-field-${encodedKey}`,
            });
          } else {
            console.warn(
              `Field ${key} not rendered in DOM (ID: ${divId}-field-${encodedKey}), skipping from unanswered list`,
            );
          }
        }
      }

      console.log(
        `Section ${sectionId} totals: ${JSON.stringify(sectionTotals, null, 2)}`,
      );

      const sectionSummary = document.getElementById(
        `${divId}-summary-${encodeFieldKey(segmentId)}-${encodeFieldKey(sectionId)}`,
      );
      if (sectionSummary) {
        //window.debug=true;
        const unansweredCount =
          sectionTotals.requiredFields - sectionTotals.answeredRequiredFields;
        // In the section summary HTML generation
        // In the section summary HTML generation

        a = window.debug
          ? `[field total: ${sectionTotals.allFields}, score: ${sectionTotals.totalScore}, achieved score: ${sectionTotals.totalScoreWeighted}, maxscore: ${sectionTotals.maxPossibleScoreWeighted}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]`
          : `[field total: ${sectionTotals.allFields}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]`;
        sectionSummary.innerHTML =
          `
          <summary style="cursor: pointer;">UnAnswered/Total/Required (${unansweredCount}/${sectionTotals.allFields}/${sectionTotals.requiredFields})</summary>
          <div style="padding: 8px;">
            ` +
          a +
          `${
            sectionTotals.unansweredFields.length > 0
              ? `
              <div style="margin-top: 10px; color: red;">
                <strong>Unanswered Required Fields:</strong>
                <ul style="margin: 5px 0; padding-left: 20px;">
                  ${sectionTotals.unansweredFields
                    .map(
                      (field) => `
                    <li style=" white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                      <a href="javascript:scrollToField('${encodeFieldKey(
                        field.key,
                      )}', '${divId}')"
                         style="text-decoration: none; color: red;"
                         title="${field.label}">
                        ${field.label}
                      </a>
                    </li>
                  `,
                    )
                    .join("")}
                </ul>
              </div>
            `
              : ""
          }
          </div>
        `;
        if (unansweredCount > 0) {
          sectionSummary.classList.add("unanswered");
        } else {
          sectionSummary.classList.remove("unanswered");
        }
      }

      const sectionHeader = document.getElementById(
        `${divId}-section-header-${encodeFieldKey(sectionId)}`,
      );
      if (sectionHeader) {
        const progressPercent =
          sectionTotals.requiredFields > 0
            ? Math.round(
                (sectionTotals.answeredRequiredFields /
                  sectionTotals.requiredFields) *
                  100,
              )
            : 100;
        globalAnsweredFields += sectionTotals.answeredRequiredFields;
        globalAnsweredRequiredFields += sectionTotals.answeredRequiredFields;
        globalRequiredFields += sectionTotals.requiredFields;
        sectionHeader.innerHTML = `
          <div style="font-weight: bold;">${sectionLabel}</div>
          <div style="margin-top: 5px;">
            [Progress: ${progressPercent}% (${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields})]
            <div style="width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;">
              <div style="width: ${progressPercent}%; background-color: #4caf50; height: 100%;"></div>
            </div>
          </div>
        `;
      }
    });
  });
  // Update your summary div by id 'summan'
  const summanDiv = document.getElementById("summan");
  if (summanDiv) {
    summanDiv.textContent = `Total fields: ${globalTotalFields} | ${globalRequiredFields} Answered required fields: ${globalAnsweredFields}`;
    renderSchemaStructureToSumman(schema, divId, lang);
  }
}

function updateJumperHelperWithUnanswered(
  summary,
  data,
  targetDivId = "jumperhelper",
) {
  const lang = window.lang || "en";
  if (!summary || !summary.segments || !summary.totals) {
    console.error("Invalid summary object:", summary);
    return "";
  }
  const panelStack = window.panelStack || [];
  const formId =
    panelStack.length > 0 ? panelStack[panelStack.length - 1] : "gform";

  if (formId === "gen") window.activeFormId = "gform";
  else if (formId === "profile") window.activeFormId = "pro";
  const generateId = (prefix) =>
    `${prefix}-${Math.random().toString(36).substr(2, 9)}`;
  const filesToggleId = generateId("files-toggle");
  // Remove the generateScrollFunction and use scrollToField consistently
  const generateScrollLink = (key) => {
    return `javascript:scrollToField('${key}', '${activeFormId}')`;
  };
  let html = "<div style='font-family: Arial, sans-serif;'>";

  // Grand Summary with unanswered fields
  const globalTotals = summary.totals;
  const fieldsPercent =
    globalTotals.allFields > 0
      ? Math.round((globalTotals.answeredFields / globalTotals.allFields) * 100)
      : 100;
  const requiredPercent =
    globalTotals.requiredFields > 0
      ? Math.round(
          (globalTotals.answeredRequiredFields / globalTotals.requiredFields) *
            100,
        )
      : 100;

  html += `<div style='border: 2px solid #4a90e2; padding: 10px; margin-bottom: 20px; border-radius: 5px;'>
            <h4 style='margin: 0 0 5px 0;'>Grand Summary</h4>
            <div style='margin-bottom: 5px;'>
              Total Fields: ${globalTotals.answeredFields}/${globalTotals.allFields}
              <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
              </div>
            </div>
            <div style='margin-bottom: 5px;'>
              Required: ${globalTotals.answeredRequiredFields}/${globalTotals.requiredFields}
              <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
              </div>
            </div>
            <div style='margin-bottom: 5px;'>
                   Total Files: <b>${globalTotals.totalFiles}</b>  <!-- Add files display -->
                 </div>`;

  // Show unanswered required fields at global level
  if (globalTotals.unansweredRequiredFields?.length > 0) {
    //const globalUnansweredToggleId = generateId("global-unanswered-toggle");

    html += `<div style='margin-top: 10px; color: orange;'>
              Unanswered Required Fields:
              <ul style='margin: 5px 0; padding-left: 20px;'>`;

    globalTotals.unansweredRequiredFields.forEach((field) => {
      html += `<li style='max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
                <a href="${generateScrollLink(field.key)}"
                   style='text-decoration: none; color: red;'
                   title="${field.label}">
                  ${field.label}
                </a>
              </li>`;
    });

    html += `</ul></div>`;
  }

  html += `</div>`;

  // Segments with their unanswered fields
  html += "<div style='display: flex; flex-wrap: wrap; gap: 10px;'>";
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    const segmentTotals = segment.totals || {};
    const segmentId = `gform-segment-${encodeFieldKey(segmentKey)}`;
    const segmentToggleId = generateId("segment-toggle");

    const fieldsPercent =
      segmentTotals.allFields > 0
        ? Math.round(
            (segmentTotals.answeredFields / segmentTotals.allFields) * 100,
          )
        : 100;
    const requiredPercent =
      segmentTotals.requiredFields > 0
        ? Math.round(
            (segmentTotals.answeredRequiredFields /
              segmentTotals.requiredFields) *
              100,
          )
        : 100;

    html += `<div style='border: 1px solid #ccc; padding: 10px; border-radius: 5px; min-width: 200px;'>
              <h4 style='margin: 0 0 5px 0;'>
                  <a href="${generateScrollLink(segmentKey)}"
                     style='text-decoration: none; color: #333;'
                     title="Jump to ${segment.label || segmentKey}">
                      ${segment.label || segmentKey}
                  </a>
              </h4>
              <div style='margin-bottom: 5px;'>
                Fields: ${segmentTotals.answeredFields}/${segmentTotals.allFields}
                <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                  <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
                </div>
              </div>
              <div style='margin-bottom: 5px;'>
                Required: ${segmentTotals.answeredRequiredFields}/${segmentTotals.requiredFields}
                <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                  <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
                </div>
              </div>`;

    // Show segment's unanswered required fields
    if (segmentTotals.unansweredRequiredFields?.length > 0) {
      html += `<div style='margin-top: 5px; color: orange; font-size: 0.9em;'>
                Unanswered:
                <ul style='margin: 3px 0; padding-left: 15px;'>`;

      segmentTotals.unansweredRequiredFields.forEach((field) => {
        const fieldId = `gform-field-${encodeFieldKey(field.key)}`;
        html += `<li style='max-width: 180px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
                  <a href="${generateScrollLink(field.key)}"
                     style='text-decoration: none; color: red;'
                     title="${field.label}">
                    ${field.label}
                  </a>
                </li>`;
      });

      html += `</ul></div>`;
    }

    // Sections within this segment
    html += `<div style='margin-top: 10px;'>`;
    Object.entries(segment.sections || {}).forEach(([sectionKey, section]) => {
      const sectionId = `gform-section-${encodeFieldKey(sectionKey)}`;
      const sectionTotals = section.totals || {};

      html += `<div style='margin-bottom: 5px;'>
                <a href="${generateScrollLink(sectionKey)}"
                   style='text-decoration: none; color: #666;'
                   title="Jump to ${section.label || sectionKey}">
                  ${section.label || sectionKey} (${sectionTotals.answeredFields}/${sectionTotals.allFields})
                </a>
              </div>`;
    });
    html += `</div>`;

    html += `</div>`;
  });
  html += `</div></div>`;

  const jumperDiv = document.getElementById(targetDivId);
  if (jumperDiv) {
    jumperDiv.innerHTML = html;
  } else {
    console.error(`Target div ${targetDivId} not found`);
  }

  return html;
}

// Summary calculation function

function QSummary(data) {
  if (!data || !data.grouping || !data.grouping.name) {
    console.warn("Invalid data format: grouping is missing");
    return { segments: [], summary: { segments: {}, totals: {} } };
  }

  const result = { segments: [] };
  const summary = {
    segments: {},
    totals: {
      allFields: 0,
      requiredFields: 0,
      answeredFields: 0,
      answeredRequiredFields: 0,
      totalScore: 0,
      totalScoreWeighted: 0,
      maxPossibleScore: 0,
      maxPossibleScoreWeighted: 0,
      totalFiles: 0,
      unansweredRequiredFields: [],
    },
  };

  const segmentMap = {};

  Object.entries(data).forEach(([key, item]) => {
    if (key === "grouping" || item.type === "section") return;

    const segment = item.segment?.[0] || data.grouping?.name[0] || "default";
    const section = item.section;
    if (!segment || !section || !data[section]) {
      console.warn(
        `Invalid field ${key}: missing segment or section, or section not found in schema`,
      );
      return;
    }

    // Initialize segment/section if not exists
    if (!summary.segments[segment]) {
      const segmentIndex = data.grouping.name.indexOf(segment);
      summary.segments[segment] = {
        label: data.grouping.label?.en[segmentIndex] || segment,
        sections: {},
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [],
        },
      };
    }
    if (!summary.segments[segment].sections[section]) {
      summary.segments[segment].sections[section] = {
        label: data[section]?.label?.en || section,
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [],
        },
      };
    }

    const currentScore = calculateCurrentScore(item);
    const maxScore = calculateMaxScore(item);
    const isAnswered = isFieldAnswered(item);

    // Handle file counting for all field types
    let fileCount = 0;
    if (item.data_type === "attachment") {
      fileCount = countFiles(item);
    } else if (item.data_type === "radiotextatt") {
      // Create a temporary field object just for counting
      fileCount = countFiles({
        value: item.value?.att || "",
        data_type: "attachment",
      });
    } else if (item.data_type === "radiotextatt" && item.value?.att) {
      fileCount = countFiles({
        value: item.value.att,
        data_type: "attachment",
      });
    } else if (item.data_type === "radioatt" && item.value?.att) {
      // Added for radioatt
      fileCount = countFiles({
        value: item.value.att,
        data_type: "attachment",
      });
    }

    // Update totals at all levels
    const sectionTotals = summary.segments[segment].sections[section].totals;
    const segmentTotals = summary.segments[segment].totals;
    const globalTotals = summary.totals;

    // Section level
    sectionTotals.allFields++;
    if (item.required) sectionTotals.requiredFields++;
    if (isAnswered) sectionTotals.answeredFields++;
    if (item.required && isAnswered) sectionTotals.answeredRequiredFields++;
    sectionTotals.totalScore += isAnswered ? item.score || 1 : 0;
    sectionTotals.totalScoreWeighted += currentScore;
    sectionTotals.maxPossibleScore += item.score || 1;
    sectionTotals.maxPossibleScoreWeighted += maxScore;
    sectionTotals.totalFiles += fileCount;

    // Segment level (aggregate from sections)
    segmentTotals.allFields++;
    if (item.required) segmentTotals.requiredFields++;
    if (isAnswered) segmentTotals.answeredFields++;
    if (item.required && isAnswered) segmentTotals.answeredRequiredFields++;
    segmentTotals.totalScore += isAnswered ? item.score || 1 : 0;
    segmentTotals.totalScoreWeighted += currentScore;
    segmentTotals.maxPossibleScore += item.score || 1;
    segmentTotals.maxPossibleScoreWeighted += maxScore;
    segmentTotals.totalFiles += fileCount;

    // Global level (aggregate from segments)
    globalTotals.allFields++;
    if (item.required) globalTotals.requiredFields++;
    if (isAnswered) globalTotals.answeredFields++;
    if (item.required && isAnswered) globalTotals.answeredRequiredFields++;
    globalTotals.totalScore += isAnswered ? item.score || 1 : 0;
    globalTotals.totalScoreWeighted += currentScore;
    globalTotals.maxPossibleScore += item.score || 1;
    globalTotals.maxPossibleScoreWeighted += maxScore;
    globalTotals.totalFiles += fileCount;

    // Track unanswered required fields at all levels
    if (item.required && !isAnswered) {
      const fieldInfo = {
        key,
        label: getLabel(item, window.lang || "en") || key,
        segment,
        section,
      };

      sectionTotals.unansweredRequiredFields.push(fieldInfo);
      segmentTotals.unansweredRequiredFields.push(fieldInfo);
      globalTotals.unansweredRequiredFields.push(fieldInfo);
    }
  });

  // Build the result.segments structure (if needed for other purposes)
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    const segmentEntry = {
      name: segmentKey,
      label: segment.label,
      sections: [],
    };

    Object.entries(segment.sections).forEach(([sectionKey, section]) => {
      segmentEntry.sections.push({
        name: sectionKey,
        label: section.label,
        totals: section.totals,
      });
    });

    result.segments.push(segmentEntry);
  });

  return { result, summary };
}
function getGroupingData(schema) {
  if (schema.grouping) return schema.grouping;
  if (schema.data_type === "grouping") {
    return {
      name: schema.name,
      label: schema.label,
      type: schema.type,
    };
  }
  return null;
}

function Sum1(summary, data) {
  const lang = window.lang || "en";
  const targetDivId = "rsSum1";
  const targetDiv = document.getElementById(targetDivId);

  if (!targetDiv) {
    console.error(`Target div ${targetDivId} not found`);
    return;
  }

  if (!summary || !summary.segments || !summary.totals) {
    targetDiv.innerHTML = "<div>No summary data available</div>";
    return;
  }

  // Enhanced file counting function
  function countAllFiles(field) {
    if (!field || !field.value) return 0;

    let count = 0;

    if (field.data_type === "attachment" && typeof field.value === "string") {
      count += field.value.split(",").filter((f) => f.trim()).length;
    }

    if (
      (field.data_type === "radioatt" || field.data_type === "radiotextatt") &&
      field.value?.att &&
      typeof field.value.att === "string"
    ) {
      count += field.value.att.split(",").filter((f) => f.trim()).length;
    }

    if (
      field.data_type === "checktextatt" &&
      field.value?.att &&
      typeof field.value.att === "string"
    ) {
      count += field.value.att.split(",").filter((f) => f.trim()).length;
    }

    return count;
  }

  // Enhanced calculation functions
  function calculateCurrentScore(field) {
    if (!isFieldAnswered(field)) return 0;

    if (field.weights) {
      if (field.data_type === "radio" || field.data_type === "radioatt") {
        const radioValue =
          field.data_type === "radioatt" ? field.value?.radio : field.value;
        return field.weights[radioValue] || 0;
      }
      if (field.data_type === "checkbox") {
        return field.value.reduce(
          (sum, idx) => sum + (field.weights[idx] || 0),
          0,
        );
      }
      if (
        field.data_type === "radiotext" ||
        field.data_type === "radiotextatt"
      ) {
        return field.weights[field.value.radio] || 0;
      }
      if (
        field.data_type === "checktext" ||
        field.data_type === "checktextatt"
      ) {
        return field.value.check.reduce(
          (sum, idx) => sum + (field.weights[idx] || 0),
          0,
        );
      }
    }
    return field.score || (isFieldAnswered(field) ? 1 : 0);
  }

  function calculateMaxScore(field) {
    if (field.weights) {
      if (
        ["radio", "radioatt", "radiotext", "radiotextatt"].includes(
          field.data_type,
        )
      ) {
        return Math.max(...field.weights);
      }
      if (["checkbox", "checktext", "checktextatt"].includes(field.data_type)) {
        return field.weights
          .filter((w) => w > 0)
          .reduce((sum, w) => sum + w, 0);
      }
    }
    return field.score || 1;
  }

  function calculateScoreAchieved(field) {
    const currentScore = calculateCurrentScore(field);
    return (field.score || 1) * currentScore;
  }

  function countFiles(field) {
    if (!field || !field.value) return 0;

    // Handle regular attachment fields
    if (field.data_type === "attachment" && typeof field.value === "string") {
      return field.value.split(",").filter((f) => f.trim()).length;
    }

    // Handle radioatt fields - check for value.att string
    if (
      field.data_type === "radioatt" &&
      field.value?.att &&
      typeof field.value.att === "string"
    ) {
      return field.value.att.split(",").filter((f) => f.trim()).length;
    }

    // Handle radiotextatt fields - check for value.att string
    if (
      field.data_type === "radiotextatt" &&
      field.value?.att &&
      typeof field.value.att === "string"
    ) {
      return field.value.att.split(",").filter((f) => f.trim()).length;
    }

    return 0;
  }

  function getFileValue(field) {
    if (!field || !field.value) return "";

    if (field.data_type === "attachment") {
      return typeof field.value === "string" ? field.value : "";
    }

    if (
      (field.data_type === "radioatt" ||
        field.data_type === "radiotextatt" ||
        field.data_type === "checktextatt") &&
      field.value?.att
    ) {
      return field.value.att;
    }

    return "";
  }
  function getFieldValueDisplay(field) {
    if (!field.value) return "";

    if (typeof field.value === "object") {
      try {
        return JSON.stringify(field.value);
      } catch (e) {
        return "[object]";
      }
    }
    return String(field.value);
  }
  let html = `
    <style>
      .sum1-table {
        width: 100%;
        border-collapse: collapse;
        font-family: Arial, sans-serif;
        font-size: 0.9em;
      }
      .sum1-table th, .sum1-table td {
        padding: 6px 8px;
        border: 1px solid #ddd;
        text-align: left;
      }
      .sum1-table th {
        background-color: #f2f2f2;
        font-weight: bold;
      }
      .sum1-segment {
        background-color: #e6f7ff;
      }
      .sum1-section {
        background-color: #f0f0f0;
      }
      .sum1-field {
        background-color: #f9f9f9;
      }
      .sum1-answered {
        color: #4CAF50;
      }
      .sum1-unanswered {
        color: #f44336;
      }
      .sum1-status {
        display: inline-block;
        width: 12px;
        height: 12px;
        border-radius: 50%;
        margin-right: 5px;
      }
      .sum1-status-answered {
        background-color: #4CAF50;
      }
      .sum1-status-unanswered {
        background-color: #f44336;
      }
      .sum1-file-value {
        max-width: 200px;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
        display: inline-block;
      }
    </style>
    <table class="sum1-table">
      <thead>
        <tr>
          <th>Name</th>
          <th>Type</th>
          <th>Status</th>
          <th>Score</th>
          <th>Achieved</th>
          <th>Score Achieved</th>
          <th>Max</th>
          <th>Files</th>
          <th>File Value</th>
          <th>Value</th>
        </tr>
      </thead>
      <tbody>
  `;
  const global = summary.totals;
  // Calculate global totals
  let globalFileCount = 0;
  let globalScoreAchieved = 0;
  for (const key in data) {
    if (key === "grouping" || data[key].type === "section") continue;
    globalFileCount += countAllFiles(data[key]);
    globalScoreAchieved += calculateScoreAchieved(data[key]);
  }

  // Global totals row

  html += `
    <tr class="sum1-global">
      <td colspan="3"><strong>GRAND  TOTAL</strong></td>
      <td>${global.answeredFields}/${global.allFields}</td>

      <td>${global.totalScore}</td>
      <td>${global.totalScoreWeighted}</td>
      <td>${globalScoreAchieved.toFixed(1)}</td>
      <td>${global.maxPossibleScoreWeighted}</td>
      <td>${globalFileCount}</td>
      <td colspan="2"></td>
    </tr>
  `;

  // Process all segments and sections
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    // Calculate segment totals
    let segmentFileCount = 0;
    let segmentScoreAchieved = 0;
    let segmentAnsweredFields = 0;
    let segmentAllFields = 0;
    // Segment header
    html += `
      <tr class="sum1-segment">
        <td colspan="2"><strong>${segment.label || segmentKey}</strong></td>
        <td>${segment.totals.answeredFields}/${segment.totals.allFields}</td>
        <td>${segment.totals.totalScore}</td>
        <td>${segment.totals.totalScoreWeighted}</td>
        <td>${segment.totals.totalScoreWeighted}</td>
        <td>${segment.totals.maxPossibleScoreWeighted}</td>
        <td>${segment.totals.totalFiles}</td>
        <td colspan="2"></td>
      </tr>
    `;

    Object.entries(segment.sections).forEach(([sectionKey, section]) => {
      // Calculate section totals
      let sectionFileCount = 0;
      let sectionScoreAchieved = 0;
      for (const key in data) {
        if (
          data[key].section === sectionKey &&
          data[key].segment?.includes(segmentKey)
        ) {
          sectionFileCount += countAllFiles(data[key]);
          sectionScoreAchieved += calculateScoreAchieved(data[key]);
        }
      }

      // Section header
      html += `
        <tr class="sum1-section">
          <td colspan="2">${section.label || sectionKey}</td>
          <td>${section.totals.answeredFields}/${section.totals.allFields}</td>
          <td>${section.totals.totalScore}</td>
          <td>${section.totals.totalScoreWeighted}</td>
          <td>${sectionScoreAchieved.toFixed(1)}</td>
          <td>${section.totals.maxPossibleScoreWeighted}</td>
          <td>${sectionFileCount}</td>
          <td colspan="2"></td>
        </tr>
      `;

      // Show ALL fields in this section
      for (const key in data) {
        if (
          data[key].section === sectionKey &&
          data[key].segment?.includes(segmentKey)
        ) {
          const field = data[key];
          if (field.type === "section") continue;

          const isAnswered = isFieldAnswered(field);
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const scoreAchieved = calculateScoreAchieved(field);
          const fileCount = countAllFiles(field);
          const fileValue = getFileValue(field);
          const fieldValue = getFieldValueDisplay(field);

          html += `
            <tr class="sum1-field">
              <td>
                <span class="sum1-status ${isAnswered ? "sum1-status-answered" : "sum1-status-unanswered"}"></span>
                ${field.label?.[lang] || key}
              </td>
              <td>${field.data_type || field.type || "field"}</td>
              <td class="${isAnswered ? "sum1-answered" : "sum1-unanswered"}">
                ${isAnswered ? "Answered" : "Unanswered"}
              </td>
              <td>${field.score || 1}</td>
              <td>${currentScore}</td>
              <td>${scoreAchieved.toFixed(1)}</td>
              <td>${maxScore}</td>
              <td>${fileCount}</td>
              <td class="sum1-file-value" title="${fileValue}">${fileValue}</td>
              <td class="sum1-file-value" title="${fieldValue}">${fieldValue}</td>
            </tr>
          `;
        }
      }
    });
  });

  html += `
      </tbody>
    </table>
    <div style="margin-top: 10px; font-size: 0.8em;">
      <span class="sum1-status sum1-status-answered"></span> Answered field |
      <span class="sum1-status sum1-status-unanswered"></span> Unanswered field
    </div>
  `;

  targetDiv.innerHTML = html;

  // Helper function to get file values
  function getFileValue(field) {
    if (!field || !field.value) return "";

    if (field.data_type === "attachment") {
      return typeof field.value === "string" ? field.value : "";
    }

    if (
      (field.data_type === "radioatt" || field.data_type === "radiotextatt") &&
      field.value?.att
    ) {
      return field.value.att;
    }

    if (field.data_type === "checktextatt" && field.value?.att) {
      return field.value.att;
    }

    return "";
  }
}
function scrollTo(key, formId = "gform") {
  const id = `${encodeURIComponent(key)}`;
  const el = document.getElementById(id);
  if (el) {
    el.scrollIntoView({ behavior: "smooth", block: "start" });
    el.style.border = "3px solid #ffa500";
    setTimeout(() => (el.style.border = ""), 2000);
  } else {
    console.warn("Scroll target not found:", id);
  }
}

function scrollToField(key, formId = "gform") {
  const id = `${formId}-field-${encodeURIComponent(key)}`;
  const el = document.getElementById(id);
  if (el) {
    el.scrollIntoView({ behavior: "smooth", block: "center" });
    el.style.border = "3px solid #ffa500";
    setTimeout(() => (el.style.border = ""), 2000);
  } else {
    console.warn("Scroll target not found:", id);
  }
}

//new function
function updateJumperHelperWithUnanswered(
  summary,
  data,
  targetDivId = "jumperhelper",
) {
  const lang = window.lang || "en";
  if (!summary || !summary.segments || !summary.totals) {
    console.error("Invalid summary object:", summary);
    return "";
  }

  const panelStack = window.panelStack || [];
  const formId =
    panelStack.length > 0 ? panelStack[panelStack.length - 1] : "gform";
  //const activeFormId =    formId === "gen" ? "gform" : formId === "profile" ? "pro" : formId;
  
  if (formId === "gen") window.activeFormId = "gform";
  else if (formId === "profile") window.activeFormId = "pro";

  // Enhanced getLabel function with better fallbacks
  const getLabel = (item) => {
    if (!item) return "";
    if (typeof item.label === "string") return item.label;
    if (item.label && (item.label[lang] || item.label.en)) {
      return item.label[lang] || item.label.en;
    }
    if (item.labels && (item.labels[lang] || item.labels.en)) {
      return item.labels[lang] || item.labels.en;
    }
    return item.key || "";
  };

  // Helper functions
  const generateId = (prefix) =>
    `${prefix}-${Math.random().toString(36).substr(2, 9)}`;
  //const generateScrollLink = (key) =>    `javascript:scrollToField('${key}', '${activeFormId}')`;
const generateScrollLink = (key, type, formId) => {
    const id = type === "segment" 
      ? `${formId}-segment-${encodeFieldKey(key)}`
      : type === "section" 
        ? `${formId}-section-${encodeFieldKey(key)}`
        : `${formId}-field-${encodeFieldKey(key)}`;
    return `javascript:scrollToField('${id}', '${formId}')`;
  };
  
  // Generate all toggle IDs in advance
  const globalUnansweredToggleId = generateId("global-unanswered-toggle");
  const collapseAllId = generateId("collapse-all");
  const sectionToggleIds = {};
  const segmentToggleIds = {};
  const progressCircleId = generateId("progress-circle");

  // Pre-process to collect all toggle IDs
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    segmentToggleIds[segmentKey] = generateId(`segment-${segmentKey}-toggle`);
    Object.entries(segment.sections || {}).forEach(([sectionKey]) => {
      sectionToggleIds[sectionKey] = generateId(`section-${sectionKey}-toggle`);
    });
  });

  let html = `<div style='font-family: Arial, sans-serif;'>`;

  // Enhanced Collapse/Expand All controls
  html += `
    <div style='margin-bottom: 10px; display: flex; gap: 10px;'>
      <button id="${collapseAllId}"
              onclick="
                const isCollapsed = this.textContent.includes('${lang === "th" ? "แสดง" : "Expand"}');
                document.querySelectorAll('[id^=segment-][id$=-toggle], [id^=section-][id$=-toggle]').forEach(el => {
                  el.style.display = isCollapsed ? 'block' : 'none';
                });
                document.querySelectorAll('[id^=segment-][id$=-toggle] + div, [id^=section-][id$=-toggle] + div').forEach(el => {
                  el.style.display = isCollapsed ? 'block' : 'none';
                });
                this.textContent = isCollapsed ?
                  '▼ ${lang === "th" ? "ซ่อนทั้งหมด" : "Collapse All"}' :
                  '▶ ${lang === "th" ? "แสดงทั้งหมด" : "Expand All"}';
                document.getElementById('${globalUnansweredToggleId}').style.display = isCollapsed ? 'block' : 'none';
              "
              style="
                padding: 5px 10px;
                background: #4a90e2;
                color: white;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                flex: 1;
              ">
        ▼ ${lang === "th" ? "ซ่อนทั้งหมด" : "Collapse All"}
      </button>

      <button onclick="
                document.getElementById('${globalUnansweredToggleId}').style.display =
                  document.getElementById('${globalUnansweredToggleId}').style.display === 'none' ? 'block' : 'none';
              "
              style="
                padding: 5px 10px;
                background: #f0ad4e;
                color: white;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                flex: 1;
              ">
        ${lang === "th" ? "สลับฟิลด์ที่ยังไม่ได้ตอบ" : "Toggle Unanswered"}
      </button>
    </div>
  `;

  // Grand Summary - unchanged from previous version
  const globalTotals = summary.totals;
  const fieldsPercent =
    globalTotals.allFields > 0
      ? Math.round((globalTotals.answeredFields / globalTotals.allFields) * 100)
      : 100;
  const requiredPercent =
    globalTotals.requiredFields > 0
      ? Math.round(
          (globalTotals.answeredRequiredFields / globalTotals.requiredFields) *
            100,
        )
      : 100;
  // Calculate progress values
  //globalTotals = summary.totals;
  const answeredRequired = globalTotals.answeredRequiredFields || 0;
  const totalRequired = globalTotals.requiredFields || 1; // Avoid division by zero
  const progressPercent = Math.round((answeredRequired / totalRequired) * 100);
  const progressColor =
    progressPercent === 100
      ? "#4CAF50"
      : progressPercent >= 50
        ? "#FFC107"
        : "#F44336";
  html += `
     <div style="display: flex; flex-direction: column; align-items: center; margin-bottom: 20px;">
       <svg width="120" height="120" id="${progressCircleId}-svg">
         <circle cx="60" cy="60" r="50" stroke="#eee" stroke-width="12" fill="none" />
         <circle id="${progressCircleId}" cx="60" cy="60" r="50"
                 stroke="${progressColor}" stroke-width="12" fill="none" stroke-linecap="round"
                 transform="rotate(-90 60 60)"
                 stroke-dasharray="314" stroke-dashoffset="${314 - (314 * progressPercent) / 100}" />
       </svg>
       <div id="${progressCircleId}-label" style="width: 160px; text-align: center; margin-top: 15px; font-size: 1.2em; font-weight: bold;">
         <div>${getLabel({ label: { en: "Answered", th: "ตอบแล้ว" } })}: <span id="${progressCircleId}-answered">${answeredRequired}</span>/<span id="${progressCircleId}-total">${totalRequired}</span></div>
         <div id="${progressCircleId}-percent">${progressPercent}%</div>
       </div>
     </div>
   `;
  html += `
    <div style='border: 2px solid #4a90e2; padding: 10px; margin-bottom: 20px; border-radius: 5px;'>
      <h4 style='margin: 0 0 5px 0;'>${getLabel({ label: { en: "Grand Summary", th: "สรุปทั้งหมด" } })}</h4>
      <div style='margin-bottom: 5px;'>
        ${getLabel({ label: { en: "Total Fields", th: "ฟิลด์ทั้งหมด" } })}: ${globalTotals.answeredFields}/${globalTotals.allFields}
        <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
          <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
        </div>
      </div>
      <div style='margin-bottom: 5px;'>
        ${getLabel({ label: { en: "Required", th: "จำเป็นต้องตอบ" } })}: ${globalTotals.answeredRequiredFields}/${globalTotals.requiredFields}
        <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
          <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
        </div>
      </div>
      <div style='margin-bottom: 5px;'>
        ${getLabel({ label: { en: "Total Files", th: "ไฟล์ทั้งหมด" } })}: <b>${globalTotals.totalFiles}</b>
      </div>
  `;

  // Global unanswered fields with toggle
  if (globalTotals.unansweredRequiredFields?.length > 0) {
    html += `
      <div style='margin-top: 10px; color: orange;'>
        <a href="javascript:void(0)"
           onclick="document.getElementById('${globalUnansweredToggleId}').style.display =
                   document.getElementById('${globalUnansweredToggleId}').style.display === 'none' ? 'block' : 'none';
                   this.innerHTML = document.getElementById('${globalUnansweredToggleId}').style.display === 'none' ?
                   '${getLabel({ label: { en: "Unanswered Required Fields: ▼", th: "ฟิลด์ที่จำเป็นยังไม่ได้ตอบ: ▼" } })}' :
                   '${getLabel({ label: { en: "Unanswered Required Fields: ▶", th: "ฟิลด์ที่จำเป็นยังไม่ได้ตอบ: ▶" } })}';"
           style="text-decoration: none; color: orange;">
          ${getLabel({ label: { en: "Unanswered Required Fields: ▼", th: "ฟิลด์ที่จำเป็นยังไม่ได้ตอบ: ▼" } })}
        </a>
        <div id="${globalUnansweredToggleId}">
          <ul style='margin: 5px 0; padding-left: 20px;'>
            ${globalTotals.unansweredRequiredFields
              .map((field) => {
                const fieldData = data[field.key] || {};
                return `
              <li style='max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
              <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${activeFormId}')"
                       style='text-decoration: none; color: red;'
                       title="${getLabel(fieldData)}">
                      ${getLabel(fieldData)} (${field.section})
                    </a>
              </li>
            `;
              })
              .join("")}
          </ul>
        </div>
      </div>
    `;
  }

  html += `</div>`;

  // Segments with their sections - now with toggle support for segments
  html += "<div style='display: flex; flex-wrap: wrap; gap: 10px;'>";

  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    const segmentTotals = segment.totals || {};
    const segmentData = data[segmentKey] || {};
    const segmentToggleId = segmentToggleIds[segmentKey];
    const fieldsPercent =
      segmentTotals.allFields > 0
        ? Math.round(
            (segmentTotals.answeredFields / segmentTotals.allFields) * 100,
          )
        : 100;
    const requiredPercent =
      segmentTotals.requiredFields > 0
        ? Math.round(
            (segmentTotals.answeredRequiredFields /
              segmentTotals.requiredFields) *
              100,
          )
        : 100;

    const segmentLabel = getLabel(segmentData) || segmentKey;

    html += `
      <div style='border: 1px solid #ccc; padding: 10px; border-radius: 5px; min-width: 200px;'>
        <h4 style='margin: 0 0 5px 0; cursor: pointer;'
            onclick="
              const segToggle = document.getElementById('${segmentToggleId}');
              segToggle.style.display = segToggle.style.display === 'none' ? 'block' : 'none';
              this.querySelector('span').textContent = segToggle.style.display === 'none' ? '▶' : '▼';
            ">
          <span>▼</span> ${segmentLabel}
        </h4>
        <div id="${segmentToggleId}">
          <div style='margin-bottom: 5px;'>
            ${getLabel({ label: { en: "Fields", th: "ฟิลด์" } })}: ${segmentTotals.answeredFields}/${segmentTotals.allFields}
            <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
              <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
            </div>
          </div>
          <div style='margin-bottom: 5px;'>
            ${getLabel({ label: { en: "Required", th: "จำเป็นต้องตอบ" } })}: ${segmentTotals.answeredRequiredFields}/${segmentTotals.requiredFields}
            <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
              <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
            </div>
          </div>
    `;

    // Group unanswered fields by section
    const unansweredBySection = {};
    if (segmentTotals.unansweredRequiredFields?.length > 0) {
      segmentTotals.unansweredRequiredFields.forEach((field) => {
        if (!unansweredBySection[field.section]) {
          unansweredBySection[field.section] = [];
        }
        unansweredBySection[field.section].push(field);
      });
    }

    // Sections within this segment
    html += `<div style='margin-top: 10px;'>`;

    Object.entries(segment.sections || {}).forEach(([sectionKey, section]) => {
      const sectionData = data[sectionKey] || {};
      const sectionTotals = section.totals || {};
      const sectionToggleId = sectionToggleIds[sectionKey];
      const hasUnanswered = unansweredBySection[sectionKey]?.length > 0;

      const sectionLabel = getLabel(sectionData) || sectionKey;

      html += `
        <div style='margin-bottom: 8px;'>
          <div style='display: flex; justify-content: space-between; align-items: center; cursor: pointer;'
               onclick="
                 const secToggle = document.getElementById('${sectionToggleId}');
                 if (secToggle) {
                   secToggle.style.display = secToggle.style.display === 'none' ? 'block' : 'none';
                   this.querySelector('span').textContent = secToggle.style.display === 'none' ? '▶' : '▼';
                 }
               ">
            <div style='flex-grow: 1;'>
              ${sectionLabel} (${sectionTotals.answeredFields}/${sectionTotals.allFields})
            </div>
            ${hasUnanswered ? `<span>▼</span>` : ""}
          </div>
          ${
            hasUnanswered
              ? `
            <div id="${sectionToggleId}" style="display: block; margin-top: 4px;">
              <ul style='margin: 3px 0; padding-left: 15px; font-size: 0.9em;'>
                ${unansweredBySection[sectionKey]
                  .map((field) => {
                    const fieldData = data[field.key] || {};
                    return `
                  <li style='max-width: 180px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
                    <a href="${generateScrollLink(field.key)}"
                       style='text-decoration: none; color: red;'
                       title="${getLabel(fieldData)}">
                      ${getLabel(fieldData)}
                    </a>
                  </li>
                `;
                  })
                  .join("")}
              </ul>
            </div>
          `
              : ""
          }
        </div>
      `;
    });

    html += `</div></div></div>`;
  });

  html += `</div></div>`;

  // Add CSS styles
  html += `
    <style>
      [id$="-toggle"] {
        transition: all 0.3s ease;
      }
      a[onclick], [onclick] {
        cursor: pointer;
        user-select: none;
      }
      a[onclick]:hover {
        text-decoration: underline !important;
      }
      .section-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
      }
      #${collapseAllId}:hover {
        background: #3a80d2;
      }
      h4[onclick]:hover, [onclick]:hover {
        background-color: #f5f5f5;
      }
    </style>
  `;

  const jumperDiv = document.getElementById(targetDivId);
  if (jumperDiv) {
    jumperDiv.innerHTML = html;
  } else {
    console.error(`Target div ${targetDivId} not found`);
  }

  return html;
}
// Helper function to animate the progress circle
function animateProgressCircle(circleId, answered, total) {
  const circle = document.getElementById(circleId);
  const percentEl = document.getElementById(`${circleId}-percent`);
  const answeredEl = document.getElementById(`${circleId}-answered`);
  const totalEl = document.getElementById(`${circleId}-total`);
  const label = document.getElementById(`${circleId}-label`);

  if (!circle || !percentEl) {
    console.error("Progress elements not found");
    return;
  }

  // Ensure we have valid numbers
  answered = answered || 0;
  total = total || 1; // Avoid division by zero
  const percent = Math.round((answered / total) * 100);
  const color =
    percent === 100 ? "#4CAF50" : percent >= 50 ? "#FFC107" : "#F44336";

  // Update counts immediately
  if (answeredEl) answeredEl.textContent = answered;
  if (totalEl) totalEl.textContent = total;
  if (label) label.style.color = color;

  // Animate progress
  const radius = 50;
  const circumference = 2 * Math.PI * radius;
  const duration = 1000;
  const start = performance.now();

  function animate(time) {
    const elapsed = time - start;
    const progress = Math.min(elapsed / duration, 1);
    const value = percent * progress;
    const offset = circumference - (value / 100) * circumference;
    circle.style.strokeDashoffset = offset;
    circle.style.stroke = color;
    if (percentEl) percentEl.textContent = `${Math.round(value)}%`;
    if (progress < 1) {
      requestAnimationFrame(animate);
    }
  }

  requestAnimationFrame(animate);
}


