// Inject styles
function addStyle(css) {
  const style = document.createElement("style");
  style.textContent = css;
  document.head.appendChild(style);
}

addStyle(`
#triggerIcon {
  cursor: pointer;
  padding: 8px 12px;
  background: #333;
  color: #fff;
  border-radius: 6px;
  display: inline-block;
  margin: 20px;
}
#menuContainer {
  position: absolute;
  background: #fff;
  border: 1px solid #ccc;
  box-shadow: 0 4px 12px rgba(0,0,0,0.1);
  padding: 8px;
  display: none;
  min-width: 200px;
  z-index: 9999;
}
.menu-group {
  margin-bottom: 10px;
}
.menu-group-title {
  font-weight: bold;
  font-size: 14px;
  color: #666;
  margin-bottom: 4px;
}
.menu-item {
  display: flex;
  align-items: center;
  padding: 6px;
  cursor: pointer;
  border-radius: 4px;
  position: relative;
}
.menu-item:hover {
  background: #eee;
}
.menu-icon {
  width: 24px;
  text-align: center;
  margin-right: 8px;
}
.menu-text {
  flex-grow: 1;
}
.menu-badge {
  position: absolute;
  top: 4px;
  right: 6px;
  background: red;
  color: white;
  border-radius: 10px;
  font-size: 11px;
  padding: 1px 6px;
  line-height: 1;
  display: inline-block;
}
`);

// Create menu container
function createDiv() {
  const menu = document.createElement("div");
  menu.id = "menuContainer";
  document.body.appendChild(menu);
  return menu;
}

// Smart position
function positionMenuSmart(triggerEl) {
  const iconRect = triggerEl.getBoundingClientRect();
  const menu = menuContainer;
  const menuWidth = menu.offsetWidth || 180;
  const menuHeight = menu.offsetHeight || 200;
  const zone = getZone(iconRect);
  let top = 0, left = 0;

  switch (zone) {
    case "TL":
    case "TC":
      top = iconRect.bottom + 8;
      left = iconRect.left;
      break;
    case "TR":
      top = iconRect.bottom + 8;
      left = iconRect.right - menuWidth;
      break;
    case "CL":
    case "CR":
    case "CC":
      top = iconRect.bottom + 8;
      left = iconRect.left;
      break;
    case "BL":
    case "BC":
    case "BR":
      top = iconRect.top - menuHeight - 8;
      left = iconRect.left;
      break;
  }

  top = Math.max(0, Math.min(top, window.innerHeight - menuHeight - 10));
  left = Math.max(0, Math.min(left, window.innerWidth - menuWidth - 10));

  menu.style.top = `${top}px`;
  menu.style.left = `${left}px`;
}

function getZone(rect) {
  const xCenter = rect.left + rect.width / 2;
  const yCenter = rect.top + rect.height / 2;
  const W = window.innerWidth;
  const H = window.innerHeight;

  if (yCenter < H / 3) {
    if (xCenter < W / 3) return "TL";
    if (xCenter < (2 * W) / 3) return "TC";
    return "TR";
  } else if (yCenter < (2 * H) / 3) {
    if (xCenter < W / 3) return "CL";
    if (xCenter < (2 * W) / 3) return "CC";
    return "CR";
  } else {
    if (xCenter < W / 3) return "BL";
    if (xCenter < (2 * W) / 3) return "BC";
    return "BR";
  }
}

// Render menu with group titles + badge support
function renderMenu(data, container) {
  container.innerHTML = "";
  data.forEach(group => {
    const groupDiv = document.createElement("div");
    groupDiv.className = "menu-group";

    if (group.group) {
      const title = document.createElement("div");
      title.className = "menu-group-title";
      title.textContent = group.group;
      groupDiv.appendChild(title);
    }

    group.items.forEach(entry => {
      const item = document.createElement("div");
      item.className = "menu-item";
      item.id = `menu-${entry.item}`;

      item.innerHTML = `
        <div class="menu-icon">${entry.icon}</div>
        <div class="menu-text">${entry.text}</div>
      `;

      if (badges[entry.item]) {
        const badge = document.createElement("span");
        badge.className = "menu-badge";
        badge.textContent = badges[entry.item];
        item.appendChild(badge);
      }

      item.onclick = () => {
        entry.onclick();
        closeMenu();
      };
      groupDiv.appendChild(item);
    });

    container.appendChild(groupDiv);
  });
}

// Auto badge data
const badges = {
  togglebar: 3,
  profile: 1,
  settings: 0,
  help: 5,
  logout: 0
};

// Sample data
const menuData = [
  {
    group: "View Options",
    items: [
      {
        item: "togglebar",
        icon: "🟦",
        text: "Assessment",
        onclick: () => alert("Assessment clicked"),
      },
      {
        item: "togglelist",
        icon: "📋",
        text: "Profile",
        onclick: () => alert("Switched to list"),
      },
    ],
  },
  {
    group: "User Actions",
    items: [
      {
        item: "profile",
        icon: "👤",
        text: "User",
        onclick: () => alert("Profile clicked"),
      },
      {
        item: "settings",
        icon: "⚙️",
        text: "Settings",
        onclick: () => alert("Settings clicked"),
      },
    ],
  },
  {
    group: "Support",
    items: [
      {
        item: "help",
        icon: "❓",
        text: "Help",
        onclick: () => alert("Help clicked"),
      },
      {
        item: "logout",
        icon: "🚪",
        text: "Logout",
        onclick: () => alert("Logged out"),
      },
    ],
  },
];

// Utils
function closeMenu() {
  menuContainer.style.display = "none";
}
// Update or remove badge dynamically
function updateBadge(itemId, count) {
  badges[itemId] = count;

  const item = document.getElementById(`menu-${itemId}`);
  if (!item) return;

  let badge = item.querySelector(".menu-badge");

  if (count > 0) {
    if (!badge) {
      badge = document.createElement("span");
      badge.className = "menu-badge";
      item.appendChild(badge);
    }
    badge.textContent = count;
    badge.style.display = "inline-block";
  } else {
    if (badge) {
      badge.remove();
    }
  }
}
function randoma() {
  const randomCount = Math.floor(Math.random() * 100);

  // Update badge object and the DOM
  updateBadge('profile', randomCount);

  // Optional delayed update
  setTimeout(() => updateBadge("profile", 2), 2000);
}


// Make any div draggable by its top edge
function dragOn(divid) {
  const el = document.getElementById(divid);
  let offsetX = 0, offsetY = 0, dragging = false;

  el.style.position = "absolute";
  el.style.cursor = "move";

  el.addEventListener("mousedown", (e) => {
    dragging = true;
    offsetX = e.clientX - el.getBoundingClientRect().left;
    offsetY = e.clientY - el.getBoundingClientRect().top;
    document.body.style.userSelect = "none";
  });

  document.addEventListener("mousemove", (e) => {
    if (!dragging) return;
    el.style.left = (e.clientX - offsetX) + "px";
    el.style.top = (e.clientY - offsetY) + "px";
  });

  document.addEventListener("mouseup", () => {
    dragging = false;
    document.body.style.userSelect = "auto";
  });
}

// Init menu system
const menuContainer = createDiv();

document.getElementById("triggerIcon").addEventListener("click", (e) => {
  if (menuContainer.style.display === "block") {
    closeMenu();
  } else {
    renderMenu(menuData, menuContainer);
    positionMenuSmart(e.currentTarget);
    menuContainer.style.display = "block";
  }
});

// Click outside to close
document.addEventListener("click", (e) => {
  if (!menuContainer.contains(e.target) && e.target.id !== "triggerIcon") {
    //closeMenu();
  }
});
