const testItemsArray = [
    {
        id: "1",
        name: "Test Item 1",
        flag: "active",
        timestamp: "2025-08-06T12:00:00Z",
        board: JSON.stringify({ user: "testUser1", details: "Sample board data" }),
        keeper: "Keeper1"
    },
    {
        id: "2",
        name: "Test Item 2",
        flag: "inactive",
        timestamp: "2025-08-06T12:01:00Z",
        board: JSON.stringify({ user: "testUser2", details: "Another board data" }),
        keeper: "Keeper2"
    }
];

let testItems = [
    {
        id: "1",
        name: "Test Item 1",
        flag: "active",
        timestamp: "2025-08-06T12:00:00Z",
        board: JSON.stringify({ user: "testUser1", details: "Sample board data" }),
        keeper: "Keeper1"
    },
    {
        id: "2",
        name: "Test Item 2",
        flag: "inactive",
        timestamp: "2025-08-06T12:01:00Z",
        board: JSON.stringify({ user: "testUser2", details: "Another board data" }),
        keeper: "Keeper2"
    }
];

function consoleRawItems() {
    console.log("Raw testItemsArray:", testItemsArray);
    console.log("Raw testItems:", testItems);
}

function testConsoleRawItems() {
    console.log("Testing consoleRawItems...");
    consoleRawItems();
    console.log("Test completed for consoleRawItems.");
}

function consoleUIRawItems(divId) {
    const div = document.getElementById(divId);
    if (!div) {
        console.error(`Div with ID "${divId}" not found`);
        return;
    }

    const html = `
        <style>
            .raw-data {
                background: #f5f5f5;
                padding: 12px;
                border-radius: 6px;
                margin-bottom: 10px;
                font-size: 14px;
                white-space: pre-wrap;
                font-family: monospace;
                max-height: 300px;
                overflow-y: auto;
            }
        </style>
        <div class="raw-data">
            <strong>testItemsArray:</strong>
            <pre>${JSON.stringify(testItemsArray, null, 2)}</pre>
            <strong>testItems:</strong>
            <pre>${JSON.stringify(testItems, null, 2)}</pre>
        </div>
    `;

    div.innerHTML = html;
}

function testConsoleUIRawItems() {
    try {
        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        const divId = "testConsoleUIRawItemsContent";
        const html = `<div id="${divId}"></div>`;

        createPanel("testConsoleUIRawItemsPanel", "CC", "ctc", 600, 400, {
            bodyContent: html,
            noFooter: true,
            title: "Test Console UI Raw Items"
        });

        setTimeout(() => {
            console.log("Testing consoleUIRawItems...");
            consoleUIRawItems(divId);
            console.log("Test completed for consoleUIRawItems.");
        }, 100);
    } catch (err) {
        console.error("Error testing consoleUIRawItems:", err);
        createPanel("testConsoleUIRawItemsPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Test Console UI Raw Items"
        });
    }
}

function renderRawItems(divId) {
    const div = document.getElementById(divId);
    if (!div) {
        console.error(`Div with ID "${divId}" not found`);
        return;
    }

    const items = testItemsArray.map(item => {
        let boardUser = "N/A";
        let boardRaw = "N/A";
        if (item.board && typeof item.board === 'string') {
            try {
                const boardObj = JSON.parse(item.board);
                boardUser = boardObj.user || "N/A";
                boardRaw = JSON.stringify(boardObj, null, 2);
            } catch {
                boardRaw = "Invalid JSON";
            }
        }
        return { ...item, boardUser, boardRaw };
    });

    const html = `
        <style>
            .item-entry {
                background: #fff;
                padding: 12px;
                border-radius: 6px;
                margin-bottom: 10px;
                border: 1px solid #ccc;
                font-size: 14px;
                cursor: pointer;
                transition: background 0.2s;
            }
            .item-entry:hover {
                background: #f8f8f8;
            }
            .item-entry pre {
                background: #f5f5f5;
                padding: 6px;
                border-radius: 4px;
                font-size: 12px;
                overflow-x: auto;
                margin-top: 4px;
                max-height: 200px;
                overflow-y: auto;
            }
            .item-meta {
                font-size: 12px;
                color: #666;
                margin-bottom: 4px;
            }
            .error {
                color: red;
                text-align: center;
                margin: 20px;
            }
        </style>
        <div id="testItemsList">
            ${items.length ? items.map(item => `
                <div class="item-entry" data-item-id="${item.id}">
                    <div><strong>ID:</strong> ${item.id}</div>
                    <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                    <div class="item-meta">Flag: ${item.flag} | Timestamp: ${item.timestamp}</div>
                    <div><strong>Board:</strong></div>
                    <pre>${item.boardRaw}</pre>
                    <div><strong>Board User:</strong> ${item.boardUser}</div>
                    <div><strong>Keeper:</strong> ${item.keeper || 'N/A'}</div>
                </div>
            `).join("") : '<div class="error">No items found</div>'}
        </div>
    `;

    div.innerHTML = html;
}

function testRenderRawItems() {
    try {
        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        const divId = "testRenderRawItemsContent";
        const html = `<div id="${divId}"></div>`;

        createPanel("testRenderRawItemsPanel", "CC", "ctc", 600, 600, {
            bodyContent: html,
            noFooter: true,
            title: "Test Render Raw Items"
        });

        setTimeout(() => {
            console.log("Testing renderRawItems...");
            renderRawItems(divId);
            console.log("Test completed for renderRawItems.");
        }, 100);
    } catch (err) {
        console.error("Error testing renderRawItems:", err);
        createPanel("testRenderRawItemsPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Test Render Raw Items"
        });
    }
}

function showTestItems() {
    consoleRawItems();
    try {
        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        const divId = "testItemsListPanelContent";
        const rawDivId = "testItemsRawPanelContent";
        const html = `
            <div id="${rawDivId}"></div>
            <div id="${divId}"></div>
        `;

        createPanel("testItemsListPanel", "CC", "ctc", 600, 600, {
            bodyContent: html,
            noFooter: true,
            title: "Test Items List"
        });

        setTimeout(() => {
            console.log("Testing showTestItems...");
            consoleUIRawItems(rawDivId);
            renderRawItems(divId);
            console.log("Test completed for showTestItems.");
        }, 100);
    } catch (err) {
        console.error("Error displaying test items:", err);
        createPanel("testItemsListPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Test Items List"
        });
    }
}

function testShowTestItems() {
    showTestItems();
}

function itemListPanel() {
    try {
        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        const items = testItems.map(item => {
            let boardUser = "N/A";
            let boardRaw = "N/A";
            if (item.board && typeof item.board === 'string') {
                try {
                    const boardObj = JSON.parse(item.board);
                    boardUser = boardObj.user || "N/A";
                    boardRaw = JSON.stringify(boardObj, null, 2);
                } catch {
                    boardRaw = "Invalid JSON";
                }
            }
            return { ...item, boardUser, boardRaw };
        });

        const divId = "itemListPanelContent";
        const html = `
            <style>
                .item-entry {
                    background: #fff;
                    padding: 12px;
                    border-radius: 6px;
                    margin-bottom: 10px;
                    border: 1px solid #ccc;
                    font-size: 14px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                }
                .item-entry:hover {
                    background: #f8f8f8;
                }
                .item-entry pre {
                    background: #f5f5f5;
                    padding: 6px;
                    border-radius: 4px;
                    font-size: 12px;
                    overflow-x: auto;
                    margin-top: 4px;
                    max-height: 200px;
                    overflow-y: auto;
                }
                .item-meta {
                    font-size: 12px;
                    color: #666;
                    margin-bottom: 4px;
                }
                .error {
                    color: red;
                    text-align: center;
                    margin: 20px;
                }
                .item-actions {
                    display: flex;
                    gap: 10px;
                }
                .action-btn {
                    padding: 6px 12px;
                    border: none;
                    border-radius: 4px;
                    cursor: pointer;
                    font-size: 12px;
                }
                .view-btn {
                    background: #4CAF50;
                    color: white;
                }
                .edit-btn {
                    background: #2196F3;
                    color: white;
                }
                .delete-btn {
                    background: #f44336;
                    color: white;
                }
                .demo-btn {
                    background: #FFC107;
                    color: black;
                }
            </style>
            <div id="${divId}">
                ${items.length ? items.map(item => `
                    <div class="item-entry" data-item-id="${item.id}">
                        <div>
                            <div><strong>ID:</strong> ${item.id}</div>
                            <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                            <div class="item-meta">Flag: ${item.flag} | Timestamp: ${item.timestamp}</div>
                            <div><strong>Board:</strong></div>
                            <pre>${item.boardRaw}</pre>
                            <div><strong>Board User:</strong> ${item.boardUser}</div>
                            <div><strong>Keeper:</strong> ${item.keeper || 'N/A'}</div>
                        </div>
                        <div class="item-actions">
                            <button class="action-btn view-btn" onclick="viewItem('${item.id}')">View</button>
                            <button class="action-btn edit-btn" onclick="editItem('${item.id}')">Edit</button>
                            <button class="action-btn delete-btn" onclick="confirmDeleteItem('${item.id}')">Delete</button>
                            <button class="action-btn demo-btn" onclick="callForDemo('${item.id}')">Request Demo</button>
                        </div>
                    </div>
                `).join("") : '<div class="error">No items found</div>'}
            </div>
        `;

        createPanel("itemListPanel", "CC", "ctc", 600, 600, {
            bodyContent: html,
            noFooter: true,
            title: "Item List Panel"
        });
    } catch (err) {
        console.error("Error displaying item list:", err);
        createPanel("itemListPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Item List Panel"
        });
    }
}

function testItemListPanel() {
    console.log("Testing itemListPanel...");
    itemListPanel();
    console.log("Test completed for itemListPanel.");
}

function viewItem(itemId) {
    const item = testItems.find(i => i.id === itemId);
    if (!item) {
        console.error(`Item with ID "${itemId}" not found`);
        return;
    }

    let boardUser = "N/A";
    let boardRaw = "N/A";
    if (item.board && typeof item.board === 'string') {
        try {
            const boardObj = JSON.parse(item.board);
            boardUser = boardObj.user || "N/A";
            boardRaw = JSON.stringify(boardObj, null, 2);
        } catch {
            boardRaw = "Invalid JSON";
        }
    }

    const html = `
        <style>
            .view-content {
                font-size: 14px;
                padding: 12px;
            }
            .view-content pre {
                background: #f5f5f5;
                padding: 6px;
                border-radius: 4px;
                font-size: 12px;
                overflow-x: auto;
                max-height: 200px;
                overflow-y: auto;
            }
            .item-meta {
                font-size: 12px;
                color: #666;
                margin-bottom: 4px;
            }
        </style>
        <div class="view-content">
            <div><strong>ID:</strong> ${item.id}</div>
            <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
            <div class="item-meta">Flag: ${item.flag} | Timestamp: ${item.timestamp}</div>
            <div><strong>Board:</strong></div>
            <pre>${boardRaw}</pre>
            <div><strong>Board User:</strong> ${item.boardUser}</div>
            <div><strong>Keeper:</strong> ${item.keeper || 'N/A'}</div>
        </div>
    `;

    createPanel(`viewPanel_${itemId}`, "CC", "ctc", 400, 400, {
        bodyContent: html,
        noFooter: true,
        title: `View Item ${itemId}`
    });
}

function editItem(itemId) {
    const item = testItems.find(i => i.id === itemId);
    if (!item) {
        console.error(`Item with ID "${itemId}" not found`);
        return;
    }

    let boardUser = "N/A";
    if (item.board && typeof item.board === 'string') {
        try {
            const boardObj = JSON.parse(item.board);
            boardUser = boardObj.user || "N/A";
        } catch {
            boardUser = "Invalid JSON";
        }
    }

    const html = `
        <style>
            .edit-content {
                font-size: 14px;
                padding: 12px;
            }
            .edit-content label {
                display: block;
                margin: 10px 0 5px;
            }
            .edit-content input, .edit-content textarea {
                width: 100%;
                padding: 6px;
                border: 1px solid #ccc;
                border-radius: 4px;
            }
            .edit-content textarea {
                resize: vertical;
                max-height: 200px;
            }
        </style>
        <div class="edit-content">
            <label>ID: ${item.id}</label>
            <label>Name:</label>
            <input type="text" id="editName_${itemId}" value="${item.name || ''}">
            <label>Flag:</label>
            <input type="text" id="editFlag_${itemId}" value="${item.flag || ''}">
            <label>Timestamp:</label>
            <input type="text" id="editTimestamp_${itemId}" value="${item.timestamp || ''}">
            <label>Board User:</label>
            <input type="text" id="editBoardUser_${itemId}" value="${boardUser}">
            <label>Keeper:</label>
            <input type="text" id="editKeeper_${itemId}" value="${item.keeper || ''}">
        </div>
    `;

    createPanel(`editPanel_${itemId}`, "CC", "ctc", 400, 400, {
        bodyContent: html,
        footerButtons: true,
        footerCloseBtn: true,
        title: `Edit Item ${itemId}`,
        footerTitle: "Save or Close"
    });

    addFooterButton(`editPanel_${itemId}`, "Save", {
        onclick: () => {
            const updatedItem = {
                id: itemId,
                name: document.getElementById(`editName_${itemId}`).value,
                flag: document.getElementById(`editFlag_${itemId}`).value,
                timestamp: document.getElementById(`editTimestamp_${itemId}`).value,
                board: JSON.stringify({ user: document.getElementById(`editBoardUser_${itemId}`).value }),
                keeper: document.getElementById(`editKeeper_${itemId}`).value
            };
            testItems = testItems.map(i => i.id === itemId ? updatedItem : i);
            console.log(`Item ${itemId} updated:`, updatedItem);
            closePanel(`editPanel_${itemId}`);
            itemListPanel();
        }
    });
}

function confirmDeleteItem(itemId) {
    const html = `
        <style>
            .confirm-content {
                font-size: 14px;
                padding: 12px;
                text-align: center;
            }
        </style>
        <div class="confirm-content">
            <p>Are you sure you want to delete item ${itemId}?</p>
        </div>
    `;

    createPanel(`confirmDeletePanel_${itemId}`, "CC", "ctc", 300, 200, {
        bodyContent: html,
        footerButtons: true,
        footerCloseBtn: true,
        title: `Confirm Delete Item ${itemId}`,
        footerTitle: "Confirm or Cancel"
    });

    addFooterButton(`confirmDeletePanel_${itemId}`, "Confirm", {
        onclick: () => {
            testItems = testItems.filter(i => i.id !== itemId);
            console.log(`Item ${itemId} deleted`);
            closePanel(`confirmDeletePanel_${itemId}`);
            itemListPanel();
        }
    });
}

function callForDemo(itemId) {
    const item = testItems.find(i => i.id === itemId);
    if (!item) {
        console.error(`Item with ID "${itemId}" not found`);
        return;
    }

    const divId = `demoPanelContent_${itemId}`;
    const html = `
        <style>
            .demo-content {
                font-size: 14px;
                padding: 12px;
            }
            .demo-content label {
                display: block;
                margin: 10px 0 5px;
            }
            .demo-content input, .demo-content textarea {
                width: 100%;
                padding: 6px;
                border: 1px solid #ccc;
                border-radius: 4px;
            }
            .progress-container {
                width: 100%;
                background: #f5f5f5;
                border-radius: 4px;
                margin-top: 10px;
                height: 20px;
                overflow: hidden;
            }
            .progress-bar {
                width: 0%;
                height: 100%;
                background: #4CAF50;
                transition: width 2s ease-in-out;
            }
            .progress-text {
                text-align: center;
                font-size: 12px;
                margin-top: 5px;
            }
        </style>
        <div class="demo-content">
            <p>Request a demo for Item ${item.name || itemId}</p>
            <label>Contact Email:</label>
            <input type="email" id="demoEmail_${itemId}" placeholder="Enter your email">
            <label>Notes:</label>
            <textarea id="demoNotes_${itemId}" placeholder="Additional notes"></textarea>
            <label>HTML Progress Bar:</label>
            <progress id="htmlProgress_${itemId}" max="100" value="0" aria-label="Demo request progress">0%</progress>
            <label>Bootstrap-style Progress Bar:</label>
            <div class="progress-container">
                <div class="progress-bar" id="progressBar_${itemId}"></div>
            </div>
            <div class="progress-text" id="progressText_${itemId}">0% Complete</div>
        </div>
    `;

    createPanel(`demoPanel_${itemId}`, "CC", "ctc", 400, 400, {
        bodyContent: html,
        footerButtons: true,
        footerCloseBtn: true,
        title: `Request Demo for Item ${itemId}`,
        footerTitle: "Submit or Close"
    });

    addFooterButton(`demoPanel_${itemId}`, "Submit", {
        onclick: () => {
            const htmlProgress = document.getElementById(`htmlProgress_${itemId}`);
            const progressBar = document.getElementById(`progressBar_${itemId}`);
            const progressText = document.getElementById(`progressText_${itemId}`);
            let progress = 0;
            const interval = setInterval(() => {
                progress += 10;
                htmlProgress.value = progress;
                progressBar.style.width = `${progress}%`;
                progressText.textContent = `${progress}% Complete`;
                if (progress >= 100) {
                    clearInterval(interval);
                    console.log(`Demo request for Item ${itemId} submitted`, {
                        email: document.getElementById(`demoEmail_${itemId}`).value,
                        notes: document.getElementById(`demoNotes_${itemId}`).value
                    });
                    closePanel(`demoPanel_${itemId}`);
                }
            }, 200);
        }
    });
}

function testCallForDemo() {
    console.log("Testing callForDemo...");
    const itemId = testItems[0]?.id || "1";
    callForDemo(itemId);
    console.log("Test completed for callForDemo.");
}

function demoHTMLProgress() {
    try {
        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        const divId = "htmlProgressDemoContent";
        const html = `
            <style>
                .demo-content {
                    font-size: 14px;
                    padding: 12px;
                }
                .progress-text {
                    text-align: center;
                    font-size: 12px;
                    margin-top: 5px;
                }
            </style>
            <div class="demo-content">
                <label>HTML Progress Bar Demo:</label>
                <progress id="htmlProgressDemo" max="100" value="0" aria-label="HTML progress demo">0%</progress>
                <div class="progress-text" id="htmlProgressText">0% Complete</div>
            </div>
        `;

        createPanel("htmlProgressDemoPanel", "CC", "ctc", 400, 200, {
            bodyContent: html,
            noFooter: true,
            title: "HTML Progress Bar Demo"
        });

        setTimeout(() => {
            console.log("Testing demoHTMLProgress...");
            const progressBar = document.getElementById("htmlProgressDemo");
            const progressText = document.getElementById("htmlProgressText");
            let progress = 0;
            const interval = setInterval(() => {
                progress += 10;
                progressBar.value = progress;
                progressText.textContent = `${progress}% Complete`;
                if (progress >= 100) {
                    clearInterval(interval);
                    console.log("HTML Progress Bar Demo completed.");
                }
            }, 200);
            console.log("Test completed for demoHTMLProgress.");
        }, 100);
    } catch (err) {
        console.error("Error testing demoHTMLProgress:", err);
        createPanel("htmlProgressDemoPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "HTML Progress Bar Demo"
        });
    }
}

function testDemoHTMLProgress() {
    demoHTMLProgress();
}

function demoBootstrapProgress() {
    try {
        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        const divId = "bootstrapProgressDemoContent";
        const html = `
            <style>
                .demo-content {
                    font-size: 14px;
                    padding: 12px;
                }
                .progress-container {
                    width: 100%;
                    background: #f5f5f5;
                    border-radius: 4px;
                    margin-top: 10px;
                    height: 20px;
                    overflow: hidden;
                }
                .progress-bar {
                    width: 0%;
                    height: 100%;
                    background: #4CAF50;
                    transition: width 2s ease-in-out;
                }
                .progress-text {
                    text-align: center;
                    font-size: 12px;
                    margin-top: 5px;
                }
            </style>
            <div class="demo-content">
                <label>Bootstrap-style Progress Bar Demo:</label>
                <div class="progress-container">
                    <div class="progress-bar" id="bootstrapProgressDemo" role="progressbar" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100"></div>
                </div>
                <div class="progress-text" id="bootstrapProgressText">0% Complete</div>
            </div>
        `;

        createPanel("bootstrapProgressDemoPanel", "CC", "ctc", 400, 200, {
            bodyContent: html,
            noFooter: true,
            title: "Bootstrap-style Progress Bar Demo"
        });

        setTimeout(() => {
            console.log("Testing demoBootstrapProgress...");
            const progressBar = document.getElementById("bootstrapProgressDemo");
            const progressText = document.getElementById("bootstrapProgressText");
            let progress = 0;
            const interval = setInterval(() => {
                progress += 10;
                progressBar.style.width = `${progress}%`;
                progressBar.setAttribute("aria-valuenow", progress);
                progressText.textContent = `${progress}% Complete`;
                if (progress >= 100) {
                    clearInterval(interval);
                    console.log("Bootstrap Progress Bar Demo completed.");
                }
            }, 200);
            console.log("Test completed for demoBootstrapProgress.");
        }, 100);
    } catch (err) {
        console.error("Error testing demoBootstrapProgress:", err);
        createPanel("bootstrapProgressDemoPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Bootstrap-style Progress Bar Demo"
        });
    }
}

function testDemoBootstrapProgress() {
    demoBootstrapProgress();
}
function testAddFooterButton() {
    try {
        if (typeof createPanel !== "function" || typeof addFooterButton !== "function") {
            console.error("Missing createPanel or addFooterButton from micro2.js");
            return;
        }

        const panelId = "testAddFooterButtonPanel";
        const html = `
            <style>
                .test-content {
                    font-size: 14px;
                    padding: 12px;
                    text-align: center;
                }
            </style>
            <div class="test-content">
                <p>Test adding a footer button.</p>
            </div>
        `;

        createPanel(panelId, "CC", "ctc", 300, 200, {
            bodyContent: html,
            footerButtons: true,
            footerCloseBtn: true,
            title: "Test Add Footer Button",
            footerTitle: "Test Footer"
        });

        console.log("Testing addFooterButton...");
        addFooterButton(panelId, "Test Button", {
            onclick: () => {
                console.log("Footer button clicked!");
                closePanel(panelId);
            }
        });
        console.log("Test completed for addFooterButton.");
    } catch (err) {
        console.error("Error testing addFooterButton:", err);
        createPanel("testAddFooterButtonPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Test Add Footer Button"
        });
    }
}

