<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Data Management System</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            line-height: 1.6;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
        }
        .control-panel {
            background: #f5f5f5;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        button {
            padding: 8px 12px;
            margin: 5px;
            background: #4CAF50;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        button:hover {
            background: #45a049;
        }
        .danger {
            background: #f44336;
        }
        .danger:hover {
            background: #d32f2f;
        }
        .info {
            background: #2196F3;
        }
        .info:hover {
            background: #0b7dda;
        }
        .warning {
            background: #FFC107;
            color: black;
        }
        .warning:hover {
            background: #FFA000;
        }
        .secondary {
            background: #9E9E9E;
        }
        .secondary:hover {
            background: #757575;
        }
        pre {
            background: #f5f5f5;
            padding: 10px;
            border-radius: 5px;
            overflow-x: auto;
        }
        .api-settings {
            margin-bottom: 20px;
        }
        input {
            padding: 8px;
            width: 300px;
            margin-right: 10px;
        }
        .notification {
            padding: 10px;
            margin: 10px 0;
            border-radius: 4px;
        }
        .warning-notification {
            background: #FFF3CD;
            border-left: 4px solid #FFC107;
        }
        .stats {
            margin-top: 10px;
            font-size: 0.9em;
            color: #666;
        }
        .tabs {
            display: flex;
            margin-bottom: 20px;
        }
        .tab {
            padding: 10px 20px;
            cursor: pointer;
            background: #e0e0e0;
            margin-right: 5px;
            border-radius: 5px 5px 0 0;
        }
        .tab.active {
            background: #f5f5f5;
            font-weight: bold;
        }
        .section {
            margin-bottom: 20px;
            padding: 15px;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
        .section-title {
            margin-top: 0;
            color: #333;
        }
        .radio-group {
            margin: 10px 0;
        }
        .radio-group label {
            margin-right: 15px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Data Management System</h1>
        
        <div class="tabs">
            <div class="tab active" onclick="showTab('data')">Data Operations</div>
            <div class="tab" onclick="showTab('users')">Users</div>
            <div class="tab" onclick="showTab('items')">Items</div>
        </div>
        
        <!-- Data Operations Tab -->
        <div id="dataTab">
            <div class="section">
                <h3 class="section-title">API Configuration</h3>
                <div class="api-settings">
                    <div>
                        <label>Users API URL:</label>
                        <input type="text" id="usersApiUrl" value="https://i.kunok.com/aui.php">
                        <button onclick="updateUsersApiUrl()">Update</button>
                    </div>
                    <div style="margin-top: 10px;">
                        <label>Items API URL:</label>
                        <input type="text" id="itemsApiUrl" value="https://j.kunok.com/api62.php">
                        <button onclick="updateItemsApiUrl()">Update</button>
                    </div>
                    <button class="info" onclick="showCurrentApiSettings()">Show Current API Settings</button>
                </div>
            </div>

            <div class="section">
                <h3 class="section-title">Data Transfer Operations</h3>
                <div class="control-panel">
                    <div class="radio-group">
                        <strong>Copy From:</strong>
                        <label><input type="radio" name="copySource" value="memory" checked> Memory Data</label>
                        <label><input type="radio" name="copySource" value="storage"> Storage Data</label>
                    </div>
                    <div class="radio-group">
                        <strong>Copy To:</strong>
                        <label><input type="radio" name="copyTarget" value="users" checked> Users</label>
                        <label><input type="radio" name="copyTarget" value="items"> Items</label>
                    </div>
                    <button onclick="deepCopyData()">Deep Copy Data</button>
                    
                    <div style="margin-top: 15px;">
                        <strong>Save To Storage:</strong>
                        <button onclick="saveDataToStorage('users')">Save to Users Storage</button>
                        <button onclick="saveDataToStorage('items')">Save to Items Storage</button>
                    </div>
                    
                    <div style="margin-top: 15px;">
                        <strong>Load From Storage:</strong>
                        <button onclick="loadFromStorageToData('users')">Load from Users Storage</button>
                        <button onclick="loadFromStorageToData('items')">Load from Items Storage</button>
                    </div>
                </div>
            </div>

            <div id="dataOutput">
                <h3>Data Output</h3>
                <pre id="dataOutputContent">No data operations performed yet</pre>
                <div id="dataStats" class="stats"></div>
            </div>
        </div>
        
        <!-- Users Tab -->
        <div id="usersTab" style="display:none;">
            <div class="section">
                <h3 class="section-title">Users Data Management</h3>
                <div class="control-panel">
                    <button onclick="loadUsersFromServer()">Load Users From Server</button>
                    <button onclick="loadUsersFromStorage()">Load Users From Storage</button>
                    <button class="danger" onclick="clearUsersStorage()">Clear Users Storage</button>
                    <button class="info" onclick="showUsersDataStructure()">Show Users Structure</button>
                    <button class="info" onclick="showUsersStorageData()">Show Storage Users</button>
                    <button class="info" onclick="showUsersMemoryData()">Show Memory Users</button>
                    <button class="secondary" onclick="clearUsersOutput()">Clear Output</button>
                </div>
                
                <div class="radio-group">
                    <strong>Load Strategy:</strong>
                    <label><input type="radio" name="usersLoadStrategy" value="server" checked> Server</label>
                    <label><input type="radio" name="usersLoadStrategy" value="storage"> Storage</label>
                    <label><input type="radio" name="usersLoadStrategy" value="manual"> Manual</label>
                </div>
            </div>

            <div id="usersOutput">
                <h3>Users Output</h3>
                <pre id="usersOutputContent">No users data loaded yet</pre>
                <div id="usersDataStats" class="stats"></div>
            </div>
        </div>
        
        <!-- Items Tab -->
        <div id="itemsTab" style="display:none;">
            <div class="section">
                <h3 class="section-title">Items Data Management</h3>
                <div class="control-panel">
                    <button onclick="loadItemsFromServer()">Load Items From Server</button>
                    <button onclick="loadItemsFromStorage()">Load Items From Storage</button>
                    <button class="danger" onclick="clearItemsStorage()">Clear Items Storage</button>
                    <button class="info" onclick="showItemsDataStructure()">Show Items Structure</button>
                    <button class="info" onclick="showItemsStorageData()">Show Storage Items</button>
                    <button class="info" onclick="showItemsMemoryData()">Show Memory Items</button>
                    <button class="secondary" onclick="clearItemsOutput()">Clear Output</button>
                </div>
                
                <div class="radio-group">
                    <strong>Load Strategy:</strong>
                    <label><input type="radio" name="itemsLoadStrategy" value="server" checked> Server</label>
                    <label><input type="radio" name="itemsLoadStrategy" value="storage"> Storage</label>
                    <label><input type="radio" name="itemsLoadStrategy" value="manual"> Manual</label>
                </div>
            </div>

            <div id="itemsOutput">
                <h3>Items Output</h3>
                <pre id="itemsOutputContent">No items data loaded yet</pre>
                <div id="itemsDataStats" class="stats"></div>
            </div>
        </div>
    </div>

    <script>
        // Global variables
        let currentData = []; // Generic data holder
        let usersData = [];
        let itemsData = [];
        let usersApiUrl = localStorage.getItem('usersApiUrl') || 'https://i.kunok.com/aui.php';
        let itemsApiUrl = localStorage.getItem('itemsApiUrl') || 'https://j.kunok.com/api62.php';

        // Tab management
        function showTab(tabName) {
            document.querySelectorAll('.tab').forEach(tab => {
                tab.classList.remove('active');
            });
            document.getElementById('dataTab').style.display = 'none';
            document.getElementById('usersTab').style.display = 'none';
            document.getElementById('itemsTab').style.display = 'none';
            
            document.querySelector(`.tab[onclick="showTab('${tabName}')"]`).classList.add('active');
            document.getElementById(`${tabName}Tab`).style.display = 'block';
        }

        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('usersApiUrl').value = usersApiUrl;
            document.getElementById('itemsApiUrl').value = itemsApiUrl;
            
            // Set default load strategies
            const usersStrategy = localStorage.getItem('usersLoadStrategy') || 'manual';
            const itemsStrategy = localStorage.getItem('itemsLoadStrategy') || 'manual';
            
            document.querySelector(`input[name="usersLoadStrategy"][value="${usersStrategy}"]`).checked = true;
            document.querySelector(`input[name="itemsLoadStrategy"][value="${itemsStrategy}"]`).checked = true;
            
            // Initialize based on strategy
            initUsersDataLoading();
            initItemsDataLoading();
        });

        // Helper functions
        function logDataOutput(message) {
            const output = document.getElementById('dataOutputContent');
            output.textContent = typeof message === 'string' ? message : JSON.stringify(message, null, 2);
            console.log('Data:', message);
        }

        function logUsersOutput(message) {
            const output = document.getElementById('usersOutputContent');
            output.textContent = typeof message === 'string' ? message : JSON.stringify(message, null, 2);
            console.log('Users:', message);
        }

        function logItemsOutput(message) {
            const output = document.getElementById('itemsOutputContent');
            output.textContent = typeof message === 'string' ? message : JSON.stringify(message, null, 2);
            console.log('Items:', message);
        }

        function showDataStats(data, source, elementId) {
            const stats = calculateDataSize(data);
            const statsElement = document.getElementById(elementId);
            statsElement.innerHTML = `
                <strong>${source} Data Statistics:</strong><br>
                Items: ${stats.items}<br>
                Size: ${stats.bytes} bytes (${stats.kilobytes} KB, ${stats.megabytes} MB)
            `;
        }

        function calculateDataSize(data) {
            const text = JSON.stringify(data);
            const bytes = new TextEncoder().encode(text).length;
            const kilobytes = bytes / 1024;
            const megabytes = kilobytes / 1024;
            
            return {
                items: Array.isArray(data) ? data.length : 1,
                bytes,
                kilobytes: kilobytes.toFixed(2),
                megabytes: megabytes.toFixed(4)
            };
        }

        // API Configuration
        function updateUsersApiUrl() {
            usersApiUrl = document.getElementById('usersApiUrl').value;
            localStorage.setItem('usersApiUrl', usersApiUrl);
            logDataOutput(`Users API URL updated to: ${usersApiUrl}`);
        }

        function updateItemsApiUrl() {
            itemsApiUrl = document.getElementById('itemsApiUrl').value;
            localStorage.setItem('itemsApiUrl', itemsApiUrl);
            logDataOutput(`Items API URL updated to: ${itemsApiUrl}`);
        }

        function showCurrentApiSettings() {
            logDataOutput({
                usersApiUrl: usersApiUrl,
                itemsApiUrl: itemsApiUrl,
                usersLoadStrategy: document.querySelector('input[name="usersLoadStrategy"]:checked').value,
                itemsLoadStrategy: document.querySelector('input[name="itemsLoadStrategy"]:checked').value
            });
        }

        // Data Transfer Operations
        function deepCopyData() {
            const source = document.querySelector('input[name="copySource"]:checked').value;
            const target = document.querySelector('input[name="copyTarget"]:checked').value;
            
            let sourceData;
            if (source === 'memory') {
                sourceData = currentData;
                if (!sourceData || sourceData.length === 0) {
                    logDataOutput('No data in memory to copy');
                    return;
                }
            } else { // storage
                const storageKey = target === 'users' ? 'usersData' : 'itemsData';
                const storedData = localStorage.getItem(storageKey);
                if (!storedData) {
                    logDataOutput(`No data in ${target} storage to copy`);
                    return;
                }
                sourceData = JSON.parse(storedData);
            }
            
            // Perform deep copy
            const copiedData = JSON.parse(JSON.stringify(sourceData));
            
            if (target === 'users') {
                usersData = copiedData;
                logDataOutput(`Copied ${sourceData.length} items to Users data`);
                logUsersOutput({
                    message: 'Data copied to Users',
                    count: usersData.length,
                    sampleItem: usersData.length > 0 ? usersData[0] : null
                });
                showUsersDataStats(usersData, 'Memory', 'usersDataStats');
            } else {
                itemsData = copiedData;
                logDataOutput(`Copied ${sourceData.length} items to Items data`);
                logItemsOutput({
                    message: 'Data copied to Items',
                    count: itemsData.length,
                    sampleItem: itemsData.length > 0 ? itemsData[0] : null
                });
                showItemsDataStats(itemsData, 'Memory', 'itemsDataStats');
            }
        }

        function saveDataToStorage(target) {
            const dataToSave = currentData;
            if (!dataToSave || dataToSave.length === 0) {
                logDataOutput('No data in memory to save');
                return;
            }
            
            const storageKey = target === 'users' ? 'usersData' : 'itemsData';
            localStorage.setItem(storageKey, JSON.stringify(dataToSave));
            logDataOutput(`Saved ${dataToSave.length} items to ${target} storage`);
            showDataStats(dataToSave, `${target} Storage`, 'dataStats');
        }

        function loadFromStorageToData(source) {
            const storageKey = source === 'users' ? 'usersData' : 'itemsData';
            const storedData = localStorage.getItem(storageKey);
            
            if (!storedData) {
                logDataOutput(`No data found in ${source} storage`);
                return;
            }
            
            currentData = JSON.parse(storedData);
            logDataOutput({
                message: `Loaded data from ${source} storage`,
                count: currentData.length,
                sampleItem: currentData.length > 0 ? currentData[0] : null
            });
            showDataStats(currentData, 'Memory', 'dataStats');
        }

        // Users Management (similar to previous implementation)
        function initUsersDataLoading() {
            const strategy = document.querySelector('input[name="usersLoadStrategy"]:checked').value;
            
            if (strategy === 'storage') {
                if (!loadUsersFromStorage()) {
                    logUsersOutput('No users data in storage. Please load from server manually.');
                }
            } 
            else if (strategy === 'server') {
                const hasLocalData = localStorage.getItem('usersData') !== null;
                if (!hasLocalData) {
                    logUsersOutput('No local users data found. Click "Load Users From Server" to fetch data.');
                } else {
                    loadUsersFromServer();
                }
            }
        }

        async function loadUsersFromServer() {
            try {
                const fullUrl = `${usersApiUrl}?action=list`;
                logUsersOutput(`Loading users data from server: ${fullUrl}`);
                
                const response = await fetch(fullUrl);
                if (!response.ok) throw new Error(`Server error: ${response.status}`);
                
                const result = await response.json();
                if (result.data && Array.isArray(result.data)) {
                    usersData = result.data;
                    logUsersOutput({
                        message: 'Users data loaded from server',
                        count: usersData.length,
                        sampleUser: usersData.length > 0 ? usersData[0] : null
                    });
                    showUsersDataStats(usersData, 'Server', 'usersDataStats');
                    saveUsersToStorage();
                } else {
                    throw new Error('Invalid users data format from server');
                }
            } catch (error) {
                logUsersOutput(`Error loading users from server: ${error.message}`);
            }
        }

        function loadUsersFromStorage() {
            const storedData = localStorage.getItem('usersData');
            if (storedData) {
                usersData = JSON.parse(storedData);
                logUsersOutput({
                    message: 'Users data loaded from storage',
                    count: usersData.length,
                    sampleUser: usersData.length > 0 ? usersData[0] : null
                });
                showUsersDataStats(usersData, 'Storage', 'usersDataStats');
                return true;
            }
            logUsersOutput('No users data found in storage');
            return false;
        }

        function saveUsersToStorage() {
            localStorage.setItem('usersData', JSON.stringify(usersData));
            logUsersOutput(`Users data saved to storage (${usersData.length} users)`);
            showUsersDataStats(usersData, 'Storage', 'usersDataStats');
        }

        function clearUsersStorage() {
            localStorage.removeItem('usersData');
            usersData = [];
            logUsersOutput('Users storage and memory data cleared');
            document.getElementById('usersDataStats').innerHTML = '';
        }

        function showUsersDataStructure() {
            if (usersData.length === 0) {
                logUsersOutput('No users data available to show structure');
                return;
            }
            
            const sampleUser = usersData[0];
            const structure = {};
            for (const key in sampleUser) {
                structure[key] = typeof sampleUser[key];
            }
            
            logUsersOutput({
                message: 'Users Data Structure (first user)',
                structure: structure,
                sample: sampleUser
            });
            showUsersDataStats(sampleUser, 'User Structure', 'usersDataStats');
        }

        function showUsersStorageData() {
            const storedData = localStorage.getItem('usersData');
            if (storedData) {
                const data = JSON.parse(storedData);
                logUsersOutput({
                    message: 'Storage Users Data',
                    count: data.length,
                    sampleUsers: data.slice(0, 3)
                });
                showUsersDataStats(data, 'Storage', 'usersDataStats');
            } else {
                logUsersOutput('No users data in storage');
            }
        }

        function showUsersMemoryData() {
            if (usersData.length === 0) {
                logUsersOutput('No users data in memory');
                return;
            }
            logUsersOutput({
                message: 'Memory Users Data',
                count: usersData.length,
                sampleUsers: usersData.slice(0, 3)
            });
            showUsersDataStats(usersData, 'Memory', 'usersDataStats');
        }

        function clearUsersOutput() {
            document.getElementById('usersOutputContent').textContent = 'Output cleared';
            document.getElementById('usersDataStats').innerHTML = '';
        }

        // Items Management (similar to Users)
        function initItemsDataLoading() {
            const strategy = document.querySelector('input[name="itemsLoadStrategy"]:checked').value;
            
            if (strategy === 'storage') {
                if (!loadItemsFromStorage()) {
                    logItemsOutput('No items data in storage. Please load from server manually.');
                }
            } 
            else if (strategy === 'server') {
                const hasLocalData = localStorage.getItem('itemsData') !== null;
                if (!hasLocalData) {
                    logItemsOutput('No local items data found. Click "Load Items From Server" to fetch data.');
                } else {
                    loadItemsFromServer();
                }
            }
        }

        async function loadItemsFromServer() {
            try {
                const fullUrl = `${itemsApiUrl}?action=list_all`;
                logItemsOutput(`Loading items data from server: ${fullUrl}`);
                
                const response = await fetch(fullUrl);
                if (!response.ok) throw new Error(`Server error: ${response.status}`);
                
                const result = await response.json();
                if (result.data && Array.isArray(result.data)) {
                    itemsData = result.data;
                    logItemsOutput({
                        message: 'Items data loaded from server',
                        count: itemsData.length,
                        sampleItem: itemsData.length > 0 ? itemsData[0] : null
                    });
                    showItemsDataStats(itemsData, 'Server', 'itemsDataStats');
                    saveItemsToStorage();
                } else {
                    throw new Error('Invalid items data format from server');
                }
            } catch (error) {
                logItemsOutput(`Error loading items from server: ${error.message}`);
            }
        }

        function loadItemsFromStorage() {
            const storedData = localStorage.getItem('itemsData');
            if (storedData) {
                itemsData = JSON.parse(storedData);
                logItemsOutput({
                    message: 'Items data loaded from storage',
                    count: itemsData.length,
                    sampleItem: itemsData.length > 0 ? itemsData[0] : null
                });
                showItemsDataStats(itemsData, 'Storage', 'itemsDataStats');
                return true;
            }
            logItemsOutput('No items data found in storage');
            return false;
        }

        function saveItemsToStorage() {
            localStorage.setItem('itemsData', JSON.stringify(itemsData));
            logItemsOutput(`Items data saved to storage (${itemsData.length} items)`);
            showItemsDataStats(itemsData, 'Storage', 'itemsDataStats');
        }

        function clearItemsStorage() {
            localStorage.removeItem('itemsData');
            itemsData = [];
            logItemsOutput('Items storage and memory data cleared');
            document.getElementById('itemsDataStats').innerHTML = '';
        }

        function showItemsDataStructure() {
            if (itemsData.length === 0) {
                logItemsOutput('No items data available to show structure');
                return;
            }
            
            const sampleItem = itemsData[0];
            const structure = {};
            for (const key in sampleItem) {
                structure[key] = typeof sampleItem[key];
            }
            
            logItemsOutput({
                message: 'Items Data Structure (first item)',
                structure: structure,
                sample: sampleItem
            });
            showItemsDataStats(sampleItem, 'Item Structure', 'itemsDataStats');
        }

        function showItemsStorageData() {
            const storedData = localStorage.getItem('itemsData');
            if (storedData) {
                const data = JSON.parse(storedData);
                logItemsOutput({
                    message: 'Storage Items Data',
                    count: data.length,
                    sampleItems: data.slice(0, 3)
                });
                showItemsDataStats(data, 'Storage', 'itemsDataStats');
            } else {
                logItemsOutput('No items data in storage');
            }
        }

        function showItemsMemoryData() {
            if (itemsData.length === 0) {
                logItemsOutput('No items data in memory');
                return;
            }
            logItemsOutput({
                message: 'Memory Items Data',
                count: itemsData.length,
                sampleItems: itemsData.slice(0, 3)
            });
            showItemsDataStats(itemsData, 'Memory', 'itemsDataStats');
        }

        function clearItemsOutput() {
            document.getElementById('itemsOutputContent').textContent = 'Output cleared';
            document.getElementById('itemsDataStats').innerHTML = '';
        }
    </script>
</body>
</html>