window.debug = false;
window.logomode = true;

function displayMessage(message, isError = false, isInfo = false) {
    let msgDiv = document.getElementById("message");
    if (!msgDiv) {
        msgDiv = document.createElement("div");
        msgDiv.id = "message";
        document.body.insertBefore(msgDiv, document.body.firstChild);
    }
    msgDiv.textContent = message;
    msgDiv.className = "message " + (isError ? "error" : isInfo ? "info" : "success");
    if (isInfo) {
        setTimeout(() => {
            if (msgDiv.className.includes("info")) {
                msgDiv.textContent = "";
                msgDiv.className = "message";
            }
        }, 3000);
    }
}

function logoset(targetId, previewId = "updateImgPreview") {
    const stringText = window.imgsrc;
    const targetInput = document.getElementById(targetId);
    const previewDiv = document.getElementById(previewId);
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';
    if (targetInput && typeof stringText !== "undefined") {
        targetInput.value = stringText;
        if (previewDiv) {
            previewDiv.innerHTML = `<img src="${stringText}" alt="preview" style="max-width: 100px; max-height: 100px; object-fit: cover; border-radius: 4px;">`;
            const img = previewDiv.querySelector('img');
            img.onerror = () => {
                console.warn("Failed to load preview image, reverting to default.");
                img.src = defaultAvatar;
            };
        }
    }
}

function togglePassword(inputId, buttonId) {
    const input = document.getElementById(inputId);
    if (input) {
        input.type = input.type === "password" ? "text" : "password";
        const button = document.getElementById(buttonId);
        if (button)
            button.textContent = input.type === "password" ? "Show" : "Hide";
    }
}

async function postto(apiUrl = "", payload = {}) {
    try {
        const response = await fetch(apiUrl, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: payload }),
        });
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error("Request failed:", error);
        displayMessage(error.message, true);
        throw error;
    }
}

function closePanel(id) {
    const panel = document.getElementById(id);
    if (panel) panel.remove();
}

async function checkUser(userId) {
    try {
        const response = await fetch(`https://i.kunok.com/aui.php?action=list&id=${encodeURIComponent(userId)}`);
        const result = await response.json();
        if (result.data && result.data.length > 0) {
            return result.data[0];
        } else {
            throw new Error('User not found');
        }
    } catch (error) {
        throw new Error('Error fetching user: ' + error.message);
    }
}

async function updateUserPanel(userId = "") {
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';

    if (!userId && window.user && window.user.id) {
        userId = window.user.id;
    }

    if (!userId) {
        displayMessage("No user ID provided", true);
        return;
    }

    try {
        displayMessage("Loading user data...", false, true);
        const user = await checkUser(userId);

        const formHTML = `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                #updateImgPreview img {
                    max-width: 100px;
                    max-height: 100px;
                    object-fit: cover;
                    border-radius: 4px;
                }
            </style>
            <div class="logo-controls">
                <div id="updateImgPreview"><img src="${user.img ? atob(user.img) : defaultAvatar}" alt="preview"></div>
                <div class="button-group" style="display: flex; gap: 10px; justify-content: center; margin-top: 10px;">
                    <button type="button" id="updateUploadLogoBtn">Upload logo</button>
                    <button type="button" id="updateApplyLogoBtn">Apply Logo</button>
                </div>
            </div>
            <form id="updateUserForm">
                <div><label for="updateId">ID:</label> <input type="text" id="updateId" value="${user.id}" style="width: 100%;></div>
                <div><label for="updateName">Name:</label> <input type="text" id="updateName" value="${user.name || ''}"></div>
                <div><label for="updateUsername">Username:</label> <input type="text" id="updateUsername" value="${user.username || ''}"></div>
                <div>
                    <label for="updatePassword">Password:</label>
                    <input type="password" id="updatePassword" value="${user.password || ''}">
                    <button type="button" id="toggleUpdatePasswordBtn">Show</button>
                </div>
                <div class="form-row">
                    <div>
                        <label for="updateFlag">Flag:</label>
                        <select id="updateFlag">
                            <option value="1" ${user.flag == 1 ? 'selected' : ''}>1 (Active)</option>
                            <option value="0" ${user.flag == 0 ? 'selected' : ''}>0 (Suspended)</option>
                            <option value="3" ${user.flag == 3 ? 'selected' : ''}>3 (Wait)</option>
                        </select>
                    </div>
                    <div>
                        <label for="updateLevel">Level:</label>
                        <select id="updateLevel">
                            <option value="1" ${user.level == 1 ? 'selected' : ''}>1 (User)</option>
                            <option value="2" ${user.level == 2 ? 'selected' : ''}>2 (Moderator)</option>
                            <option value="3" ${user.level == 3 ? 'selected' : ''}>3 (Admin)</option>
                        </select>
                    </div>
                </div>
                <div>
                    <label for="updateImg">Image URL:</label>
                    <input type="text" id="updateImg" value="${user.img ? atob(user.img) : ''}">
                </div>
                <button type="submit">Update User</button>
            </form>
            <div id="message" class="message"></div>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("updateUserPanel", "CC", "ctc", 400, 600, {
            bodyContent: formHTML,
            noFooter: true,
            title: 'Update User'
        });

        const form = document.getElementById("updateUserForm");
        if (form) {
            form.addEventListener("submit", async (e) => {
                e.preventDefault();
                const payload = {
                    id: userId,
                    name: document.getElementById("updateName").value,
                    username: document.getElementById("updateUsername").value,
                    password: document.getElementById("updatePassword").value,
                    flag: parseInt(document.getElementById("updateFlag").value),
                    level: parseInt(document.getElementById("updateLevel").value),
                    img: document.getElementById("updateImg").value ? btoa(document.getElementById("updateImg").value) : ''
                };
                try {
                    const response = await postto("https://i.kunok.com/aui.php?action=update", payload);
                    displayMessage(response.message || `User ${userId} updated successfully`);
form.reset();
//document.getElementById("updateImgPreview").innerHTML = `<img src="${defaultAvatar}" alt="preview">`;

// Wait 2 seconds to show message before redirect
// setTimeout(() => { closePanel("updateUserPanel"); window.location.href = "index.html";}, 2000);

                } catch (error) {
                    console.error("Update user error:", error);
                    displayMessage(`Error updating user ${userId}: ${error.message}`, true);
                }
            });

            document.getElementById("toggleUpdatePasswordBtn").addEventListener("click", () =>
                togglePassword("updatePassword", "toggleUpdatePasswordBtn")
            );
            document.getElementById("updateUploadLogoBtn").addEventListener("click", () => {
                window.logomode = true;
                toggleUploadPanel(); // from upl3.js
            });
            document.getElementById("updateApplyLogoBtn").addEventListener("click", () =>
                logoset("updateImg", "updateImgPreview")
            );
        }

    } catch (error) {
        displayMessage(error.message, true);
    }
}
async function listUsersPanel() {
    try {
        // Display loading message
        displayMessage("Loading users...", false, true);

        // Fetch user list from the API
        const response = await fetch("https://j.kunok.com/aui.php?action=list", {
            method: "GET",
            headers: { "Content-Type": "application/json" },
        });

        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP error! status: ${response.status}`);
        }

        const result = await response.json();
        const users = result.data || [];

        if (users.length === 0) {
            displayMessage("No users found", false, true);
            return;
        }

        // Generate table HTML for user list
        const tableHTML = `
            <style>
                .user-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 10px;
                }
                .user-table th, .user-table td {
                    border: 1px solid #ccc;
                    padding: 8px;
                    text-align: left;
                }
                .user-table th {
                    background-color: #f4f4f4;
                }
                .user-table img {
                    max-width: 50px;
                    max-height: 50px;
                    object-fit: cover;
                    border-radius: 4px;
                }
                .action-buttons {
                    display: flex;
                    gap: 5px;
                }
            </style>
            <table class="user-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Username</th>
                        <th>Flag</th>
                        <th>Level</th>
                        <th>Image</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    ${users
                        .map(
                            (user) => `
                                <tr>
                                    <td>${user.id}</td>
                                    <td>${user.name || ""}</td>
                                    <td>${user.username || ""}</td>
                                    <td>${
                                        user.flag == 1
                                            ? "Active"
                                            : user.flag == 0
                                            ? "Suspended"
                                            : user.flag == 3
                                            ? "Wait"
                                            : user.flag
                                    }</td>
                                    <td>${
                                        user.level == 1
                                            ? "User"
                                            : user.level == 2
                                            ? "Moderator"
                                            : user.level == 3
                                            ? "Admin"
                                            : user.level
                                    }</td>
                                    <td><img src="${
                                        user.img ? atob(user.img) : "https://placehold.co/100x100/666/fff/png"
                                    }" alt="avatar" onerror="this.src='https://placehold.co/100x100/666/fff/png';"></td>
                                    <td class="action-buttons">
                                        <button onclick="updateUserPanel('${user.id}')">Update</button>
                                        <button onclick="deleteUser('${user.id}')">Delete</button>
                                    </td>
                                </tr>
                            `
                        )
                        .join("")}
                </tbody>
            </table>
            <div id="message" class="message"></div>
        `;

        // Create panel for user list
        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            displayMessage("Panel system not available", true);
            return;
        }

        createPanel("listUsersPanel", "CC", "ctc", 800, 600, {
            bodyContent: tableHTML,
            noFooter: true,
            title: "User List",
        });

        displayMessage(`${users.length} user(s) loaded`, false, true);

    } catch (error) {
        console.error("List users error:", error);
        displayMessage(`Error loading users: ${error.message}`, true);
    }
}

// Helper function to delete a user
async function deleteUser(userId) {
    if (!confirm(`Are you sure you want to delete user ${userId}?`)) return;

    try {
        displayMessage("Deleting user...", false, true);
        const response = await postto("https://j.kunok.com/aui.php?action=del", { id: userId });
        displayMessage(response.message || `User ${userId} deleted successfully`, false, true);

        // Refresh the user list after deletion
        listUsersPanel();
    } catch (error) {
        console.error("Delete user error:", error);
        displayMessage(`Error deleting user ${userId}: ${error.message}`, true);
    }
}