const host = "https://i.kunok.com/";

function displayMessage(message, isError = false, isInfo = false, panelId = "itemMessage") {
    let msgDiv = document.getElementById(panelId);
    if (!msgDiv) {
        msgDiv = document.createElement("div");
        msgDiv.id = panelId;
        document.body.insertBefore(msgDiv, document.body.firstChild);
    }
    msgDiv.textContent = message;
    msgDiv.className = "message " + (isError ? "error" : isInfo ? "info" : "success");
    if (isInfo) {
        setTimeout(() => {
            if (msgDiv.className.includes("info")) {
                msgDiv.textContent = "";
                msgDiv.className = "message";
            }
        }, 3000);
    }
}

async function postto(apiUrl = "", payload = {}) {
    try {
        const response = await fetch(apiUrl, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: payload }),
        });
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error("Request failed:", error);
        displayMessage(error.message, true);
        throw error;
    }
}

function closePanel(id) {
    const panel = document.getElementById(id);
    if (panel) panel.remove();
}

async function listItemsPanel() {
    let items = [];

    async function refreshSingleItem(itemId) {
        try {
            const res = await fetch(`${host}api622.php?id=${encodeURIComponent(itemId)}`);
            const json = await res.json();
            const item = json.data && json.data.length > 0 ? json.data[0] : json.item || json;

            if (!item || !item.id) return;

            const box = document.querySelector(`.item-entry[data-item-id="${item.id}"]`);
            if (box) {
                const flagText = item.flag == 1 ? 'Active' : item.flag == 0 ? 'Suspended' : item.flag == 3 ? 'Wait' : item.flag;
                box.innerHTML = `
                    <div><strong>ID:</strong> ${item.id}</div>
                    <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                    <div class="item-meta">Flag: ${flagText} | Timestamp: ${item.timestamp}</div>
                    <div><strong>Board:</strong></div>
                    <pre>${item.board || 'N/A'}</pre>
                    <div><strong>Keeper:</strong> ${item.keeper || 'N/A'}</div>
                `;
                box.style.transition = 'background-color 0.6s ease';
                box.style.backgroundColor = '#ffffcc';
                setTimeout(() => {
                    box.style.backgroundColor = '#ffffff';
                }, 600);
            }
        } catch (err) {
            console.error("Error refreshing single item:", err);
        }
    }

    function attachClickEvents() {
        const entries = document.querySelectorAll(".item-entry");
        entries.forEach(entry => {
            entry.addEventListener("click", () => {
                const itemId = entry.getAttribute("data-item-id");
                if (itemId) {
                    updateItemPanel(itemId, () => refreshSingleItem(itemId));
                } else {
                    displayMessage("Item ID not found", true);
                }
            });
        });
    }

    try {
        const response = await fetch(`${host}api62.php?action=list_all&fields=id,name,flag,timestamp,board,keeper`);
        const result = await response.json();
        items = result.data || [];

        if (!items.length) {
            displayMessage("No items found", false, true);
            return;
        }

        const html = `
            <style>
                .item-entry {
                    background: #fff;
                    padding: 12px;
                    border-radius: 6px;
                    margin-bottom: 10px;
                    border: 1px solid #ccc;
                    font-size: 14px;
                    cursor: pointer;
                    transition: background 0.2s;
                }
                .item-entry:hover {
                    background: #f8f8f8;
                }
                .item-entry pre {
                    background: #f5f5f5;
                    padding: 6px;
                    border-radius: 4px;
                    font-size: 12px;
                    overflow-x: auto;
                    margin-top: 4px;
                }
                .item-meta {
                    font-size: 12px;
                    color: #666;
                    margin-bottom: 4px;
                }
                .message.error {
                    color: red;
                    text-align: center;
                    margin: 20px;
                }
                .message.success {
                    color: green;
                }
                .message.info {
                    color: blue;
                }
            </style>
            <div id="itemsList">
                ${items.map(item => `
                    <div class="item-entry" data-item-id="${item.id}">
                        <div><strong>ID:</strong> ${item.id}</div>
                        <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                        <div class="item-meta">Flag: ${item.flag == 1 ? 'Active' : item.flag == 0 ? 'Suspended' : item.flag == 3 ? 'Wait' : item.flag} | Timestamp: ${item.timestamp}</div>
                        <div><strong>Board:</strong></div>
                        <pre>${item.board || 'N/A'}</pre>
                        <div><strong>Keeper:</strong> ${item.keeper || 'N/A'}</div>
                    </div>
                `).join("")}
            </div>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("listItemsPanel", "CC", "ctc", 600, 600, {
            bodyContent: html,
            noFooter: true,
            title: "Items List"
        });

        setTimeout(attachClickEvents, 100);
    } catch (err) {
        console.error("Error loading items:", err);
        createPanel("listItemsPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="message error">${err.message}</div>`,
            noFooter: true,
            title: "Items List"
        });
    }
}

async function updateItemPanel(itemId = "", onUpdateDone = null) {
    try {
        displayMessage("Loading item data...", false, true, "itemMessageEX");
        const response = await fetch(`${host}api622.php?id=${encodeURIComponent(itemId)}`);
        const result = await response.json();
        const item = result.data && result.data.length > 0 ? result.data[0] : result.item || result;

        if (!item || !item.id) {
            displayMessage("Item not found", true, false, "itemMessageEX");
            return;
        }

        const formHTML = `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                textarea { width: 100%; height: 100px; resize: vertical; font-family: monospace; font-size: 13px; padding: 6px; }
                input[type="text"], input[type="number"] { width: 100%; padding: 6px; }
                .actions { margin-top: 10px; display: flex; gap: 10px; flex-wrap: wrap; }
                .message { margin-top: 10px; font-size: 13px; }
                .message.error { color: red; }
                .message.success { color: green; }
                .message.info { color: blue; }
            </style>
            <form id="updateItemForm">
                <div><label for="itemId">ID:</label> <input type="text" id="itemId" value="${item.id}" readonly></div>
                <div><label for="itemName">Name:</label> <textarea id="itemName">${item.name || ''}</textarea></div>
                <div><label for="itemAge">Age:</label> <input type="number" id="itemAge" value="${item.age || ''}"></div>
                <div>
                    <label for="itemFlag">Flag:</label>
                    <select id="itemFlag">
                        <option value="1" ${item.flag == 1 ? 'selected' : ''}>1 (Active)</option>
                        <option value="0" ${item.flag == 0 ? 'selected' : ''}>0 (Suspended)</option>
                        <option value="3" ${item.flag == 3 ? 'selected' : ''}>3 (Wait)</option>
                    </select>
                </div>
                <div><label for="itemBucket">Bucket:</label> <textarea id="itemBucket">${item.bucket || ''}</textarea></div>
                <div><label for="itemCup">Cup:</label> <textarea id="itemCup">${item.cup || ''}</textarea></div>
                <div><label for="itemBoard">Board:</label> <textarea id="itemBoard">${item.board || ''}</textarea></div>
                <div><label for="itemKeeper">Keeper:</label> <textarea id="itemKeeper">${item.keeper || ''}</textarea></div>
                <div class="actions">
                    <button type="submit">Update Item</button>
                </div>
            </form>
            <div id="itemMessageEX" class="message"></div>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("updateItemPanel", "CC", "ctc", 500, 640, {
            bodyContent: formHTML,
            noFooter: true,
            title: "Update Item"
        });

        const form = document.getElementById("updateItemForm");
        if (form) {
            form.addEventListener("submit", async (e) => {
                e.preventDefault();
                const payload = {
                    id: itemId,
                    name: document.getElementById("itemName").value.trim(),
                    age: parseInt(document.getElementById("itemAge").value) || 0,
                    flag: parseInt(document.getElementById("itemFlag").value),
                    bucket: document.getElementById("itemBucket").value.trim(),
                    cup: document.getElementById("itemCup").value.trim(),
                    board: document.getElementById("itemBoard").value.trim(),
                    keeper: document.getElementById("itemKeeper").value.trim()
                };

                try {
                    const response = await postto(`${host}aui.php?action=update`, payload);
                    displayMessage(response.message || `Item ${itemId} updated successfully`, false, false, "itemMessageEX");
                    if (typeof onUpdateDone === "function") {
                        onUpdateDone();
                    }
                    closePanel("updateItemPanel");
                    listItemsPanel();
                } catch (error) {
                    console.error("Update item error:", error);
                    displayMessage(`Error updating item ${itemId}: ${error.message}`, true, false, "itemMessageEX");
                }
            });
        }

    } catch (error) {
        console.error("Failed to load item:", error);
        displayMessage(error.message, true, false, "itemMessageEX");
        createPanel("updateItemPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="message error">${error.message}</div>`,
            noFooter: true,
            title: "Update Item"
        });
    }
}