(() => {
  const ID = 'tinyListener';
  if (document.getElementById(ID)) return;

  /* 1️⃣  pill shell */
  const pill = document.createElement('div');
  pill.id = ID;
  pill.innerHTML = `
    <div class="avatar" id="pillAvatar"></div>
    <div class="dot" data-action="menu" title="Toggle menu"></div>
    <div class="dot close-dot" data-action="close" title="Close pill"></div>
  `;
  document.body.appendChild(pill);

  /* 2️⃣  styles */
  function addStyle(css) {
    const style = document.createElement('style');
    style.textContent = css;
    document.head.appendChild(style);
  }
  addStyle(`
    #tinyListener {
      position: fixed; left: 8px; bottom: 8px;
      display: flex; align-items: center; gap: 6px;
      padding: 4px 8px 4px 4px;
      background: rgba(30,30,30,.2);
      color: #fff; border-radius: 20px;
      font-size: 14px; cursor: grab; user-select: none;
      z-index: 9999; transition: transform .2s;
    }
    #tinyListener:active { cursor: grabbing; }
    #tinyListener .avatar {
      width: 24px; height: 24px; border-radius: 50%;
      background: #666; /* Gray placeholder if image fails */
      display: block;
    }
    #tinyListener .dot {
      width: 18px; height: 18px;
      border-radius: 50%;
      background: #fff;
      cursor: pointer;
    }
    #tinyListener .close-dot {
      background: #f33;
    }
    #tinyListener .dot:hover { transform: scale(1.25); }
  `);

  /* 3️⃣  drag & click logic */
  let dragOffsetX = 0, dragOffsetY = 0, isDragging = false;
  pill.addEventListener('mousedown', start);
  pill.addEventListener('touchstart', start, { passive: false });

  function start(e) {
    if (e.target.classList.contains('dot') || e.target.classList.contains('close-dot')) return;
    e.preventDefault();
    const r = pill.getBoundingClientRect();
    const cX = e.touches ? e.touches[0].clientX : e.clientX;
    const cY = e.touches ? e.touches[0].clientY : e.clientY;
    dragOffsetX = cX - r.left;
    dragOffsetY = cY - r.top;
    isDragging = true;

    document.addEventListener('mousemove', move);
    document.addEventListener('mouseup', stop);
    document.addEventListener('touchmove', move, { passive: false });
    document.addEventListener('touchend', stop);
  }
  function move(e) {
    if (!isDragging) return;
    const cX = e.touches ? e.touches[0].clientX : e.clientX;
    const cY = e.touches ? e.touches[0].clientY : e.clientY;
    const newLeft = cX - dragOffsetX;
    const newTop = cY - dragOffsetY;

    const r = pill.getBoundingClientRect();
    const maxLeft = window.innerWidth - r.width;
    const maxTop = window.innerHeight - r.height;
    pill.style.left = Math.max(0, Math.min(newLeft, maxLeft)) + 'px';
    pill.style.bottom = 'auto';
    pill.style.top = Math.max(0, Math.min(newTop, maxTop)) + 'px';
  }
  function stop() {
    if (isDragging) {
      localStorage.setItem('pillPosition', JSON.stringify({
        left: parseFloat(pill.style.left) || 8,
        top: parseFloat(pill.style.top) || (window.innerHeight - pill.getBoundingClientRect().height - 8)
      }));
    }
    isDragging = false;
    document.removeEventListener('mousemove', move);
    document.removeEventListener('mouseup', stop);
    document.removeEventListener('touchmove', move, { passive: false });
    document.removeEventListener('touchend', stop);
  }

  /* 4️⃣  Load saved position */
  const savedPosition = localStorage.getItem('pillPosition');
  if (savedPosition) {
    try {
      const { left, top } = JSON.parse(savedPosition);
      pill.style.left = left + 'px';
      pill.style.bottom = 'auto';
      pill.style.top = top + 'px';
       /* ✅ NEW: reset if off-screen */
    if (!isOnScreen(pill)) {
      resetToDefault();
    }
    } catch (e) {
      console.warn('Failed to load pill position:', e);
    }
  }

  /* 5️⃣  Cache and update avatars */
  let cachedAvatar = null;
  function updateAvatar() {
    if (cachedAvatar !== null) {
      applyAvatar(cachedAvatar);
      return;
    }

    const defaultAvatar = 'png.png'; // CDN-hosted placeholder
    let user_img = defaultAvatar;
    const user_imgr = localStorage.getItem("img") || "";
    console.log("Stored img in localStorage:", user_imgr); // Debug
    if (user_imgr && /^https?:\/\//.test(user_imgr)) {
      user_img = user_imgr; // Use stored URL if it’s valid
    } else if (user_imgr) {
      console.warn("Stored image is not a valid URL, clearing and using default:", user_imgr);
      localStorage.removeItem("img");
    }

    cachedAvatar = user_img;
    applyAvatar(user_img);
  }
function isOnScreen(el) {
  const { left, top, right, bottom } = el.getBoundingClientRect();
  return right >= 0 && left <= window.innerWidth &&
         bottom >= 0 && top <= window.innerHeight;
}

function resetToDefault() {
  pill.style.left = '8px';
  pill.style.bottom = '8px';
  pill.style.top = 'auto';
  localStorage.removeItem('pillPosition');
}
  function applyAvatar(imgSrc) {
    const defaultAvatar = 'png.png';
    const pillAvatar = document.getElementById('pillAvatar');
    if (pillAvatar) {
      const img = document.createElement('img');
      img.id = 'pillAvatar';
      img.className = 'avatar';
      img.src = imgSrc;
      img.alt = 'User Avatar';
      img.style.cssText = 'width: 24px; height: 24px; border-radius: 50%; display: block;';
      img.onerror = () => {
        console.warn("Failed to load avatar image, reverting to default.");
        img.src = defaultAvatar;
      };
      pillAvatar.replaceWith(img);
    }
    const avatarEl = document.getElementById('avatar');
    if (avatarEl) {
      avatarEl.src = imgSrc;
      avatarEl.alt = 'User Avatar';
      avatarEl.onerror = () => {
        console.warn("Failed to load bottom bar avatar, reverting to default.");
        avatarEl.src = defaultAvatar;
      };
    }
  }
/* 8️⃣  Global re-show helper */
/* 8️⃣  Global re-show helper */
window.showAvatarPill = () => {
  if (document.getElementById(ID)) return;          // already on screen
  // Re-insert the script exactly as it was delivered
  const s = document.createElement('script');
  s.src = 'avatar.js';           // <-- adjust path if necessary
  s.onload = () => s.remove();   // clean up
  document.head.appendChild(s);
  if (e.target.id === 'pillAvatar') {
      pill.style.left = '8px';
      pill.style.bottom = '8px';
      pill.style.top = 'auto';
      localStorage.setItem('pillPosition', JSON.stringify({
        left: 8,
        top: window.innerHeight - pill.getBoundingClientRect().height - 8
      }));
    }
};
  window.updateAvatar = updateAvatar;
  window.updateBottomBarAvatar = updateAvatar;

  /* 6️⃣  Click handlers */
  pill.addEventListener('click', e => {
    const target = e.target.closest('[data-action]');
    if (!target) return;
    const act = target.dataset.action;
    if (act === 'menu') {
      if (typeof toggleLeftMenu === 'function') {
        toggleLeftMenu();
      } else {
        console.warn('toggleLeftMenu function not defined');
      }
    }
    if (act === 'close') pill.remove();
  });

  /* 7️⃣  Double-click to reset position */
  pill.addEventListener('dblclick', e => {
    if (e.target.id === 'pillAvatar') {
      pill.style.left = '8px';
      pill.style.bottom = '8px';
      pill.style.top = 'auto';
      localStorage.setItem('pillPosition', JSON.stringify({
        left: 8,
        top: window.innerHeight - pill.getBoundingClientRect().height - 8
      }));
    }
  });

  /* Initialize avatar */
  updateAvatar();
})();