let currentMode = 'auto';
let colorSchemeListener = null;

// Add dynamic styles to the DOM
addStyle( `
        #colorSetting .color-setting-container {
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            font-family: Arial, sans-serif;
            transition: all 0.5s ease;
            padding: 20px;
        }
        #colorSetting .current-mode {
            margin-bottom: 10px;
            font-weight: bold;
            padding: 8px 15px;
            border-radius: 20px;
            display: inline-block;
            background-color: #f0f0f0;
        }
        #colorSetting .mode-auto { color: #2e7d32; }
        #colorSetting .mode-manual { color: #ff8f00; }
        #colorSetting .mode-toggle {
            margin-bottom: 20px;
            display: flex;
            gap: 10px;
        }
        #colorSetting .mode-toggle button.active {
            box-shadow: 0 0 0 2px white;
        }
        #colorSetting .controls {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 20px;
            margin-bottom: 30px;
        }
        #colorSetting .dots-container {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
        }
        #colorSetting .dot {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            cursor: pointer;
            display: flex;
            justify-content: center;
            align-items: center;
            font-size: 20px;
            user-select: none;
            border: 1px solid #ccc;
            transition: transform 0.2s;
        }
        #colorSetting .dot:hover {
            transform: scale(1.1);
        }
        #colorSetting .button-container {
            display: flex;
            gap: 10px;
            margin-top: 20px;
            flex-wrap: wrap;
            justify-content: center;
        }
        #colorSetting .color-setting-button {
            padding: 8px 16px;
            cursor: pointer;
            border: none;
            border-radius: 4px;
            background: #444;
            color: white;
            min-width: 80px;
            transition: opacity 0.2s;
        }
        #colorSetting .color-setting-button:hover {
            opacity: 0.8;
        }
        #colorSetting #day-btn { background-color: #4a90e2; }
        #colorSetting #night-btn { background-color: #d23669; }
        #colorSetting #auto-btn { background-color: #2e7d32; }
        #colorSetting #manual-btn { background-color: #ff8f00; }
        #colorSetting #save-theme-btn { background-color: #6a1b9a; }
        #colorSetting #revert-btn { background-color: #9c27b0; }
        #colorSetting #red-dot { background-color: #943333; color: white; }
        #colorSetting #green-dot { background-color: #96CA96; color: black; }
        #colorSetting #blue-dot { background-color: #9A9AE5; color: white; }
        #colorSetting #yellow-dot { background-color: #D9D991; color: black; }
        #colorSetting #white-dot { background-color: #ffffff; color: black; }
        #colorSetting #black-dot { background-color: #000000; color: white; }
        #colorSetting .color-settings {
            display: flex;
            gap: 30px;
            margin-top: 20px;
        }
        #colorSetting .color-setting {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 10px;
        }
        #colorSetting .color-label {
            font-weight: bold;
            font-size: 14px;
        }
        #colorSetting .color-picker {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        #colorSetting .color-input {
            width: 80px;
            height: 40px;
            padding: 0;
            border: 1px solid #ccc;
            border-radius: 4px;
        }
        #colorSetting .current-color {
            display: flex;
            align-items: center;
            gap: 5px;
        }
        #colorSetting .color-display {
            width: 40px;
            height: 40px;
            border-radius: 4px;
            border: 1px solid #ccc;
        }
        #colorSetting .color-code {
            width: 80px;
            padding: 5px;
            border: 1px solid #ccc;
            border-radius: 4px;
            text-align: center;
            font-family: monospace;
        }
        #colorSetting .custom-themes {
            margin-top: 30px;
            display: flex;
            flex-direction: column;
            gap: 10px;
            width: 100%;
            max-width: 400px;
        }
        #colorSetting .theme-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            background-color: rgba(255,255,255,0.2);
        }
        #colorSetting .theme-preview {
            display: flex;
            gap: 5px;
        }
        #colorSetting .theme-bg-preview, #colorSetting .theme-text-preview {
            width: 20px;
            height: 20px;
            border-radius: 50%;
            border: 1px solid #ccc;
        }
        #colorSetting .theme-actions {
            display: flex;
            gap: 5px;
        }
        #colorSetting .theme-name {
            flex-grow: 1;
            padding: 0 10px;
            font-weight: bold;
        }
        @media (max-width: 600px) {
            #colorSetting .color-settings {
                flex-direction: column;
                gap: 15px;
            }
            #colorSetting .dots-container {
                flex-wrap: wrap;
                justify-content: center;
            }
        }
    `);


// Toast notification function
function toast(message, { pos = "TR", slide = "ttb" }, { background = "black", color = "white" }, duration = 2000) {
    const toast = document.createElement("div");
    toast.textContent = message;
    toast.style.cssText = `
        position: fixed;
        ${pos === "TR" ? "top: 10px; right: 10px;" : "top: 10px; left: 10px;"}
        max-width: 300px;
        padding: 8px 12px;
        font-size: 14px;
        font-family: Arial, sans-serif;
        background: ${background};
        color: ${color};
        border-radius: 4px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        z-index: 1000;
        opacity: 0;
        transition: opacity 0.3s ease, transform 0.3s ease;
        ${slide === "ttb" ? "transform: translateY(-20px);" : ""}
    `;
    document.body.appendChild(toast);

    setTimeout(() => {
        toast.style.opacity = "1";
        toast.style.transform = slide === "ttb" ? "translateY(0)" : "";
    }, 10);

    setTimeout(() => {
        toast.style.opacity = "0";
        setTimeout(() => toast.remove(), 300);
    }, duration);
}

// Initialize the application
function initialize() {
    addStyle();
    renderColorSettings();

    const savedSettings = localStorage.getItem('themeSettings');
    if (savedSettings) {
        const settings = JSON.parse(savedSettings);
        if (settings.mode === 'manual') {
            setManualMode(false);
            if (settings.currentTheme) {
                loadThemeFromSettings(settings.currentTheme);
            }
        } else {
            setAutoMode(false);
        }
        if (settings.customThemes) {
            loadCustomThemes(settings.customThemes);
        }
    } else {
        setAutoMode(false);
    }

    updateModeButtons();
    updateModeDisplay();
}

// Render color settings UI
function renderColorSettings() {
    const container = document.getElementById('colorSetting');
    if (!container) {
        toast('❌ Color settings container not found', { pos: 'TR', slide: 'ttb' }, { background: 'red', color: 'white' }, 3000);
        return;
    }

    container.innerHTML = `
        <div class="color-setting-container">
            <div class="current-mode mode-auto" id="current-mode-display">Current Mode: Auto</div>
            <div class="mode-toggle">
                <button id="auto-btn" class="color-setting-button active" onclick="setAutoMode()">Auto Mode</button>
                <button id="manual-btn" class="color-setting-button" onclick="setManualMode()">Manual Mode</button>
            </div>
            <div class="controls">
                <div class="dots-container">
                    <div class="dot" id="red-dot" onclick="setCustomTheme('#943333', '#ffffff')">O</div>
                    <div class="dot" id="green-dot" onclick="setCustomTheme('#96CA96', '#000000')">O</div>
                    <div class="dot" id="blue-dot" onclick="setCustomTheme('#9A9AE5', '#ffffff')">O</div>
                    <div class="dot" id="yellow-dot" onclick="setCustomTheme('#D9D991', '#000000')">O</div>
                    <div class="dot" id="white-dot" onclick="setCustomTheme('#ffffff', '#000000')">O</div>
                    <div class="dot" id="black-dot" onclick="setCustomTheme('#000000', '#ffffff')">O</div>
                </div>
                <div class="color-settings">
                    <div class="color-setting">
                        <div class="color-label">Background Color</div>
                        <div class="color-picker">
                            <input type="color" id="bg-custom-color" value="#ff0000" class="color-input">
                          <!--  <button class="color-setting-button" onclick="applyCustomBgColor()">Apply</button> -->
                            <button class="color-setting-button" id="change-bg-btn" onclick="changeBackgroundOnly()">Change BG</button>
                        </div>
                        <div class="current-color">
                            <div class="color-display" id="bg-color-display"></div>
                            <input type="text" class="color-code" id="bg-color-code" 
                                   onchange="applyBgColorFromCode()" 
                                   onkeypress="if(event.key === 'Enter') applyBgColorFromCode()">
                        </div>
                    </div>
                    <div class="color-setting">
                        <div class="color-label">Text Color</div>
                        <div class="color-picker">
                            <input type="color" id="text-custom-color" value="#000000" class="color-input">
                        <!--    <button class="color-setting-button" onclick="applyCustomTextColor()">Apply</button> -->
                            <button class="color-setting-button" id="change-text-btn" onclick="changeTextOnly()">Change Text</button>
                        </div>
                        <div class="current-color">
                            <div class="color-display" id="text-color-display"></div>
                            <input type="text" class="color-code" id="text-color-code" 
                                   onchange="applyTextColorFromCode()" 
                                   onkeypress="if(event.key === 'Enter') applyTextColorFromCode()">
                        </div>
                    </div>
                </div>
            </div>
            <div class="button-container">
                <button class="color-setting-button" id="day-btn" onclick="setDayMode()">Day</button>
                <button class="color-setting-button" id="night-btn" onclick="setNightMode()">Night</button>
                <button class="color-setting-button" id="revert-btn" onclick="revertColors()">Revert Colors</button>
                <button class="color-setting-button" id="save-theme-btn" onclick="promptSaveCustomTheme()">Save Theme</button>
                <button class="color-setting-button" onclick="clearTheme()">Reset</button>
            </div>
            <div class="custom-themes" id="custom-themes-container">
                <h3>Saved Custom Themes</h3>
            </div>
        </div>
    `;

    // Ensure global functions are accessible
    window.setAutoMode = setAutoMode;
    window.setManualMode = setManualMode;
    window.setCustomTheme = setCustomTheme;
    window.applyCustomBgColor = applyCustomBgColor;
    window.applyCustomTextColor = applyCustomTextColor;
    window.changeBackgroundOnly = changeBackgroundOnly;
    window.changeTextOnly = changeTextOnly;
    window.applyBgColorFromCode = applyBgColorFromCode;
    window.applyTextColorFromCode = applyTextColorFromCode;
    window.setDayMode = setDayMode;
    window.setNightMode = setNightMode;
    window.revertColors = revertColors;
    window.promptSaveCustomTheme = promptSaveCustomTheme;
    window.clearTheme = clearTheme;
}

// Set auto mode based on browser preference
function setAutoMode(shouldSave = true) {
    if (colorSchemeListener) {
        colorSchemeListener.remove();
    }
    
    currentMode = 'auto';
    updateModeDisplay();
    
    if (window.matchMedia) {
        const colorSchemeQuery = window.matchMedia('(prefers-color-scheme: dark)');
        
        if (colorSchemeQuery.matches) {
            setNightMode();
        } else {
            setDayMode();
        }
        
        colorSchemeListener = colorSchemeQuery.addEventListener('change', (e) => {
            if (e.matches) {
                setNightMode();
            } else {
                setDayMode();
            }
        });
    } else {
        setDayNightMode();
    }
    
    if (shouldSave) {
        saveSettings();
    }
    
    updateModeButtons();
}

// Set manual mode
function setManualMode(shouldSave = true) {
    if (colorSchemeListener) {
        colorSchemeListener.remove();
        colorSchemeListener = null;
    }
    
    currentMode = 'manual';
    updateModeDisplay();
    
    const savedSettings = localStorage.getItem('themeSettings');
    if (savedSettings) {
        const settings = JSON.parse(savedSettings);
        if (settings.currentTheme) {
            loadThemeFromSettings(settings.currentTheme);
        }
    }
    
    if (shouldSave) {
        saveSettings();
    }
    
    updateModeButtons();
}

// Update mode toggle buttons appearance
function updateModeButtons() {
    const autoBtn = document.getElementById('auto-btn');
    const manualBtn = document.getElementById('manual-btn');
    if (autoBtn && manualBtn) {
        autoBtn.classList.toggle('active', currentMode === 'auto');
        manualBtn.classList.toggle('active', currentMode === 'manual');
    }
}

// Update the current mode display
function updateModeDisplay() {
    const display = document.getElementById('current-mode-display');
    if (display) {
        display.textContent = `Current Mode: ${currentMode.charAt(0).toUpperCase() + currentMode.slice(1)}`;
        display.className = `current-mode mode-${currentMode}`;
    }
}

// Set day/night mode based on current time
function setDayNightMode() {
    const hours = new Date("2025-07-06T10:28:00+07:00").getHours();
    const isDayTime = hours >= 6 && hours < 18;
    
    if (isDayTime) {
        setDayMode();
    } else {
        setNightMode();
    }
}

// Set day mode
function setDayMode() {
    applyTheme('#e6f2ff', '#333333');
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Set night mode
function setNightMode() {
    applyTheme('#ffe6e6', '#333333');
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Set custom theme
function setCustomTheme(bgColor, textColor) {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    applyTheme(bgColor, textColor);
    
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Apply theme colors
function applyTheme(bgColor, textColor) {
    document.body.style.backgroundColor = bgColor;
    document.body.style.color = textColor;
    updateCurrentColorDisplay(bgColor, textColor);
}

// Apply custom background color from color picker
function applyCustomBgColor() {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    const color = document.getElementById('bg-custom-color').value;
    document.body.style.backgroundColor = color;
    updateCurrentColorDisplay(color, document.body.style.color);
    
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Apply custom text color from color picker
function applyCustomTextColor() {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    const color = document.getElementById('text-custom-color').value;
    document.body.style.color = color;
    updateCurrentColorDisplay(document.body.style.backgroundColor, color);
    
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Change only background color
function changeBackgroundOnly() {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    const bgColor = ensureHex(document.getElementById('bg-custom-color').value);
    document.body.style.backgroundColor = bgColor;
    updateCurrentColorDisplay(bgColor, document.body.style.color);
    
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Change only text color
function changeTextOnly() {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    const textColor = ensureHex(document.getElementById('text-custom-color').value);
    document.body.style.color = textColor;
    updateCurrentColorDisplay(document.body.style.backgroundColor, textColor);
    
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Apply background color from code input
function applyBgColorFromCode() {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    let color = document.getElementById('bg-color-code').value;
    if (!color.startsWith('#')) color = '#' + color;
    
    if (/^#[0-9A-F]{6}$/i.test(color)) {
        document.getElementById('bg-custom-color').value = color;
        changeBackgroundOnly();
    } else {
        toast('❌ Please enter a valid hex color (e.g., #FF0000)', { pos: 'TR', slide: 'ttb' }, { background: 'red', color: 'white' }, 3000);
    }
}

// Apply text color from code input
function applyTextColorFromCode() {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    let color = document.getElementById('text-color-code').value;
    if (!color.startsWith('#')) color = '#' + color;
    
    if (/^#[0-9A-F]{6}$/i.test(color)) {
        document.getElementById('text-custom-color').value = color;
        changeTextOnly();
    } else {
        toast('❌ Please enter a valid hex color (e.g., #FF0000)', { pos: 'TR', slide: 'ttb' }, { background: 'red', color: 'white' }, 3000);
    }
}

// Update current color displays
function updateCurrentColorDisplay(bgColor, textColor) {
    const hexBg = ensureHex(bgColor);
    const hexText = ensureHex(textColor);
    
    const bgDisplay = document.getElementById('bg-color-display');
    const bgCode = document.getElementById('bg-color-code');
    const bgInput = document.getElementById('bg-custom-color');
    if (bgDisplay && bgCode && bgInput) {
        bgDisplay.style.backgroundColor = hexBg;
        bgCode.value = hexBg.toUpperCase();
        bgInput.value = hexBg;
    }
    
    const textDisplay = document.getElementById('text-color-display');
    const textCode = document.getElementById('text-color-code');
    const textInput = document.getElementById('text-custom-color');
    if (textDisplay && textCode && textInput) {
        textDisplay.style.backgroundColor = hexText;
        textCode.value = hexText.toUpperCase();
        textInput.value = hexText;
    }
}

// Format color code
function formatColorCode(color) {
    return color.startsWith('#') ? color.toUpperCase() : '#' + color.toUpperCase();
}

// Save current theme to localStorage
function saveCurrentTheme() {
    const theme = {
        bgColor: ensureHex(document.body.style.backgroundColor),
        textColor: ensureHex(document.body.style.color),
        isCustom: true
    };
    
    const settings = JSON.parse(localStorage.getItem('themeSettings') || '{}');
    settings.currentTheme = theme;
    settings.mode = currentMode;
    
    localStorage.setItem('themeSettings', JSON.stringify(settings));
}

// Prompt user to save a custom theme with a name
function promptSaveCustomTheme() {
    const themeName = prompt('Enter a name for this theme:');
    if (themeName) {
        saveCustomTheme(themeName);
    }
}

// Save custom theme with a name
function saveCustomTheme(name) {
    const theme = {
        name: name,
        bgColor: ensureHex(document.body.style.backgroundColor),
        textColor: ensureHex(document.body.style.color),
        timestamp: new Date('2025-07-06T10:28:00+07:00').getTime()
    };
    
    const settings = JSON.parse(localStorage.getItem('themeSettings') || '{}');
    if (!settings.customThemes) {
        settings.customThemes = [];
    }
    
    const existingIndex = settings.customThemes.findIndex(t => t.name === name);
    if (existingIndex >= 0) {
        settings.customThemes[existingIndex] = theme;
    } else {
        settings.customThemes.push(theme);
    }
    
    localStorage.setItem('themeSettings', JSON.stringify(settings));
    loadCustomThemes(settings.customThemes);
    toast(`✅ Theme "${name}" saved!`, { pos: 'TR', slide: 'ttb' }, { background: 'green', color: 'white' }, 2000);
}

// Load custom themes from settings
function loadCustomThemes(themes) {
    const container = document.getElementById('custom-themes-container');
    if (!container) return;
    
    while (container.children.length > 1) {
        container.removeChild(container.lastChild);
    }
    
    themes.sort((a, b) => b.timestamp - a.timestamp);
    
    themes.forEach(theme => {
        const themeItem = document.createElement('div');
        themeItem.className = 'theme-item';
        
        const preview = document.createElement('div');
        preview.className = 'theme-preview';
        
        const bgPreview = document.createElement('div');
        bgPreview.className = 'theme-bg-preview';
        bgPreview.style.backgroundColor = theme.bgColor;
        
        const textPreview = document.createElement('div');
        textPreview.className = 'theme-text-preview';
        textPreview.style.backgroundColor = theme.textColor;
        
        preview.appendChild(bgPreview);
        preview.appendChild(textPreview);
        
        const nameSpan = document.createElement('div');
        nameSpan.className = 'theme-name';
        nameSpan.textContent = theme.name;
        
        const actions = document.createElement('div');
        actions.className = 'theme-actions';
        
        const loadBtn = document.createElement('button');
        loadBtn.className = 'color-setting-button';
        loadBtn.textContent = 'Load';
        loadBtn.onclick = () => loadCustomTheme(theme);
        
        const deleteBtn = document.createElement('button');
        deleteBtn.className = 'color-setting-button';
        deleteBtn.textContent = 'Delete';
        deleteBtn.onclick = () => deleteCustomTheme(theme.name);
        
        actions.appendChild(loadBtn);
        actions.appendChild(deleteBtn);
        
        themeItem.appendChild(preview);
        themeItem.appendChild(nameSpan);
        themeItem.appendChild(actions);
        
        container.appendChild(themeItem);
    });
}

// Load a custom theme
function loadCustomTheme(theme) {
    setManualMode();
    applyTheme(theme.bgColor, theme.textColor);
    saveCurrentTheme();
    toast(`✅ Theme "${theme.name}" loaded!`, { pos: 'TR', slide: 'ttb' }, { background: 'green', color: 'white' }, 2000);
}

// Delete a custom theme
function deleteCustomTheme(name) {
    if (confirm(`Are you sure you want to delete the theme "${name}"?`)) {
        const settings = JSON.parse(localStorage.getItem('themeSettings') || '{}');
        if (settings.customThemes) {
            settings.customThemes = settings.customThemes.filter(t => t.name !== name);
            localStorage.setItem('themeSettings', JSON.stringify(settings));
            loadCustomThemes(settings.customThemes || []);
            toast(`✅ Theme "${name}" deleted!`, { pos: 'TR', slide: 'ttb' }, { background: 'green', color: 'white' }, 2000);
        }
    }
}

// Load theme from settings object
function loadThemeFromSettings(theme) {
    applyTheme(theme.bgColor, theme.textColor);
}

// Clear saved theme
function clearTheme() {
    if (confirm('Reset all settings and themes?')) {
        localStorage.removeItem('themeSettings');
        setAutoMode();
        const container = document.getElementById('custom-themes-container');
        if (container) {
            container.innerHTML = '<h3>Saved Custom Themes</h3>';
        }
        toast('✅ All settings have been reset!', { pos: 'TR', slide: 'ttb' }, { background: 'green', color: 'white' }, 2000);
    }
}

// Save settings
function saveSettings() {
    const settings = JSON.parse(localStorage.getItem('themeSettings') || '{}');
    settings.mode = currentMode;
    localStorage.setItem('themeSettings', JSON.stringify(settings));
}

// Revert background and text colors
function revertColors() {
    if (currentMode === 'auto') {
        setManualMode();
    }
    
    const currentBg = ensureHex(document.body.style.backgroundColor);
    const currentText = ensureHex(document.body.style.color);
    
    applyTheme(currentText, currentBg);
    
    if (currentMode === 'manual') {
        saveCurrentTheme();
    }
}

// Convert RGB to Hex
function rgbToHex(rgb) {
    if (rgb.startsWith('#')) return rgb;
    
    const result = /^rgb\((\d+),\s*(\d+),\s*(\d+)\)$/i.exec(rgb);
    if (!result) return '#000000';
    
    const r = parseInt(result[1]).toString(16).padStart(2, '0');
    const g = parseInt(result[2]).toString(16).padStart(2, '0');
    const b = parseInt(result[3]).toString(16).padStart(2, '0');
    
    return `#${r}${g}${b}`;
}

// Ensure hex format
function ensureHex(color) {
    if (!color) return '#000000';
    if (color.startsWith('#')) return color.length === 7 ? color : `#${color.substring(1).padStart(6, '0')}`;
    if (color.startsWith('rgb')) return rgbToHex(color);
    return `#${color.padStart(6, '0')}`;
}

// Initialize the app
//document.addEventListener('DOMContentLoaded', initialize);