function createDiv(id, content = '') {
    const div = document.createElement('div');
    div.id = id;
    if (content) div.innerHTML = content;
    document.body.appendChild(div);
    return div; // Return the element for chaining
}

function setTo(id, content) {
    const element = document.getElementById(id);
    if (!element) {
        console.error('Element with id "' + id + '" not found');
        return false;
    }
    
    // Set content - using innerHTML if content contains HTML tags, otherwise textContent
    if (/<[a-z][\s\S]*>/i.test(content)) {
        element.innerHTML = content;
    } else {
        element.textContent = content;
    }
    
    return true;
}
// Move element by setting left/top positions (position must be absolute/fixed/relative)
function moveXY(id, x, y) {
    const el = document.getElementById(id);
    if (el) {
        el.style.position = el.style.position || 'absolute';
        el.style.left = typeof x === 'number' ? x + 'px' : x;
        el.style.top = typeof y === 'number' ? y + 'px' : y;
    }
    return el;
}

// Set width and height (height defaults to width if not specified)
// Modified to work with both IDs and element references
function setWH(target, width, height = width) {
    const el = typeof target === 'string' ? document.getElementById(target) : target;
    if (el) {
        el.style.width = typeof width === 'number' ? width + 'px' : width;
        el.style.height = typeof height === 'number' ? height + 'px' : height;
    }
    return el;
}


// Set background color
function setBackground(id, backgroundColor) {
    const el = document.getElementById(id);
    if (el) {
        el.style.backgroundColor = backgroundColor;
    }
    return el;
}

// Set text color
function setColor(id, color) {
    const el = document.getElementById(id);
    if (el) {
        el.style.color = color;
    }
    return el;
}

// Set class (replaces all existing classes)
function setClass(id, className) {
    const el = document.getElementById(id);
    if (el) {
        el.className = className;
    }
    return el;
}

// Add class (keeps existing classes)
function addClass(id, className) {
    const el = document.getElementById(id);
    if (el) {
        el.classList.add(className);
    }
    return el;
}

// Remove class
function removeClass(id, className) {
    const el = document.getElementById(id);
    if (el) {
        el.classList.remove(className);
    }
    return el;
}
function showDiv(id) {
    const el = document.getElementById(id);
    if (el) el.style.display = 'block';
    return el;
}
function hideDiv(id) {
    const el = document.getElementById(id);
    if (el) el.style.display = 'none';
    return el;
}
function removeDiv(id) {
    const el = document.getElementById(id);
    if (el) el.remove();
    return null; // Element no longer exists
}
function addStyle(stylesText) {
    const styleElement = document.createElement('style');
    styleElement.textContent = stylesText;
    document.head.appendChild(styleElement);
}
function dragDiv(id) {
    const el = document.getElementById(id);
    if (!el) return null;

    let isDragging = false;
    let offsetX, offsetY;

    // Make element positionable if not already
    el.style.position = 'absolute';
    el.style.cursor = 'grab';

    // Mouse down (start drag)
    el.addEventListener('mousedown', (e) => {
        isDragging = true;
        offsetX = e.clientX - el.getBoundingClientRect().left;
        offsetY = e.clientY - el.getBoundingClientRect().top;
        el.style.cursor = 'grabbing';
        e.preventDefault(); // Prevent text selection
    });

    // Mouse move (during drag)
    document.addEventListener('mousemove', (e) => {
        if (!isDragging) return;
        el.style.left = `${e.clientX - offsetX}px`;
        el.style.top = `${e.clientY - offsetY}px`;
    });

    // Mouse up (end drag)
    document.addEventListener('mouseup', () => {
        isDragging = false;
        el.style.cursor = 'grab';
    });
// Touch start
el.addEventListener('touchstart', (e) => {
    isDragging = true;
    offsetX = e.touches[0].clientX - el.getBoundingClientRect().left;
    offsetY = e.touches[0].clientY - el.getBoundingClientRect().top;
    e.preventDefault();
});

// Touch move
document.addEventListener('touchmove', (e) => {
    if (!isDragging) return;
    el.style.left = `${e.touches[0].clientX - offsetX}px`;
    el.style.top = `${e.touches[0].clientY - offsetY}px`;
});

// Touch end
document.addEventListener('touchend', () => {
    isDragging = false;
});
    return el; // Chainable
}
/*
// Create a div first
createDiv('box1', 'Hello World!');

// Position it at 100px, 50px
moveXY('box1', 100, 50);

// Make it 200px square
setWH('box1', 200);

// Or rectangle: 200x100
setWH('box1', 200, 100);

// Set background to red
setBackground('box1', 'red');

// Set text color to white
setColor('box1', 'white');

// Set class (replaces all existing classes)
setClass('box1', 'active-box');

// Add additional class (keeps existing classes)
addClass('box1', 'rounded-corners');

// Remove a class
removeClass('box1', 'active-box');

You can chain these functions like:
setWH('box1', 100)
    .style.border = '1px solid black';
createDiv('myDiv', 'Hello World!')
    .style.border = '1px solid black';

hideDiv('myDiv'); // Hides the div
showDiv('myDiv'); // Shows it again
removeDiv('myDiv'); // Deletes it from DOM

// Create a draggable box
createDiv('dragBox', 'Drag me!')
    .style.width = '100px'
    .style.height = '100px'
    .style.background = 'lightcoral'
    .style.textAlign = 'center'
    .style.lineHeight = '100px';

// Make it draggable
dragDiv('dragBox');
*/
// Example usage:
//createDiv('myDiv');
//setTo('myDiv', 'This is plain text');  // Uses textContent
//setTo('myDiv', '<strong>This is HTML</strong>');  // Uses innerHTML
// Create a draggable box
// 1. Create and style the div
// 1. Create the element
//const box = createDiv('dragBox', 'Drag me!');

// 2. Chain your functions


//setWH('dragBox',200);
//setBackground('dragBox','red')
    

// 3. Make draggable


