function makeClock(targetId) {
  const target = document.getElementById(targetId);
  if (!target) return;

  target.innerHTML = `
    <div class="clock-container">
      <div id="clockScale" class="clock-scale">
        <div class="analog-clock" id="analogClock">
          <div class="hand hour" id="hourHand"></div>
          <div class="hand minute" id="minuteHand"></div>
          <div class="hand second" id="secondHand"></div>
          <div class="center-cap"></div>
        </div>
      </div>

      <div class="info-panel">
        <div class="day-night-container">
          <div class="info date" id="dateInfo"></div>
          <div class="info day" id="dayInfo"></div>             
          <div class="day-night day-indicator" id="dayNight"></div>      
        </div>
        <div class="day-night-container">
          <div class="digital-clock" id="digitalClock">00:00:00</div>
          <div class="day-night-indicator" id="dayNightIndicator"></div>
        </div>
        <div class="countdown" id="countdown">Countdown: 00:00:00</div>
        <div class="expired" id="expiredMsg" style="display:none;">EXPIRED!</div>
      </div>
    </div>
  `;

  const hourHand = target.querySelector('#hourHand');
  const minuteHand = target.querySelector('#minuteHand');
  const secondHand = target.querySelector('#secondHand');
  const digitalClock = target.querySelector('#digitalClock');
  const countdown = target.querySelector('#countdown');
  const dateInfo = target.querySelector('#dateInfo');
  const dayInfo = target.querySelector('#dayInfo');
  const dayNight = target.querySelector('#dayNight');
  const dayNightIndicator = target.querySelector('#dayNightIndicator');
  const analogClock = target.querySelector("#analogClock");

  const expireDate = new Date("2025-12-31T23:59:59");
  const countdownEndTime = new Date(Date.now() + 3600000); // 1 hour from now

  function padZero(num) {
    return num < 10 ? `0${num}` : num;
  }

  function createNumbers(large = false, radius = 93) {
    for (let i = 1; i <= 12; i++) {
      const num = document.createElement("div");
      num.className = "clock-number";
      num.textContent = i;

      const angle = (i - 3) * 30;
      const x = 120 + radius * Math.cos(angle * Math.PI / 180);
      const y = 120 + radius * Math.sin(angle * Math.PI / 180);

      num.style.left = `${x}px`;
      num.style.top = `${y}px`;

      if (large && [3, 6, 9, 12].includes(i)) {
        num.classList.add("large");
      }

      analogClock.appendChild(num);
    }
  }

  function createTicks() {
    for (let i = 0; i < 60; i++) {
      const tick = document.createElement("div");
      tick.className = (i % 5 === 0) ? "tickHour" : "tick";
      tick.style.transform = `rotate(${i * 6}deg) translate(-50%, 0)`;
      analogClock.appendChild(tick);
    }
  }

  function updateClock() {
    const now = new Date();
    const ms = now.getMilliseconds();
    const sec = now.getSeconds() + ms / 1000;
    const min = now.getMinutes() + sec / 60;
    const hr = now.getHours() % 12 + min / 60;

    hourHand.style.transform = `rotate(${hr * 30}deg)`;
    minuteHand.style.transform = `rotate(${min * 6}deg)`;
    secondHand.style.transform = `rotate(${sec * 6}deg)`;

    digitalClock.textContent = `${padZero(now.getHours())}:${padZero(now.getMinutes())}:${padZero(now.getSeconds())}`;

    const days = ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"];
    const months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
    dateInfo.textContent = `Date: ${now.getDate()} ${months[now.getMonth()]} ${now.getFullYear()}`;
    dayInfo.textContent = `${days[now.getDay()]}`;
    dayNight.textContent = now.getHours() >= 6 && now.getHours() < 18 ? "Day" : "Night";
    dayNightIndicator.className = now.getHours() >= 6 && now.getHours() < 18
      ? "day-night-indicator day"
      : "day-night-indicator night";

    const diff = expireDate - now;
    if (diff <= 0) {
      countdown.textContent = "Countdown: EXPIRED";
      target.querySelector("#expiredMsg").style.display = "block";
    } else {
      const daysLeft = Math.floor(diff / (1000 * 60 * 60 * 24));
      const hoursLeft = Math.floor((diff / (1000 * 60 * 60)) % 24);
      const minsLeft = Math.floor((diff / (1000 * 60)) % 60);
      const secsLeft = Math.floor((diff / 1000) % 60);
      countdown.textContent = `Countdown: ${daysLeft}d ${padZero(hoursLeft)}h ${padZero(minsLeft)}m ${padZero(secsLeft)}s`;
      target.querySelector("#expiredMsg").style.display = "none";
    }

    requestAnimationFrame(updateClock);
  }

  createTicks();
  createNumbers(true);
  updateClock();

  // Optional: scale it
  target.querySelector("#clockScale").style.transform = "scale(0.55)";
}
function addClockStyle() {
  const style = document.createElement('style');
  style.textContent = `
  .clock-container {
      display: flex;
      flex-direction: column;
      align-items: center;
      justify-content: flex-start;
      gap: 1px; /* Reduce space between elements */
      padding: 1px;
      margin: 0;
    }

    .info-panel {
      margin-top: 4px; /* reduced from 12px */
    }
    #clocket {
      margin-top: 10px;
    }
  `;
  document.head.appendChild(style);
}

addStyle(`

  .clock-container {
    display: flex;
    flex-direction: column;
    align-items: center;
    font-family: sans-serif;
  }

  .analog-clock {
    position: relative;
    width: 240px;
    height: 240px;
    border-radius: 50%;
    
    box-shadow: inset 0 0 15px rgba(0,0,0,0.3), 0 10px 20px rgba(0,0,0,0.4);
    margin-bottom: 10px;
  }

  .hand {
    position: absolute;
    left: 50%;
    transform-origin: 50% 100%;
    bottom: 50%;
  }

  .hour { width: 8px; height: 55px; background: #222; margin-left: -4px; }
  .minute { width: 6px; height: 85px; background: #777; margin-left: -3px; }
  .second { width: 2px; height: 90px; background: red; margin-left: -1px; }

  .center-cap {
    position: absolute;
    width: 14px;
    height: 14px;
    background: red;
    border-radius: 50%;
    top: 50%; left: 50%;
    transform: translate(-50%, -50%);
    z-index: 10;
  }

  .tick, .tickHour {
    position: absolute;
    top: 0;
    left: 50%;
    transform-origin: center 120px;
    background: #ccc;
  }

  .tick { width: 2px; height: 8px; }
  .tickHour { width: 4px; height: 18px; background: #aaa; }

  .clock-number {
    position: absolute;
    transform: translate(-50%, -50%);
    font-size: 12px;
    color: #555;
  }

  .clock-number.large {
    font-size: 18px;
    font-weight: bold;
    color: #000;
  }

  .day-night-container {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
  }

  .digital-clock {
    font-size: 20px;
    font-weight: bold;
  }

  .day-night-indicator {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    animation: pulse 2s infinite;
  }

  .day-night-indicator.day { background: #f4d99f; box-shadow: 0 0 6px #f15e03; }
  .day-night-indicator.night { background: #2196F3; box-shadow: 0 0 6px #2196F3; }

  @keyframes pulse {
    0% { transform: scale(1); opacity: 1; }
    50% { transform: scale(1.3); opacity: 0.7; }
    100% { transform: scale(1); opacity: 1; }
  }

  .countdown {
    margin-top: 6px;
    font-size: 14px;
    color: #666;
  }
`);

