function genform(divId, data) {
  console.log("🧪 genform data keys:", Object.keys(data));
  console.log("🧪 schema.grouping:", data.grouping);
  const container = document.getElementById(divId);
  if (!container) {
    console.error(`Div with ID ${divId} not found`);
    return;
  }

  // Store reference to original data (no deep copy)
  container._formData = data;
  const lang = window.lang || "en";

  // Normalize values on load
  for (const key in data) {
    if (key === "grouping" || data[key].type === "section") continue;
    const field = data[key];
    if (field.data_type === "radiotext" || field.data_type === "radiotextatt" || field.data_type === "radioatt") {
      if (!field.value || field.value.radio === "" || field.value.text === "Others (please specify)/อื่น ๆ (โปรดระบุ)") {
        field.value = { radio: null, text: "", att: "" };
      }
    } else if (field.data_type === "checktext" || field.data_type === "checktextatt") {
      if (!field.value || !Array.isArray(field.value.check)) {
        field.value = { check: [], text: "", att: "" };
      }
    }
  }

  // Initialize container HTML
  container.innerHTML = `
    <div style="font-family: Arial, sans-serif;">
      <div style="display: none; justify-content: space-between; align-items: center;">
        <h3 style="font-size: 0.8em;">${getLabel({ label: { en: "Form in", th: "FORM" } }, lang)} ${divId}</h3>
      </div>
      <div id="${divId}-fields"></div>
      <div style="display:none;">
        <textarea id="${divId}-input" placeholder="Enter JSON schema"></textarea>
        <button id="${divId}-copy-btn">Copy to Schema</button>
        <textarea id="${divId}-output" readonly></textarea>
      </div>
    </div>
  `;

  const fieldsDiv = document.getElementById(`${divId}-fields`);
  const schema = container._formData;
  const grouping = getGroupingData(schema);

  // Organize segments and sections
  const segments = new Map();

  if (grouping) {
    const names = Array.isArray(grouping.name) ? grouping.name : [];
    const types = Array.isArray(grouping.type) ? grouping.type : [];
    const labels = {
      en: Array.isArray(grouping.label?.en) ? grouping.label.en : [],
      th: Array.isArray(grouping.label?.th) ? grouping.label.th : [],
    };
    console.log("Grouping:", grouping);
    console.log("Names:", names, "Types:", types, "Labels:", labels);

    names.forEach((name, i) => {
      if (types[i] === "segment") {
        const segmentLabel = {
          en: labels.en[i] || name,
          th: labels.th[i] || name,
        };
        console.log(`Creating segment: ${name}, Label:`, segmentLabel);
        segments.set(name, {
          label: segmentLabel,
          sections: new Map(),
        });
      }
    });
  } else {
    console.warn("No valid grouping found in schema. Using default segment.");
    segments.set("default", {
      label: {
        en: "Default",
        th: "ค่าเริ่มต้น",
      },
      sections: new Map(),
    });
  }

  // Map fields to segments and sections
  for (const key in schema) {
    if (key === "grouping" || schema[key].type !== "field") continue;
    const field = schema[key];
    const segment = field.segment?.[0] || (grouping?.name && Array.isArray(grouping.name) ? grouping.name[0] : grouping?.name) || "default";

    if (!segments.has(segment)) {
      segments.set(segment, {
        label: { en: segment, th: segment },
        sections: new Map(),
      });
    }

    if (field.section) {
      if (!segments.get(segment).sections.has(field.section)) {
        if (schema[field.section]) {
          segments.get(segment).sections.set(field.section, getLabel(schema[field.section], lang) || field.section);
        } else {
          console.warn(`Section ${field.section} for field ${key} not found in schema`);
        }
      }
    } else {
      console.warn(`Field ${key} has no section assigned`);
    }
  }

  // Generate form structure
  segments.forEach((segmentData, segmentId) => {
    const segDetails = document.createElement("details");
    segDetails.className = "segment";
    segDetails.style.cssText = "border-radius: 5px; background-color: rgba(255,255,255,.3); backdrop-filter: blur(15px); border: 1px solid #4a90e2; margin-bottom: 10px; padding: 10px;";
    segDetails.id = `${divId}-segment-${encodeFieldKey(segmentId)}`;
    segDetails.open = true;
    segDetails.innerHTML = `<summary style="font-weight: bold; cursor: pointer;">${getLabel({ label: { en: "Segment", th: "Segment" } }, lang)}: ${getLabel(segmentData.label, lang) || segmentId}</summary>`;

    const segDiv = document.createElement("div");
    segDiv.style.padding = "10px";

    segmentData.sections.forEach((sectionLabel, sectionId) => {
      const secDetails = document.createElement("details");
      secDetails.className = "section";
      secDetails.id = `${divId}-section-${encodeFieldKey(sectionId)}`;
      secDetails.style.cssText = "box-shadow: 0 14px 12px rgba(0, 0, 0, 0.2); border-radius: 5px; background-color: rgba(255,255,255,.1); backdrop-filter: blur(20px); border: 1px solid green; margin: 8px 0; padding: 4px 8px;";
      secDetails.open = true;

      let sectionTotals = {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredFields: [],
      };

      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (field.section !== sectionId || !field.segment?.includes(segmentId)) continue;

        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const isAnswered = isFieldAnswered(field);
        const fileCount = countFiles(field);

        sectionTotals.allFields++;
        if (field.required) sectionTotals.requiredFields++;
        if (isAnswered) sectionTotals.answeredFields++;
        if (field.required && isAnswered) sectionTotals.answeredRequiredFields++;
        sectionTotals.totalScore += isAnswered ? field.score || 1 : 0;
        sectionTotals.totalScoreWeighted += currentScore;
        sectionTotals.maxPossibleScore += field.score || 1;
        sectionTotals.maxPossibleScoreWeighted += maxScore;
        sectionTotals.totalFiles += fileCount;
        if (field.required && !isFieldAnswered(field)) {
          if (document.getElementById(`${divId}-field-${encodeFieldKey(key)}`)) {
            sectionTotals.unansweredFields.push({ key, label: getLabel(field, lang) || key });
          } else {
            console.warn(`Field ${key} not rendered in DOM, skipping from unanswered list`);
          }
        }
      }

      const progressPercent = sectionTotals.requiredFields > 0
        ? Math.round((sectionTotals.answeredRequiredFields / sectionTotals.requiredFields) * 100)
        : 100;
      secDetails.innerHTML = `
        <summary style="cursor: pointer; margin-top: 10px; padding: 8px; background-color: rgba(200,190,200,.1); border-radius: 4px; font-size: 0.9em;">
          <span style="font-weight: bold;">${sectionLabel}</span>
        </summary>
        <div id="${divId}-section-header-${encodeFieldKey(sectionId)}" style="margin-bottom: 8px; padding: 8px; background-color: rgba(200,200,200,.1); ">
          
            [Progress: ${progressPercent}% (${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields})]
            <div style="width: 100%; height: 10px; border-radius: 5px; overflow: hidden;">
              <div style="width: ${progressPercent}%; background-color: #4caf50; height: 100%;"></div>
          
          </div>
        </div>
      `;

      const secDiv = document.createElement("div");
      secDiv.style.padding = "18px 0px";

      let hasField = false;
      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (field.section !== sectionId || !field.segment?.includes(segmentId)) continue;

        hasField = true;
        const fieldDiv = document.createElement("div");
        fieldDiv.className = "form-field";
        fieldDiv.id = `${divId}-field-${encodeFieldKey(key)}`;
        fieldDiv.style.cssText = "border-radius: 5px; background-color: rgba(255,255,255,.3); backdrop-filter: blur(10px); padding: 10px; margin-bottom: 10px; border-radius: 4px;";
        if (field.required && !isFieldAnswered(field)) {
          fieldDiv.style.border = "2px solid red";
        }

        const label = document.createElement("label");
        label.style.cssText = "margin: 10px 0px; box-shadow: 0 14px 12px rgba(0, 0, 0, 0.1); padding: 10px 5px; border-radius: 5px; background-color: rgba(255,255,255,.1); backdrop-filter: blur(5px); font-weight: 500; margin-right: 6px; display: block; padding-bottom: 5px;";
        if (field.required) label.classList.add("required-field");
        label.innerHTML = (getLabel(field, lang) || key) + (field.required ? "" : "");
        label.htmlFor = `${divId}-${encodeFieldKey(key)}`;
        if (field.score !== undefined || field.weights) {
          const scoreDisplay = document.createElement("span");
          scoreDisplay.className = "score-display";
          const maxScore = calculateMaxScore(field);
          const currentScore = calculateCurrentScore(field);
          if (typeof window.score !== "undefined" && window.score) {
            scoreDisplay.textContent = `(Score: ${currentScore}/${maxScore})`;
          }
          label.appendChild(scoreDisplay);
        }
        fieldDiv.appendChild(label);

        const input = createInputElement(divId, key, field, lang, schema);
        fieldDiv.appendChild(input);
        fieldDiv.appendChild(createResetButton(divId, key, field, input, lang, schema, fieldDiv));

        const scoreBar = document.createElement("div");
        scoreBar.className = "score-bar";
        scoreBar.style.cssText = "margin-top: 5px; font-size: 0.9em; color: #555;";
        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const scoreAchieved = calculateScoreAchieved(field);
        const weight = field.weights ? field.weights.join(", ") : "1";
        const fileCount = countFiles(field);
        let valueText = "";
        if (typeof field.value === "object" && field.value !== null) {
          try {
            valueText = JSON.stringify(field.value);
          } catch (e) {
            valueText = "[object]";
          }
        } else {
          valueText = field.value || "";
        }
        scoreBar.textContent = window.debug
          ? `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, score achieved: ${scoreAchieved.toFixed(1)}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`
          : `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
        fieldDiv.appendChild(scoreBar);

        secDiv.appendChild(fieldDiv);
      }

      if (hasField) {
        const sectionSummary = document.createElement("details");
        sectionSummary.className = "section-summary";
        sectionSummary.id = `${divId}-summary-${encodeFieldKey(segmentId)}-${encodeFieldKey(sectionId)}`;
        sectionSummary.style.cssText = "margin-top: 10px; padding: 8px; background-color: rgba(200,200,200,.7); border-radius: 4px; font-size: 0.9em;";
        sectionSummary.open = true;
        const unansweredCount = sectionTotals.requiredFields - sectionTotals.answeredRequiredFields;
        sectionSummary.innerHTML = `
          <summary style="cursor: pointer;">UnAnswered/Total/TotalRequired (${unansweredCount}/${sectionTotals.allFields}/${sectionTotals.requiredFields})</summary>
          <div style="padding: 8px;">
            [field total: ${sectionTotals.allFields}, score: ${sectionTotals.totalScore}, achieved score: ${sectionTotals.totalScoreWeighted}, maxscore: ${sectionTotals.maxPossibleScoreWeighted}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]
            ${sectionTotals.unansweredFields.length > 0 ? `
              <div style="margin-top: 10px; color: red;">
                <strong>Unanswered Required Fields:</strong>
                <ul style="margin: 5px 0; padding-left: 20px;">
                  ${sectionTotals.unansweredFields.map(field => `
                    <li style="max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                      <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${divId}')"
                         style="text-decoration: none; color: red;"
                         title="${field.label}">
                        ${field.label}
                      </a>
                    </li>
                  `).join("")}
                </ul>
              </div>
            ` : ""}
          </div>
        `;
        secDiv.appendChild(sectionSummary);
        segDiv.appendChild(secDetails);
        secDetails.appendChild(secDiv);
      }
    });

    segDetails.appendChild(segDiv);
    fieldsDiv.appendChild(segDetails);
  });

  // Handle schema copy button
  const copyBtn = document.getElementById(`${divId}-copy-btn`);
  copyBtn.addEventListener("click", () => {
    const inputEl = document.getElementById(`${divId}-input`);
    try {
      const newSchema = JSON.parse(inputEl.value);
      genform(divId, newSchema);
    } catch (e) {
      console.error(`Invalid JSON in ${divId}-input: ${e.message}`);
      alert(getLabel({ label: { en: `Invalid JSON: ${e.message}`, th: `JSON ไม่ถูกต้อง: ${e.message}` } }, lang));
    }
  });

  listenform(divId);

}

function listenform(divId) {
  const schema = document.getElementById(divId)._formData;
  const lang = window.lang || "en";

  for (const key in schema) {
    if (key === "grouping" || schema[key].type === "section") continue;
    const field = schema[key];
    const inputType = field.data_type || field.type;
    const inputId = `${divId}-${encodeFieldKey(key)}`;
    const fieldDiv = document.getElementById(`${divId}-field-${encodeFieldKey(key)}`);

    if (["string", "textarea", "text", "number"].includes(inputType)) {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("input", () => {
          schema[key].value = inputType === "number" ? parseFloat(inputEl.value) || 0 : inputEl.value.trim();
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "radio") {
      document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}"]`).forEach(el => {
        el.addEventListener("change", () => {
          schema[key].value = parseInt(el.value);
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      });
    } else if (inputType === "checkbox") {
      document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
        el.addEventListener("change", () => {
          schema[key].value = Array.from(document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]:checked`)).map(e => parseInt(e.value));
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      });
    } else if (inputType === "select") {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("change", () => {
          schema[key].value = inputEl.value;
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "attachment") {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("input", () => {
          schema[key].value = inputEl.value.trim();
          //document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = `Files: ${countFiles(field)}`;
           // Update file count display immediately
      const fileCount = countFiles(schema[key]);
      document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = `Files: ${fileCount}`;
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "radiotext" || inputType === "radiotextatt" || inputType === "radioatt") {
      document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
        el.addEventListener("change", () => {
          schema[key].value.radio = parseInt(el.value);
          // REMOVED ALL VISIBILITY TOGGLING
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      });
      const textInput = document.getElementById(`${divId}-${encodeFieldKey(key)}-text`);
      if (textInput) {
        textInput.addEventListener("input", () => {
          schema[key].value.text = textInput.value.trim();
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      }
      
      const attInput = document.getElementById(`${divId}-${encodeFieldKey(key)}-att`);
      if (attInput) {
        // ENSURE ATTACHMENT INPUT IS ALWAYS VISIBLE
        attInput.style.display = "block";
        attInput.addEventListener("input", () => {
          schema[key].value.att = attInput.value.trim();
          document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = `Files: ${countFiles(field)}`;
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "checktext" || inputType === "checktextatt") {
      document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
        el.addEventListener("change", () => {
          schema[key].value.check = Array.from(document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]:checked`)).map(e => parseInt(e.value));
          // REMOVED ALL VISIBILITY TOGGLING
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      });
      
      const textInput = document.getElementById(`${divId}-${encodeFieldKey(key)}-text`);
      if (textInput) {
        textInput.addEventListener("input", () => {
          schema[key].value.text = textInput.value.trim();
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      }
      
      const attInput = document.getElementById(`${divId}-${encodeFieldKey(key)}-att`);
      if (attInput) {
        // ENSURE ATTACHMENT INPUT IS ALWAYS VISIBLE
        attInput.style.display = "block";
        attInput.addEventListener("input", () => {
          schema[key].value.att = attInput.value.trim();
          document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = `Files: ${countFiles(field)}`;
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        });
      }
    }
  }

  if (document.getElementById("jumperhelper")) {
    const summary = QSummary(schema).summary;
    updateJumperHelperWithUnanswered(summary, schema);
  }
}


function isFieldAnswered(field) {
  if (!field) return false;
  const inputType = field.data_type || field.type;
  if (!inputType) return false;

  const isValidText = text => text && text.trim() !== "" && text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
  const isValidAttachment = field => field.value?.att && countFiles({ value: field.value.att, data_type: "attachment" }) > 0;

  switch (inputType) {
    case "checkbox":
      return Array.isArray(field.value) && field.value.length > 0;
    case "radio":
    case "select":
      return field.value !== null && field.value !== "";
    case "attachment":
    case "file":
      return field.value && countFiles(field) > 0;
    case "radiotext":
      return field.value?.radio !== null || isValidText(field.value?.text);
    case "radioatt":
      return field.value?.radio !== null || isValidAttachment(field);
    case "radiotextatt":
      return field.value?.radio !== null || isValidText(field.value?.text) || isValidAttachment(field);
    case "checktext":
      return Array.isArray(field.value?.check) && (field.value.check.length > 0 || isValidText(field.value?.text));
    case "checktextatt":
      return Array.isArray(field.value?.check) && (field.value.check.length > 0 || isValidText(field.value?.text) || isValidAttachment(field));
    default:
      return field.value !== null && field.value !== "" && field.value !== undefined;
  }
}

function createInputElement(divId, key, field, lang, schema) {
  const inputType = field.data_type || field.type;
  let input;

  if (inputType === "textarea") {
    input = document.createElement("textarea");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.value = field.value || "";
    input.style.width = "260px";
  } else if (inputType === "radioatt") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    const radioContainer = document.createElement("div");
    radioContainer.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText = "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}-radio" value="${val}" ${field.value && field.value.radio === val ? "checked" : ""}> ${opt}`;
      radioContainer.appendChild(radioLabel);
    });
    input.appendChild(radioContainer);
    const attDiv = document.createElement("div");
    attDiv.style.marginTop = "5px";
    const attInput = document.createElement("input");
    attInput.type = "text";
    attInput.id = `${divId}-${encodeFieldKey(key)}-att`;
    attInput.value = (field.value && field.value.att) || "";
    attInput.placeholder = getLabel({ label: { en: "Enter file name(s)", th: "ป้อนชื่อไฟล์" } }, lang);
    attInput.style.width = "260px";
    attInput.style.display = field.value?.radio === 0 ? "block" : "none";
    attDiv.appendChild(attInput);
    const fileCount = document.createElement("div");
    fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
    fileCount.style.marginTop = "5px";
    fileCount.textContent = `Files: ${countFiles(field)}`;
    input.appendChild(fileCount);
    attDiv.appendChild(createAddFileButton(divId, key, field, schema, lang));
    attDiv.appendChild(createSaveAttachmentButton(divId, key, field, schema, lang));
    input.appendChild(attDiv);
  } else if (inputType === "radio") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText = "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}" value="${val}" ${field.value === val ? "checked" : ""}> ${opt}`;
      input.appendChild(radioLabel);
    });
  } else if (inputType === "checkbox") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const checkLabel = document.createElement("label");
      checkLabel.style.cssText = "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      checkLabel.innerHTML = `<input type="checkbox" name="${divId}-${encodeFieldKey(key)}[]" value="${val}" ${field.value && field.value.includes(val) ? "checked" : ""}> ${opt}`;
      input.appendChild(checkLabel);
    });
  } else if (inputType === "select") {
    input = document.createElement("select");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.style.width = "260px";
    getOptions(field, lang).forEach((opt, idx) => {
      const option = document.createElement("option");
      const value = idx === 0 ? "" : opt;
      option.value = value;
      option.textContent = opt;
      option.selected = field.value === value;
      input.appendChild(option);
    });
 } else if (inputType === "attachment") {
  input = document.createElement("div");
  const textInput = document.createElement("input");
  textInput.type = "text";
  textInput.id = `${divId}-${encodeFieldKey(key)}`;  // Consistent ID without -att suffix
  textInput.value = field.value || "";
  textInput.placeholder = getLabel({ label: { en: "Enter file names (comma-separated)", th: "Enter file names (comma-separated,คั่นด้วยเครื่องหมายจุลภาค)" } }, lang);
  textInput.style.width = "100%";
  input.appendChild(textInput);

  const fileCount = document.createElement("div");
  fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
  fileCount.style.marginTop = "5px";
  fileCount.textContent = `Files: ${countFiles(field)}`;
  input.appendChild(fileCount);

  const buttonDiv = document.createElement("div");
  buttonDiv.style.marginTop = "5px";
  buttonDiv.appendChild(createAddFileButton(divId, key, field, schema, lang));
  buttonDiv.appendChild(createSaveAttachmentButton(divId, key, field, schema, lang));
  input.appendChild(buttonDiv);
} else if (inputType === "radiotext" || inputType === "radiotextatt") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    const radioContainer = document.createElement("div");
    radioContainer.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText = "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}-radio" value="${val}" ${field.value && field.value.radio === val ? "checked" : ""}> ${opt}`;
      radioContainer.appendChild(radioLabel);
    });
    input.appendChild(radioContainer);
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.id = `${divId}-${encodeFieldKey(key)}-text`;
    textInput.value = (field.value && field.value.text) || "";
    textInput.placeholder = getLabel({ label: { en: "Enter details (e.g., Others)", th: "ป้อนรายละเอียด (เช่น อื่น ๆ)" } }, lang);
    textInput.style.width = "260px";
    textInput.style.marginTop = "5px";
    textInput.style.display = field.value?.radio === getOptions(field, lang).length - 1 ? "block" : "none";
    input.appendChild(textInput);
    if (inputType === "radiotextatt") {
      const attDiv = document.createElement("div");
      attDiv.style.marginTop = "5px";
      const attInput = document.createElement("input");
      attInput.type = "text";
      attInput.id = `${divId}-${encodeFieldKey(key)}-att`;
      attInput.value = (field.value && field.value.att) || "";
      attInput.placeholder = getLabel({ label: { en: "Enter file name(s)", th: "ป้อนชื่อไฟล์" } }, lang);
      attInput.style.width = "260px";
      attDiv.appendChild(attInput);
      const fileCount = document.createElement("div");
      fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
      fileCount.style.marginTop = "5px";
      fileCount.textContent = `Files: ${countFiles(field)}`;
      input.appendChild(fileCount);
      attDiv.appendChild(createAddFileButton(divId, key, field, schema, lang));
      attDiv.appendChild(createSaveAttachmentButton(divId, key, field, schema, lang));
      input.appendChild(attDiv);
    }
  } else if (inputType === "checktext" || inputType === "checktextatt") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    const checkContainer = document.createElement("div");
    checkContainer.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const checkLabel = document.createElement("label");
      checkLabel.style.cssText = "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      checkLabel.innerHTML = `<input type="checkbox" name="${divId}-${encodeFieldKey(key)}[]" value="${val}" ${field.value && field.value.check.includes(val) ? "checked" : ""}> ${opt}`;
      checkContainer.appendChild(checkLabel);
    });
    input.appendChild(checkContainer);
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.id = `${divId}-${encodeFieldKey(key)}-text`;
    textInput.value = (field.value && field.value.text) || "";
    textInput.placeholder = getLabel({ label: { en: "Enter details (e.g., Others)", th: "ป้อนรายละเอียด (เช่น อื่น ๆ)" } }, lang);
    textInput.style.width = "260px";
    textInput.style.marginTop = "5px";
    textInput.style.display = field.value?.check.includes(getOptions(field, lang).length - 1) ? "block" : "none";
    input.appendChild(textInput);
    if (inputType === "checktextatt") {
      const attDiv = document.createElement("div");
      attDiv.style.marginTop = "5px";
      const attInput = document.createElement("input");
      attInput.type = "text";
      attInput.id = `${divId}-${encodeFieldKey(key)}-att`;
      attInput.value = (field.value && field.value.att) || "";
      attInput.placeholder = getLabel({ label: { en: "Enter file name(s)", th: "ป้อนชื่อไฟล์" } }, lang);
      attInput.style.width = "260px";
      attDiv.appendChild(attInput);
      const fileCount = document.createElement("div");
      fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
      fileCount.style.marginTop = "5px";
      fileCount.textContent = `Files: ${countFiles(field)}`;
      input.appendChild(fileCount);
      attDiv.appendChild(createAddFileButton(divId, key, field, schema, lang));
      attDiv.appendChild(createSaveAttachmentButton(divId, key, field, schema, lang));
      input.appendChild(attDiv);
    }
  } else {
    input = document.createElement("input");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.type = inputType === "string" ? "text" : inputType || "text";
    input.value = field.value || (inputType === "number" ? 0 : "");
    input.style.width = "260px";
  }

  return input;
}

function createAddFileButton_01(divId, key, field, schema, lang) {
  const addFileBtn = document.createElement("button");
  addFileBtn.textContent = getLabel({ label: { en: "Add file", th: "เพิ่มไฟล์" } }, lang);
  addFileBtn.style.cssText = "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
  addFileBtn.onclick = () => {
    window.logomode = false;
    toggleUploadPanel();
    updateFileListFromInput(`${divId}-${encodeFieldKey(key)}-att`);
    const count = countFiles(schema[key]);
    document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = `Files: ${count}`;
    schema[key].value.att = document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value;
    const fieldDiv = document.getElementById(`${divId}-field-${encodeFieldKey(key)}`);
    if (fieldDiv) {
      updateFieldDisplay(fieldDiv, field);
      updateSectionSummaries(divId, schema, lang);
    } else {
      console.warn("fieldDiv not found for field:", key);
    }
  };
  return addFileBtn;
}
function createAddFileButton_01(divId, key, field, schema, lang) {
  const addFileBtn = document.createElement("button");
  addFileBtn.textContent = getLabel({ label: { en: "Add file", th: "Add file" } }, lang);
  addFileBtn.style.cssText = "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
  
  addFileBtn.onclick = () => {
    window.logomode = false;
    toggleUploadPanel();
    
    updateFileListFromInput(`${divId}-${encodeFieldKey(key)}-att`);
    
    // Set up a completion callback
    window.uploadCompleteCallback = () => {
      const attInput = document.getElementById(`${divId}-${encodeFieldKey(key)}-att`);
      if (attInput) {
        // Trigger full update sequence
        const inputEvent = new Event('input', { bubbles: true });
        attInput.dispatchEvent(inputEvent);
        
        // Direct schema update
        if (field.data_type.includes("att")) {
          field.value.att = attInput.value.trim();
        } else {
          field.value = attInput.value.trim();
        }
        
        // Update displays
        const fieldDiv = document.getElementById(`${divId}-field-${encodeFieldKey(key)}`);
        if (fieldDiv) {
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
        }
      }
    };
  };
  
  return addFileBtn;
}
function createAddFileButton(divId, key, field, schema, lang) {
  const addFileBtn = document.createElement("button");
  addFileBtn.textContent = getLabel({ label: { en: "Add file", th: "เพิ่มไฟล์" } }, lang);
  addFileBtn.style.cssText = "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
  
  addFileBtn.onclick = () => {
    window.logomode = false;
    toggleUploadPanel();
    
    // Determine the correct input ID based on field type
    const inputId = field.data_type === "attachment" 
      ? `${divId}-${encodeFieldKey(key)}`  // For simple attachment fields
      : `${divId}-${encodeFieldKey(key)}-att`;  // For radioatt/checktextatt
    
    updateFileListFromInput(inputId);
    
    // Set up a completion callback
    window.uploadCompleteCallback = () => {
      const attInput = document.getElementById(inputId);
      if (attInput) {
        // Trigger full update sequence
        const inputEvent = new Event('input', { bubbles: true });
        attInput.dispatchEvent(inputEvent);
        
        // Update the correct field value structure
        if (field.data_type === "attachment") {
          field.value = attInput.value.trim();
        } else if (field.data_type.includes("att")) {
          field.value.att = attInput.value.trim();
        }
        
        // Update displays
        const fieldDiv = document.getElementById(`${divId}-field-${encodeFieldKey(key)}`);
        if (fieldDiv) {
          updateFieldDisplay(fieldDiv, field);
          updateSectionSummaries(divId, schema, lang);
          
          // Update file count display
          const fileCountDisplay = fieldDiv.querySelector(`#${divId}-filecount-${encodeFieldKey(key)}`);
          if (fileCountDisplay) {
            fileCountDisplay.textContent = `Files: ${countFiles(field)}`;
          }
        }
      }
    };
  };
  
  return addFileBtn;
}
function createSaveAttachmentButton(divId, key, field, schema, lang) {
  const saveAttBtn = document.createElement("button");
  saveAttBtn.textContent = getLabel({ label: { en: "Save Attachment", th: "บันทึกไฟล์แนบ" } }, lang);
  saveAttBtn.style.cssText = "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
  saveAttBtn.onclick = () => {
    schema[key].value.att = document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value;
    document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = `Files: ${countFiles(schema[key])}`;
    const fieldDiv = document.getElementById(`${divId}-field-${encodeFieldKey(key)}`);
    if (fieldDiv) {
      updateFieldDisplay(fieldDiv, field);
      updateSectionSummaries(divId, schema, lang);
    } else {
      console.warn("fieldDiv not found for field:", key);
    }
  };
  return saveAttBtn;
}

// Helper functions
function getOptions(field, lang) {
  if (Array.isArray(field.options)) {
    return field.options;
  }
  return field.options?.[lang] || field.options?.en || [];
}

function getLabel(field, lang) {
  return field.label?.[lang] || field.label?.en || "";
}

function encodeFieldKey(key) {
  return encodeURIComponent(key).replace(/'/g, "'");
}

function countFiles(field) {
  if (!field || !field.value) return 0;
  if (field.data_type === "attachment" && typeof field.value === "string") {
    return field.value.split(",").filter(f => f.trim()).length;
  }
  if (["radiotextatt", "radioatt", "checktextatt"].includes(field.data_type) && field.value?.att && typeof field.value.att === "string") {
    return field.value.att.split(",").filter(f => f.trim()).length;
  }
  return 0;
}

function calculateCurrentScore(field) {
  if (!isFieldAnswered(field)) return 0;
  if (field.weights) {
    if (["radio", "radioatt", "radiotext", "radiotextatt"].includes(field.data_type)) {
      const radioValue = field.data_type.includes("radio") && field.value?.radio !== undefined ? field.value.radio : field.value;
      return field.weights[radioValue] || 0;
    }
    if (["checkbox", "checktext", "checktextatt"].includes(field.data_type)) {
      return field.value?.check?.reduce((sum, idx) => sum + (field.weights[idx] || 0), 0) || 0;
    }
  }
  return field.score || (isFieldAnswered(field) ? 1 : 0);
}

function calculateMaxScore(field) {
  if (field.weights) {
    if (["radio", "radioatt", "radiotext", "radiotextatt"].includes(field.data_type)) {
      return Math.max(...field.weights);
    }
    if (["checkbox", "checktext", "checktextatt"].includes(field.data_type)) {
      return field.weights.filter(w => w > 0).reduce((sum, w) => sum + w, 0);
    }
  }
  return field.score || 1;
}

function calculateScoreAchieved(field) {
  const currentScore = calculateCurrentScore(field);
  return (field.score || 1) * currentScore;
}