/**
 * Saves board data to the server with user ID, timestamp, and status
 * @param {string} userId - The user ID to associate with the board data
 * @param {object} boardData - The board data to save
 * @returns {Promise<boolean>} - Returns true if save was successful, false otherwise
 */
async function saveBoardData(userId, boardData) {
    // Early exit checks
    if (!navigator.onLine) {
        console.warn("⚠️ Offline — cannot save to server");
        toast(
            "⚠️ Offline - cannot save",
            { pos: "TR", slide: "ttb" },
            { background: "orange" },
        );
        return false;
    }

    if (!userId) {
        console.warn("⚠️ User ID is required");
        toast(
            "⚠️ User ID is required",
            { pos: "TR", slide: "ttb" },
            { background: "orange" },
        );
        return false;
    }

    if (!boardData || Object.keys(boardData).length === 0) {
        toast(
            "⚠️ No board data to save",
            { pos: "TR", slide: "rtl" },
            { background: "red" },
        );
        return false;
    }

    // Create progress panel
    createPanel("boardSaveProgressPanel", "TR", "ctc", 60, 220, {
        title: "SAVING BOARD",
        verticalProgressBar: 100,
        noHeader: true,
        progressBar: 100,
        noFooter: true,
        bodyContent: `
        <div class="progress-container" style="display: flex; flex-direction: column; align-items: center;">
            <span class="progress-label">Saving board...</span>
            <div class="progress-bar vertical">
                <div class="progress-fill vertical" style="height: 0%"></div>
            </div>
        </div>`,
    });

    setTimeout(() => {
        toast(
            "Saving board data...",
            { pos: "TR", slide: "rtl" },
            { background: "red", width: 200 },
            1000,
        );
        moveXY("boardSaveProgressPanel", window.innerWidth - 100, 80);
    }, 50);

    const panel = document.getElementById("boardSaveProgressPanel");
    panel.style.left = window.innerWidth - 70 + "px";
    panel.style.top = "10px";

    try {
        const startTime = Date.now();
        const timestamp = new Date().toISOString();

        // Prepare payload with board data and status
        const payload = {
            data: {
                id: userId,
                name: userId,
                board: JSON.stringify({
                    data: boardData,
                    status: "saved",
                    timestamp: timestamp
                }),
                timestamp: timestamp,
            },
        };

        // Get cached ETag if available
        const getUrl = `https://j.kunok.com/api622.php?id=${encodeURIComponent(userId)}`;
        const cachedETag = sessionStorage.getItem(`etag:${getUrl}`);
        const headers = {
            "Content-Type": "application/json",
            ...(cachedETag && { "If-Match": cachedETag }),
        };

        // Show saving progress
        updateProgress(panel, 30, "Uploading board data...");

        const response = await fetch("https://j.kunok.com/api622.php", {
            method: "POST",
            headers: headers,
            body: JSON.stringify(payload),
        });

        if (!response.ok) {
            const text = await response.text();
            if (response.status === 412) {
                throw new Error(
                    "Board was modified by another user. Please reload and try again.",
                );
            }
            throw new Error(`HTTP ${response.status}: ${text}`);
        }

        // Update progress
        updateProgress(panel, 70, "Verifying save...");

        // Update ETag if available
        const etag = response.headers.get("ETag");
        if (etag) {
            sessionStorage.setItem(`etag:${getUrl}`, etag);
        }

        // Verify the save by fetching the data back
        const verifyResponse = await fetch(getUrl);
        if (!verifyResponse.ok) {
            throw new Error("Failed to verify saved board data");
        }

        const verifiedData = await verifyResponse.json();
        if (!verifiedData.data) {
            throw new Error("No data in verification response");
        }

        // Compare the saved data with what we sent
        const isDataValid = compareSavedData(
            payload.data,
            verifiedData.data,
        );
        if (!isDataValid) {
            throw new Error("Board data verification failed");
        }

        // Update progress to complete
        updateProgress(panel, 100, "Board saved successfully!");

        console.log(`✅ Board save completed in ${Date.now() - startTime}ms`);
        toast(
            "✅ Board saved successfully",
            { pos: "TR", slide: "ttb" },
            { background: "green" },
        );

        return true;
    } catch (e) {
        console.error("❌ Board save failed:", e.message);
        toast(
            `❌ Board save failed: ${e.message}`,
            { pos: "TR", slide: "ttb" },
            { background: "red" },
        );
        return false;
    } finally {
        // Close progress panel after a delay
        setTimeout(() => closePanel("boardSaveProgressPanel"), 1500);
    }
}

// Helper function to compare saved data (assuming this exists)
function compareSavedData(original, saved) {
    // Implement your comparison logic here
    return JSON.stringify(original) === JSON.stringify(saved);
}