(function () {
  function createDraggablePanelUp(options = {}) {
    const {
      top = 100,
      left = 10,
      width = 200,
      collapsedHeight = 70,
      expandedHeight = 200,
      expandedWidth = width,
      avatars = [],
      showExternalAvatar = true,
      externalAvatarURL = 'https://i.pravatar.cc/41',
      expandedContent = '',
      toggleOnClick = false
    } = options;

    const panelId = 'miniPanel_' + Math.random().toString(36).substr(2, 5);
    const panel = document.createElement('div');
    panel.id = panelId;

    panel.style.cssText = `
      position: absolute;
      top: ${top}px;
      left: ${left}px;
      width: ${width}px;
      height: ${collapsedHeight}px;
      background: #f0f0f0;
      border: 1px solid #aaa;
      border-radius: 0 0 8px 8px;
      padding: 0;
      transition: height 0.3s, width 0.3s, top 0.3s;
      display: flex;
      flex-direction: column-reverse;
      box-sizing: border-box;
      z-index: 1000;
      overflow: hidden;
      font-family: sans-serif;
    `;

    // Expanded content area (above control row)
    const expandArea = document.createElement('div');
    expandArea.className = 'panel-expanded-content';
    expandArea.style.cssText = `
      display: none;
      flex: 1;
      padding: 10px;
      overflow: auto;
      background: #fff;
    `;
    expandArea.innerHTML = expandedContent;

    // Bottom control row (avatars + button)
    const controlRow = document.createElement('div');
    controlRow.style.cssText = `
      display: flex;
      flex-direction: row;
      align-items: center;
      gap: 10px;
      padding: 10px;
      background: #f0f0f0;
      border-top: 1px solid #ccc;
    `;

    // Avatars
    avatars.forEach((url, i) => {
      const avatar = document.createElement('div');
      avatar.className = 'avatar';
      avatar.title = `Avatar ${i + 1}`;
      avatar.style.cssText = `
        width: 40px;
        height: 40px;
        flex-shrink: 0;
        background: url('${url}') no-repeat center/cover;
        border-radius: 50%;
        cursor: grab;
      `;
      avatar.addEventListener('mousedown', (e) => {
        panel.dispatchEvent(new MouseEvent('mousedown', e));
      });
      controlRow.appendChild(avatar);
    });

    // Expand button
    if (!toggleOnClick) {
      const btn = document.createElement('button');
      btn.innerText = 'Expand';
      btn.className = 'toggle-btn';
      btn.style.cssText = `
        margin-left: auto;
        cursor: pointer;
        background: #ccc;
        padding: 5px 10px;
        border: none;
        border-radius: 5px;
      `;
      btn.onclick = togglePanel;
      controlRow.appendChild(btn);
    }

    // Combine elements
    panel.appendChild(controlRow);
    panel.appendChild(expandArea);
    document.body.appendChild(panel);

    // Expand/collapse logic
    function togglePanel() {
      const isExpanded = panel.classList.contains('expanded');

      if (!isExpanded) {
        const newTop = parseInt(panel.style.top) - (expandedHeight - collapsedHeight);
        if (newTop < 0) {
          alert("Will over top of screen!");
        } else {
          panel.style.top = newTop + 'px';
        }
        panel.style.height = expandedHeight + 'px';
        panel.style.width = expandedWidth + 'px';
        expandArea.style.display = 'block';
      } else {
        panel.style.top = top + 'px';
        panel.style.height = collapsedHeight + 'px';
        panel.style.width = width + 'px';
        expandArea.style.display = 'none';
      }

      panel.classList.toggle('expanded');
    }

    if (toggleOnClick) {
      panel.addEventListener('click', (e) => {
        if (e.target.tagName !== 'BUTTON') togglePanel();
      });
    }

    // Dragging (top anchor)
    makeDraggableTopAnchor(panel);

    // External avatar
    if (showExternalAvatar) {
      const ext = document.createElement('div');
      ext.className = 'external-avatar';
      ext.style.cssText = `
        position: absolute;
        left: ${left + width + 10}px;
        top: ${top}px;
        width: 40px;
        height: 40px;
        background: url('${externalAvatarURL}') no-repeat center/cover;
        border-radius: 50%;
        cursor: pointer;
      `;
      ext.title = 'Double-click me!';
      ext.ondblclick = () => alert('hi');
      makeDraggableTopAnchor(ext);
      document.body.appendChild(ext);
    }

    return panelId;
  }

  function makeDraggableTopAnchor(el) {
    let startX = 0, startY = 0, startLeft = 0, startTop = 0, isDragging = false;

    el.addEventListener('mousedown', (e) => {
      if (e.target.tagName === 'BUTTON') return;
      isDragging = true;
      startX = e.clientX;
      startY = e.clientY;
      const style = window.getComputedStyle(el);
      startLeft = parseInt(style.left, 10);
      startTop = parseInt(style.top, 10);
      el.style.cursor = 'grabbing';
      e.preventDefault();
    });

    document.addEventListener('mousemove', (e) => {
      if (isDragging) {
        const dx = e.clientX - startX;
        const dy = e.clientY - startY;
        el.style.left = (startLeft + dx) + 'px';
        el.style.top = (startTop + dy) + 'px';
      }
    });

    document.addEventListener('mouseup', () => {
      isDragging = false;
      el.style.cursor = 'grab';
    });
  }

  // Global
  window.createDraggablePanelUp = createDraggablePanelUp;
})();
