function updateTooltips(elementId, text, position = 'top') {
  const element = document.getElementById(elementId);
  if (!element) return;

  // Find or create tooltip
  let tooltip = element.querySelector('.custom-tooltip');
  if (!tooltip) {
    tooltip = document.createElement('div');
    tooltip.className = 'custom-tooltip';
    
    // Create content element
    const content = document.createElement('div');
    content.className = 'tooltip-content';
    tooltip.appendChild(content);
    
    // Create arrow
    const arrow = document.createElement('div');
    arrow.className = 'tooltip-arrow';
    tooltip.appendChild(arrow);
    
    element.appendChild(tooltip);
    
    // Add hover events
    element.addEventListener('mouseenter', () => showTooltip(element));
    element.addEventListener('mouseleave', () => hideTooltip(element));
  }

  // Update content
  const content = tooltip.querySelector('.tooltip-content');
  content.textContent = text;
  tooltip.dataset.position = position;

  // Position immediately if hovered
  if (element.matches(':hover')) showTooltip(element);
}

function showTooltip(element) {
  const tooltip = element.querySelector('.custom-tooltip');
  if (!tooltip) return;
  
  positionTooltip(element, tooltip);
  tooltip.style.visibility = 'visible';
  tooltip.style.opacity = '1';
}

function positionTooltip(element, tooltip) {
  const elementRect = element.getBoundingClientRect();
  const tooltipRect = tooltip.getBoundingClientRect();
  const position = tooltip.dataset.position || 'top';
  const arrow = tooltip.querySelector('.tooltip-arrow');
  
  // Reset positions
  tooltip.style.top = '';
  tooltip.style.bottom = '';
  tooltip.style.left = '';
  tooltip.style.right = '';
  
  // Calculate scroll offset
  const scrollY = window.scrollY || window.pageYOffset;
  const scrollX = window.scrollX || window.pageXOffset;

  // Position tooltip
  switch (position) {
    case 'top':
      tooltip.style.top = `${elementRect.top + scrollY - tooltipRect.height - 10}px`;
      tooltip.style.left = `${elementRect.left + scrollX + (elementRect.width - tooltipRect.width) / 2}px`;
      positionArrow(arrow, 'bottom');
      break;
      
    case 'bottom':
      tooltip.style.top = `${elementRect.bottom + scrollY + 10}px`;
      tooltip.style.left = `${elementRect.left + scrollX + (elementRect.width - tooltipRect.width) / 2}px`;
      positionArrow(arrow, 'top');
      break;
      
    case 'left':
      tooltip.style.top = `${elementRect.top + scrollY + (elementRect.height - tooltipRect.height) / 2}px`;
      tooltip.style.left = `${elementRect.left + scrollX - tooltipRect.width - 10}px`;
      positionArrow(arrow, 'right');
      break;
      
    case 'right':
      tooltip.style.top = `${elementRect.top + scrollY + (elementRect.height - tooltipRect.height) / 2}px`;
      tooltip.style.left = `${elementRect.right + scrollX + 10}px`;
      positionArrow(arrow, 'left');
      break;
  }
  
  // Adjust if going off-screen
  adjustForViewport(tooltip);
}

function adjustForViewport(tooltip) {
  const rect = tooltip.getBoundingClientRect();
  
  // Check right edge
  if (rect.right > window.innerWidth) {
    tooltip.style.left = `${parseInt(tooltip.style.left) - (rect.right - window.innerWidth)}px`;
  }
  
  // Check left edge
  if (rect.left < 0) {
    tooltip.style.left = '10px';
  }
  
  // Check bottom edge
  if (rect.bottom > window.innerHeight) {
    tooltip.style.top = `${parseInt(tooltip.style.top) - (rect.bottom - window.innerHeight)}px`;
  }
  
  // Check top edge
  if (rect.top < 0) {
    tooltip.style.top = '10px';
  }
}

function positionArrow(arrow, direction) {
  arrow.style.cssText = `
    position: absolute;
    width: 8px;
    height: 8px;
    background: #333;
    transform: rotate(45deg);
  `;
  
  switch (direction) {
    case 'top':
      arrow.style.bottom = '-4px';
      arrow.style.left = '50%';
      arrow.style.marginLeft = '-4px';
      break;
    case 'bottom':
      arrow.style.top = '-4px';
      arrow.style.left = '50%';
      arrow.style.marginLeft = '-4px';
      break;
    case 'left':
      arrow.style.right = '-4px';
      arrow.style.top = '50%';
      arrow.style.marginTop = '-4px';
      break;
    case 'right':
      arrow.style.left = '-4px';
      arrow.style.top = '50%';
      arrow.style.marginTop = '-4px';
      break;
  }
}

function hideTooltip(element) {
  const tooltip = element.querySelector('.custom-tooltip');
  if (tooltip) {
    tooltip.style.visibility = 'hidden';
    tooltip.style.opacity = '0';
  }
}
// CSS for tooltips (should be in your stylesheet)
const tooltipCSS = `
.custom-tooltip {
  position: absolute;
  visibility: hidden;
  background: #333;
  color: white;
  padding: 8px 12px;
  border-radius: 4px;
  font-size: 14px;
  z-index: 1000;
  max-width: 200px;
  opacity: 0;
  transition: opacity 0.2s;
}
.custom-tooltip .tooltip-content {
  white-space: nowrap;
}
.custom-tooltip .tooltip-arrow {
  position: absolute;
  width: 8px;
  height: 8px;
  background: #333;
  transform: rotate(45deg);
}
.custom-tooltip.visible {
  visibility: visible;
  opacity: 1;
}
`;

// Add CSS to head
if (!document.querySelector('#tooltip-styles')) {
  const style = document.createElement('style');
  style.id = 'tooltip-styles';
  style.textContent = tooltipCSS;
  document.head.appendChild(style);
}