async function updateItemPanel(itemId = "") {
    const API = "https://j.kunok.com/api62.php";
    const defaultBoard = JSON.stringify({ user: "", note: "", data: {} }, null, 2);

    function renderForm(item) {
        return `
        <style>
            .form-row { display: flex; gap: 10px; }
            .form-row > div { flex: 1; }
            textarea {
                width: 100%;
                min-height: 100px;
                font-family: monospace;
                font-size: 13px;
                padding: 6px;
                resize: vertical;
            }
        </style>
        <form id="updateItemForm">
            <div><label for="itemId">ID:</label> <input type="text" id="itemId" value="${item.id}" disabled></div>
            <div><label for="itemName">Name:</label> <input type="text" id="itemName" value="${item.name || ''}" required></div>
            <div class="form-row">
                <div>
                    <label for="itemFlag">Flag:</label>
                    <select id="itemFlag">
                        <option value="1" ${item.flag == 1 ? "selected" : ""}>1 (Active)</option>
                        <option value="0" ${item.flag == 0 ? "selected" : ""}>0 (Hidden)</option>
                        <option value="3" ${item.flag == 3 ? "selected" : ""}>3 (Deleted)</option>
                    </select>
                </div>
                <div>
                    <label for="itemKeeper">Keeper:</label>
                    <input type="text" id="itemKeeper" value="${item.keeper || ''}">
                </div>
            </div>
            <div class="form-row">
                <div><label for="itemBucket">Bucket:</label> <input type="text" id="itemBucket" value="${item.bucket || ''}"></div>
                <div><label for="itemCup">Cup:</label> <input type="text" id="itemCup" value="${item.cup || ''}"></div>
            </div>
            <div>
                <label for="itemBoard">Board (JSON):</label>
                <textarea id="itemBoard">${item.board || defaultBoard}</textarea>
            </div>
            <button type="submit">Update Item</button>
        </form>
        <div id="itemMessage" class="message"></div>
        `;
    }

    function displayMessage(msg, isError = false) {
        const box = document.getElementById("itemMessage");
        if (box) {
            box.textContent = msg;
            box.style.color = isError ? "red" : "green";
        }
    }

    function closeAfterDelay(panelId, delay = 2000) {
        setTimeout(() => {
            closePanel(panelId);
            window.location.href = "index.html";
        }, delay);
    }

    async function fetchItemData(id) {
        const res = await fetch(`${API}?id=${encodeURIComponent(id)}`);
        const json = await res.json();
        if (!json.data) throw new Error("Item not found");
        return json.data;
    }

    if (!itemId) {
        const urlParams = new URLSearchParams(window.location.search);
        itemId = urlParams.get("id");
    }

    if (!itemId) {
        createPanel("updateItemPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="message error">No item ID provided</div>`,
            noFooter: true,
            title: "Update Item"
        });
        return;
    }

    try {
        const item = await fetchItemData(itemId);

        createPanel("updateItemPanel", "CC", "ctc", 500, 600, {
            bodyContent: renderForm(item),
            noFooter: true,
            title: "Update Item"
        });

        const form = document.getElementById("updateItemForm");
        if (form) {
            form.addEventListener("submit", async (e) => {
                e.preventDefault();

                const payload = {
                    id: item.id,
                    name: document.getElementById("itemName").value.trim(),
                    flag: parseInt(document.getElementById("itemFlag").value),
                    bucket: document.getElementById("itemBucket").value.trim(),
                    cup: document.getElementById("itemCup").value.trim(),
                    board: document.getElementById("itemBoard").value.trim(),
                    keeper: document.getElementById("itemKeeper").value.trim()
                };

                try {
                    const res = await fetch(API, {
                        method: "POST",
                        headers: { "Content-Type": "application/json" },
                        body: JSON.stringify({ data: payload })
                    });
                    const result = await res.json();
                    if (!res.ok || result.error) throw new Error(result.error || "Unknown error");

                    displayMessage(result.message || "Item updated");
                    closeAfterDelay("updateItemPanel", 2000);
                } catch (err) {
                    displayMessage("Error: " + err.message, true);
                }
            });
        }
    } catch (err) {
        console.error("Failed to load item:", err);
        createPanel("updateItemPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="message error">${err.message}</div>`,
            noFooter: true,
            title: "Update Item"
        });
    }
}
