addStyle(`
      body {
        margin: 0;
        font-family: sans-serif;
        overflow: hidden;
      }
      div {
        margin: 0px;
      }
      .panel {
        position: absolute;
        background: rgba(95, 95, 95, 0.1);
        backdrop-filter: blur(15px);
        border: 1px solid rgba(200, 200, 200, 0.5);
        box-shadow: 2px 2px 12px rgba(0, 0, 0, 0.2);
        display: flex;
        flex-direction: column;
        transition: transform 0.3s ease, left 0.3s ease, top 0.3s ease, opacity 0.3s ease;
        opacity: 1;
        z-index: 100;
      }
      .panel-header,
      .panel-footer {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 5px 10px;
        user-select: none;
        background: rgba(195, 195, 195, 0.5);
        backdrop-filter: blur(15px);
      }
      .panel-body {
        flex: 1;
        padding: 10px;
        position: relative;
        overflow: auto;
      }
      .header-title,
      .footer-title {
        font-weight: bold;
      }
      .header-buttons,
      .footer-buttons {
        display: flex;
        gap: 5px;
      }
      .close-btn {
        position: absolute;
        top: 2px;
        right: 5px;
        color: white;
        border: none;
        font-size: 18px;
        background: red;
        cursor: pointer;
        z-index: 10;
      }
      .body-close-btn {
        position: absolute;
        top: 10px;
        right: 10px;
        color: grey;
        background: none;
        border: none;
        font-size: 18px;
        cursor: pointer;
      }
      .panel-footer {
        border-top: 1px solid #ccc;
      }
      .footer-center {
        justify-content: center;
      }
      .panel.toast {
        box-sizing: border-box;
        border-radius: 8px;
        display: inline-block;
        border: none !important;
        box-shadow: none !important;
        outline: none !important;
        padding: 20px;
        background: transparent !important;
        backdrop-filter: none !important;
      }
      .panel.toast .panel-body {
        background: red;
        color: #fff;
        padding: 10px;
        border-radius: 8px;
        width: 100%;
        height: 100%;
        box-sizing: border-box;
        border: 0px;
        box-shadow: 0 14px 12px rgba(0, 0, 0, 0.4);
        overflow: hidden;
      }
      .panel.toast .close-btn {
        width: 24px;
        height: 24px;
        border: 3px solid rgba(200, 200, 200, 0.3);
        border-radius: 50%;
        position: absolute;
        top: 4px;
        right: 4px;
        font-size: 16px;
        line-height: 20px;
        background: none;
        color: grey;
      }
      .panel.toast .close-btn:hover {
        border: 5px solid transparent;
        background: white;
        color: #ff0000;
      }
      .panel .close-btn {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 24px;
        height: 24px;
        border: 2px solid transparent;
        border-radius: 50%;
        background: rgba(200, 200, 200, 0.2);
        cursor: pointer;
        font-size: 16px;
        font-weight: 700;
        color: #ddd;
        line-height: 24px;
        text-align: center;
        box-sizing: border-box;
        padding: 0;
        margin: 0;
      }
      .panel.toasts {
        background-color: rgba(100, 200, 2, 1);
        border-radius: 10px;
      }
      .panel .close-btn:hover {
        border: none;
        background: red;
        color: white;
      }
      .panel.toast .body-close-btn:hover {
        color: white;
      }
      .footer-buttons button.close-footer-btn {
        background: rgba(100, 100, 100, 0.2);
        border: none;
        border-radius: 6px;
      }
      .footer-buttons button.close-footer-btn:hover {
        background: rgba(100, 100, 100, 0.5);
      }
      @media (prefers-color-scheme: dark) {
        body {
          background: rgba(45, 95, 115, 0.5);
        }
      }
      @media (prefers-color-scheme: light) {
        body {
          background: rgba(215, 115, 105, 0.5);
        }
      }
      .modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.2);
        z-index: 1998;
        backdrop-filter: blur(10px);
      }
    `);

panelStack = [];
dragInfo = null;
draggingPanel = null;
isDragging = false;
dragOffsetX = 0;
dragOffsetY = 0;

function addStyle(css) {
  const style = document.createElement("style");
  style.textContent = css;
  document.head.appendChild(style);
}

// Helper function to get coordinates from mouse or touch event
function getEventCoordinates(e) {
  if (e.type.startsWith("touch")) {
    const touch = e.touches[0] || e.changedTouches[0];
    return { clientX: touch.clientX, clientY: touch.clientY };
  }
  return { clientX: e.clientX, clientY: e.clientY };
}

function AAstartDrag(e, id) {
  // Delegate to startDrag for consistency
  startDrag(e, id);
}

function startDrag(e, id) {
  // Prevent drag if it's a double-click (for mouse events only)
  if (e.detail > 1 && !e.type.startsWith("touch")) {
    return;
  }

  const panel = document.getElementById(id);
  if (!panel || panel.dataset.noDrag === "true") return;

  // Prevent dragging if the target is a no-drag element or an interactive element
  if (
    e.target.closest(".no-drag") ||
    [
      "BUTTON",
      "A",
      "TEXTAREA",
      "INPUT",
      "SELECT",
      "FORM",
      "DETAILS",
      "SUMMARY",
    ].includes(e.target.tagName)
  ) {
    return;
  }

  if (window.modal) return;

  bringToFront(id);
  draggingPanel = panel;
  isDragging = true;

  const rect = panel.getBoundingClientRect();
  const coords = getEventCoordinates(e);
  dragOffsetX = coords.clientX - rect.left;
  dragOffsetY = coords.clientY - rect.top;

  panel.style.transition = "none";

  // Add touch and mouse event listeners
  document.addEventListener("mousemove", dragMove);
  document.addEventListener("mouseup", stopDrag);
  document.addEventListener("touchmove", dragMove, { passive: false });
  document.addEventListener("touchend", stopDrag);
}

function dragMove(e) {
  if (!isDragging || !draggingPanel) return;

  // Prevent default scrolling for touchmove to allow panel dragging
  if (e.type === "touchmove") {
    e.preventDefault();
  }

  const coords = getEventCoordinates(e);
  let x = coords.clientX - dragOffsetX;
  let y = coords.clientY - dragOffsetY;

  draggingPanel.style.left = x + "px";
  draggingPanel.style.top = y + "px";
}

function stopDrag(e) {
  isDragging = false;
  if (draggingPanel) {
    draggingPanel.style.transition =
      "transform 0.3s ease, left 0.3s ease, top 0.3s ease, opacity 0.3s ease";
  }
  draggingPanel = null;

  // Remove all drag-related event listeners
  document.removeEventListener("mousemove", dragMove);
  document.removeEventListener("mouseup", stopDrag);
  document.removeEventListener("touchmove", dragMove);
  document.removeEventListener("touchend", stopDrag);
}

function createPanel(id, pos, dir, w, h, mode = {}) {
  const existingPanel = document.getElementById(id);
  if (existingPanel) {
    if (mode.forceReload) {
      // Remove existing panel before creating new one
      existingPanel.remove();
      panelStack = panelStack.filter((pid) => pid !== id);
      console.log(`Forced reload of panel "${id}"`);
    } else {
      console.warn(`Panel with ID "${id}" already exists, skipping creation`);
      return;
    }
  }

  if (document.getElementById(id)) {
    console.warn(`Panel with ID "${id}" already exists, skipping creation`);
    return;
  }

  panelStack.push(id);

  if (mode.onlyHeader) {
    mode.noBody = true;
    mode.noFooter = true;
    mode.noHeader = false;
    mode.bodyCloseButton = false;
  } else if (mode.onlyBody) {
    mode.noHeader = true;
    mode.noFooter = true;
    mode.bodyCloseButton = false;
  } else if (mode.toast) {
    mode.noHeader = true;
    mode.noFooter = true;
    mode.bodyCloseButton = true;
  }

  if (mode.noESC) {
    if (!window.noESCpanels) window.noESCpanels = {};
    window.noESCpanels[id] = true;
  }

  const p = document.createElement("div");
  p.id = id;
  p.className = `panel ${mode.toast ? "toast" : ""} ${id === "uploadPanel" ? "upload-panel" : ""}`;
  p.dataset.slideFrom = dir;

  if (mode.resizeAble) {
    p.style.resize = "both";
    p.style.overflow = "auto";
  }

  let headerHTML = "";
  if (!mode.noHeader) {
    headerHTML = `
          <div class="panel-header" ${!mode.noDrag ? `onmousedown="startDrag(event, '${id}')" ontouchstart="startDrag(event, '${id}')"` : ""}>
            <div class="header-title">${mode.title || id}</div>
            ${
              mode.progressBar && !mode.progressBarNH
                ? ` <!-- Only show in header if progressBarNH is false -->
              <div class="header-progress-container">
                <div class="progress-bar">
                  <div class="progress-fill" id="progress-fill-${id}"></div>
                </div>
              </div>`
                : ""
            }
              ${mode.progressAnimation ? '<span class="header-progress"></span>' : ""}
            <div class="header-buttons">
              <button class="close-btn" onclick="closePanel('${id}')">&times;</button>
            </div>
          </div>`;
  }

  let bodyHTML = "";
  if (!mode.noBody) {
    if (mode.toast) {
      bodyHTML = `<div class="panel-body">${mode.bodyContent || ""}</div>`;
    } else {
      bodyHTML = `<div class="panel-body">${mode.bodyContent || ""}</div>`;
    }
  }

  let footerHTML = "";
  if (!mode.noFooter || mode.footerButtons || mode.footerCloseBtn) {
    const footerTitle = mode.footerTitle ?? "Footer";
    const centerFooter = !footerTitle || footerTitle.trim().length === 0;
    let footerButtons = "";
    if (mode.footerCloseBtn) {
      footerButtons += `<button class="close-footer-btn" onclick="closePanel('${id}')">Close</button>`;
    }
    footerHTML = `
          <div class="panel-footer${centerFooter ? " footer-center" : ""}">
            <div class="footer-title">
              ${footerTitle || ""}
              ${mode.progressBar ? `<span id="progress-text-${id}">${footerTitle ? "0%" : ""}</span><div class="progress-bar"><div id="progress-fill-${id}" class="progress-fill" style="width: ${mode.progressBar}%"></div></div>` : ""}
            </div>
            <div class="footer-buttons">
              <div class="left-buttons"></div>
              <div class="center-buttons"></div>
              <div class="right-buttons">${footerButtons}</div>
            </div>
          </div>`;
  }

  p.innerHTML = headerHTML + bodyHTML + footerHTML;
  document.body.appendChild(p);

  if (!mode.noBody && mode.bodyCloseButton) {
    addBodyButton(id, "×", {
      className: "body-close-btn",
      align: "right",
      onclick: (pid) => closePanel(pid),
    });
  }

  if (mode.overflow && !mode.noBody) {
    p.querySelector(".panel-body")?.style.setProperty("overflow", "auto");
  }

  if (mode.radius) {
    p.style.borderRadius =
      typeof mode.radius === "string" ? mode.radius : "8px";
    if (!mode.noHeader) {
      p.style.overflow = "hidden";
    }
  }

  // Add drag event listeners for body and header if noDrag is false
  if (!mode.noDrag && (mode.noHeader || mode.onlyBody) && !mode.noBody) {
    const body = p.querySelector(".panel-body");
    body?.addEventListener("mousedown", (e) => startDrag(e, id));
    body?.addEventListener("touchstart", (e) => startDrag(e, id), { passive: false });
  }
  if (!mode.noDrag && mode.onlyHeader) {
    const header = p.querySelector(".panel-header");
    header?.addEventListener("mousedown", (e) => startDrag(e, id));
    header?.addEventListener("touchstart", (e) => startDrag(e, id), { passive: false });
  }
  const footer = p.querySelector(".panel-footer");
  if (footer && !mode.noDrag) {
    footer.addEventListener("mousedown", (e) => startDrag(e, id));
    footer.addEventListener("touchstart", (e) => startDrag(e, id), { passive: false });
  }

  if (mode.timeClose) {
    setTimeout(() => {
      if (!mode._closed) closePanel(id);
    }, mode.timeClose);
  }
  positionPanel(p, pos, w, h);
  bringToFront(p);

  let finalLeft = parseFloat(p.style.left);
  let finalTop = parseFloat(p.style.top);
  let startLeft = finalLeft;
  let startTop = finalTop;

  if (dir === "ltr") startLeft = -w;
  if (dir === "rtl") startLeft = innerWidth;
  if (dir === "ttb") startTop = -h;
  if (dir === "btt") startTop = innerHeight;

  if (dir === "ctc") {
    p.style.transform = "scale(0)";
    p.style.opacity = 0;
  } else {
    p.style.left = startLeft + "px";
    p.style.top = startTop + "px";
    p.style.opacity = 0;
  }
  // Start progress animation if enabled
  if (mode.progressBar) {
    animateProgressBar(id, mode.progressBar);
  }
  // Start progress animation if enabled
  if (mode.progressAnimation) {
    animateProgress(id);
  }
  if (mode.verticalProgressBar) {
    animateVerticalProgressBar(id, mode.verticalProgressBar);
  }
  requestAnimationFrame(() => {
    p.style.transition =
      "transform 0.4s ease, left 0.4s ease, top 0.4s ease, opacity 0.4s ease";
    if (dir === "ctc") {
      p.style.transform = "scale(1)";
      p.style.opacity = 1;
    } else {
      p.style.left = finalLeft + "px";
      p.style.top = finalTop + "px";
      p.style.opacity = 1;
    }
  });

  setTimeout(() => {
    const footers = p.querySelectorAll(".panel-footer");
    console.log(`Panel "${id}" has ${footers.length} .panel-footer elements`);
  }, 0);
}

function animateProgress(id) {
  const panel = document.getElementById(id);
  if (!panel) return;
  const progressElement = panel.querySelector(".header-progress");
  if (!progressElement) return;

  const totalFrames = 12; // Total length of progress animation including brackets
  const animationSpeed = 200; // ms per frame
  let percentage = 0;
  let position = 3; // Start with [...[]........]

  const frames = [
    "[..[0%].........]",
    "[..[10%].........]",
    "[...[20%]........]",
    "[....[30%].......]",
    "[.....[40%]......]",
    "[......[50%].....]",
    "[.......[60%]....]",
    "[........[70%]...]",
    "[.........[80%]..]",
    "[.........[90%]..]",
    "[.........[100%].]",
    "[----- 100% -----]",
  ];

  let frameIndex = 0;

  function update() {
    if (frameIndex >= frames.length) return;
    progressElement.textContent = frames[frameIndex];
    frameIndex++;
    setTimeout(update, animationSpeed);
  }

  update();
}

function animateProgressBar(id, targetPercent) {
  const panel = document.getElementById(id);
  if (!panel) return;
  const progressFill = document.getElementById(`progress-fill-${id}`);
  if (!progressFill) return;
  const progressText = document.getElementById(`progress-text-${id}`);
  let currentPercent = 1;
  const duration = 2000; // 2 seconds for animation
  const startTime = performance.now();

  function animate(time) {
    const elapsed = time - startTime;
    const progress = Math.min(elapsed / duration, 1);
    currentPercent = 1 + progress * (targetPercent - 1);
    progressFill.style.width = `${currentPercent}%`;
    if (progressText) {
      progressText.textContent = `${Math.round(currentPercent)}%`;
    }
    if (progress < 1) {
      requestAnimationFrame(animate);
    }
  }

  requestAnimationFrame(animate);
}

function animateVerticalProgressBar(id, targetPercent) {
  const panel = document.getElementById(id);
  if (!panel) {
    console.error(`Panel ${id} not found`);
    return;
  }
  const progressFill = panel.querySelector(".progress-fill.vertical");
  if (!progressFill) {
    console.error(`Vertical progress fill element not found for panel ${id}`);
    return;
  }
  const progressLabel = panel.querySelector(".progress-label");

  let currentPercent = 0;
  const duration = 2000; // 2 seconds for animation
  const startTime = performance.now();

  function animate(time) {
    const elapsed = time - startTime;
    const progress = Math.min(elapsed / duration, 1);
    currentPercent = progress * targetPercent;
    progressFill.style.height = `${currentPercent}%`;
    if (progressLabel) {
      progressLabel.textContent = `${Math.round(currentPercent)}%`;
    }
    if (progress < 1) {
      requestAnimationFrame(animate);
    }
  }

  requestAnimationFrame(animate);
}

function bringToFront(p) {
  if (typeof p === "string") p = document.getElementById(p);
  if (!p) return;

  let highest = [...document.querySelectorAll(".panel")].reduce(
    (z, el) => Math.max(z, parseInt(el.style.zIndex) || 0),
    100,
  );
  p.style.zIndex = highest + 1;

  panelStack = panelStack.filter((pid) => pid !== p.id);
  panelStack.push(p.id);
}

function closePanel(id) {
  const p = document.getElementById(id);
  if (!p) return;

  let dir = p.dataset.slideFrom;
  let left = parseFloat(p.style.left);
  let top = parseFloat(p.style.top);
  let w = p.offsetWidth;
  let h = p.offsetHeight;
  let endLeft = left;
  let endTop = top;

  if (dir === "ltr") endLeft = -w;
  if (dir === "rtl") endLeft = innerWidth;
  if (dir === "ttb") endTop = -h;
  if (dir === "btt") endTop = innerHeight;

  if (dir === "ctc") {
    p.style.transform = "scale(0)";
    p.style.opacity = 0;
  } else {
    p.style.left = endLeft + "px";
    p.style.top = endTop + "px";
    p.style.opacity = 0;
  }

  setTimeout(() => {
    p.remove();
    panelStack = panelStack.filter((pid) => pid !== id);
  }, 400);
}

function positionPanel(p, pos, w, h) {
  let l = 0,
    t = 0;
  switch (pos) {
    case "TL":
      break;
    case "TC":
      l = (innerWidth - w) / 2;
      break;
    case "T1":
      l = (innerWidth - w) / 2;
      t = 20;
      break;
    case "TR":
      l = innerWidth - w;
      break;
    case "TR1":
      l = innerWidth - w - 10;
      t = 10;
      break;
    case "CL":
      t = (innerHeight - h) / 2;
      break;
    case "CC":
      l = (innerWidth - w) / 2;
      t = (innerHeight - h) / 2;
      break;
    case "CR":
      l = innerWidth - w;
      t = (innerHeight - h) / 2;
      break;
    case "BL":
      t = innerHeight - h;
      break;
    case "BC":
      l = (innerWidth - w) / 2;
      t = innerHeight - h;
      break;
    case "BR":
      l = innerWidth - w;
      t = innerHeight - h;
      break;
  }
  p.style.left = l + "px";
  p.style.top = t + "px";
  p.style.width = w + "px";
  p.style.height = h + "px";
}

document.addEventListener("keydown", function (e) {
  if (e.key === "Escape" && panelStack.length) {
    if (window.modal) return;
    for (let i = panelStack.length - 1; i >= 0; i--) {
      let id = panelStack[i];
      if (!window.noESCpanels || !window.noESCpanels[id]) {
        closePanel(id);
        break;
      }
    }
  }
});

function setContent(id, part, content = "", background = null, color = null) {
  console.log("part:" + part);
  const p = document.getElementById(id);
  if (!p) {
    console.log("Panel not found:", id);
    return;
  }
  const sel = {
    header: ".panel-header",
    body: ".panel-body",
    footer: ".panel-footer",
  }[part];
  const section = p.querySelector(sel);
  if (!section) {
    console.warn("Section missing:", part);
    return;
  }
  if (background) {
    section.style.background = background;
  }
  if (color) {
    section.style.color = color;
  }
  if (part === "header") {
    const titleElement = section.querySelector(".header-title");
    if (titleElement) {
      titleElement.textContent = content;
    } else {
      console.error(`Title element not found in header of panel "${id}".`);
    }
  }
  if (part === "footer") {
    const title = section.querySelector(".footer-title");
    const buttonDiv =
      section.querySelector(".footer-buttons") || document.createElement("div");
    buttonDiv.className = "footer-buttons";
    if (title) {
      title.innerHTML = content;
    } else {
      section.innerHTML = "";
      const titleDiv = document.createElement("div");
      titleDiv.className = "footer-title";
      titleDiv.textContent = content;
      section.appendChild(titleDiv);
      section.appendChild(buttonDiv);
    }
  }
}

function addBodyButton(id, label, options = {}) {
  const p = document.getElementById(id);
  if (!p) {
    console.warn(`Panel not found: ${id}`);
    return;
  }

  let body = p.querySelector(".panel-body");
  if (!body) {
    console.warn(`Panel body not found for panel: ${id}`);
    return;
  }

  const position = options.position || "top";
  let buttonsDiv = body.querySelector(`.body-buttons-${position}`);
  if (!buttonsDiv) {
    buttonsDiv = document.createElement("div");
    buttonsDiv.className =
      position === "top" || position === "bottom"
        ? `body-buttons body-buttons-horizontal body-buttons-${position}`
        : `body-buttons body-buttons-${position}`;

    if (options.title) {
      const titleDiv = document.createElement("div");
      titleDiv.className = "buttons-title";
      titleDiv.textContent = options.title;
      buttonsDiv.appendChild(titleDiv);
    }
    if (position === "top") {
      body.insertBefore(buttonsDiv, body.firstChild);
    } else if (position === "bottom") {
      body.appendChild(buttonsDiv);
    } else if (position === "left" || position === "right") {
      body.insertBefore(buttonsDiv, body.firstChild);
    }
  }

  const align = options.align || "left";
  let alignContainer = buttonsDiv.querySelector(`.${align}-buttons`);
  if (!alignContainer) {
    alignContainer = document.createElement("div");
    alignContainer.className = `${align}-buttons`;
    if (align === "left") {
      buttonsDiv.prepend(alignContainer);
    } else if (align === "center") {
      const leftButtons = buttonsDiv.querySelector(".left-buttons");
      if (leftButtons) {
        leftButtons.after(alignContainer);
      } else {
        buttonsDiv.prepend(alignContainer);
      }
    } else if (align === "right") {
      buttonsDiv.appendChild(alignContainer);
    }
  }

  const button = document.createElement("button");
  button.textContent = label;

  if (options.background !== undefined) {
    if (options.background === "") {
      button.style.background = "none";
    } else {
      button.style.background = options.background;
    }
  }

  if (options.color) {
    button.style.color = options.color;
  }
  if (options.className) {
    button.className = options.className;
  }

  if (options.onclick) {
    button.onclick = () => {
      options.onclick(id, label);
      console.log(`Body button "${label}" clicked in panel "${id}"`);
      window.lastButtonClicked = { id, label, timestamp: Date.now() };
    };
  } else {
    button.onclick = () => {
      console.log(`Body button "${label}" clicked in panel "${id}"`);
      window.lastButtonClicked = { id, label, timestamp: Date.now() };

      if (["Close", "Cancel"].includes(label)) {
        console.log(`Cancelled action in panel "${id}"`);
        closePanel(id);
      } else if (["Yes", "Confirm", "OK"].includes(label)) {
        console.log(`Confirmed action in panel "${id}"`);
      }
    };
  }

  alignContainer.appendChild(button);
}

function addFooterButton(id, label, options = {}) {
  const p = document.getElementById(id);
  if (!p) {
    console.warn(`Panel not found: ${id}`);
    return;
  }

  let footer = p.querySelector(".panel-footer");
  if (!footer) {
    console.warn(
      `No .panel-footer found for panel "${id}", checking for existing footers`,
    );
    const existingFooters = p.querySelectorAll(".panel-footer");
    if (existingFooters.length > 0) {
      console.error(
        `Multiple .panel-footer found (${existingFooters.length}) in panel "${id}", using first`,
      );
      footer = existingFooters[0];
    } else {
      console.log(`Creating new .panel-footer for panel "${id}"`);
      footer = document.createElement("div");
      footer.className = "panel-footer";
      footer.innerHTML = `
            <div class="footer-title"></div>
            <div class="footer-buttons">
              <div class="left-buttons"></div>
              <div class="center-buttons"></div>
              <div class="right-buttons"></div>
            </div>
          `;
      p.appendChild(footer);
    }
  }

  let buttonsDiv = footer.querySelector(".footer-buttons");
  if (!buttonsDiv) {
    console.warn(`No .footer-buttons found in panel "${id}", creating one`);
    buttonsDiv = document.createElement("div");
    buttonsDiv.className = "footer-buttons";
    buttonsDiv.innerHTML = `
          <div class="left-buttons"></div>
          <div class="center-buttons"></div>
          <div class="right-buttons"></div>
        `;
    footer.appendChild(buttonsDiv);
  }

  const align = options.align || "left";
  let alignContainer = buttonsDiv.querySelector(`.${align}-buttons`);
  if (!alignContainer) {
    alignContainer = document.createElement("div");
    alignContainer.className = `${align}-buttons`;
    if (align === "left") {
      buttonsDiv.prepend(alignContainer);
    } else if (align === "center") {
      const leftButtons = buttonsDiv.querySelector(".left-buttons");
      if (leftButtons) {
        leftButtons.after(alignContainer);
      } else {
        buttonsDiv.prepend(alignContainer);
      }
    } else if (align === "right") {
      buttonsDiv.appendChild(alignContainer);
    }
  }

  const existingButton = alignContainer.querySelector(
    `button[data-label="${label}"]`,
  );
  if (existingButton) {
    console.log(
      `Footer button "${label}" already exists in panel "${id}", skipping add`,
    );
    return;
  }

  const button = document.createElement("button");
  button.textContent = label;
  button.dataset.label = label;
// 🎯 NEW: Tooltip support
  button.title = options.title || label;
  if (options.background) {
    button.style.background = options.background;
  }
  if (options.color) {
    button.style.color = options.color;
  }
  if (options.className) {
    button.className = options.className;
  }

  if (options.onclick) {
    button.onclick = () => {
      options.onclick(id, label);
      console.log(`Footer button "${label}" clicked in panel "${id}"`);
      window.lastButtonClicked = { id, label, timestamp: Date.now() };
    };
  } else {
    button.onclick = () => {
      console.log(`Footer button "${label}" clicked in panel "${id}"`);
      window.lastButtonClicked = { id, label, timestamp: Date.now() };

      if (["Yes", "Confirm", "OK"].includes(label)) {
        console.log(`Confirmed action in panel "${id}"`);
      } else if (["No", "Cancel", "Close"].includes(label)) {
        console.log(`Cancelled action in panel "${id}"`);
        closePanel(id);
      }
    };
  }

  alignContainer.appendChild(button);
  console.log(`Added footer button "${label}" to panel "${id}"`);
}



function togglePanelDrag(id) {
  const panel = document.getElementById(id);
  if (!panel) return;

  const isLocked = panel.dataset.noDrag === "true";
  panel.dataset.noDrag = !isLocked;

  // Add visual feedback
  if (isLocked) {
    // Unlocking - green feedback
    panel.style.transform = "scale(1.05)";
    panel.style.boxShadow = "0 0 0 3px rgba(76, 175, 80, 0.7)";
  } else {
    // Locking - red feedback
    panel.style.transform = "scale(1.05)";
    panel.style.boxShadow = "0 0 0 3px rgba(255, 0, 0, 0.5)";
  }

  setTimeout(() => {
    panel.style.transform = "scale(1)";
    panel.style.boxShadow = "2px 2px 12px rgba(0, 0, 0, 0.2)";
  }, 300);

  console.log(`Panel "${id}" is now ${!isLocked ? "locked" : "unlocked"}`);
}