window.debug = false;
window.logomode = true;
host="https://i.kunok.com/";
function displayMessage(message, isError = false, isInfo = false) {
    let msgDiv = document.getElementById("message");
    if (!msgDiv) {
        msgDiv = document.createElement("div");
        msgDiv.id = "message";
        document.body.insertBefore(msgDiv, document.body.firstChild);
    }
    msgDiv.textContent = message;
    msgDiv.className = "message " + (isError ? "error" : isInfo ? "info" : "success");
    if (isInfo) {
        setTimeout(() => {
            if (msgDiv.className.includes("info")) {
                msgDiv.textContent = "";
                msgDiv.className = "message";
            }
        }, 3000);
    }
}

function logoset(targetId, previewId = "updateImgPreview") {
    const stringText = window.imgsrc;
    const targetInput = document.getElementById(targetId);
    const previewDiv = document.getElementById(previewId);
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';
    if (targetInput && typeof stringText !== "undefined") {
        targetInput.value = stringText;
        if (previewDiv) {
            previewDiv.innerHTML = `<img src="${stringText}" alt="preview" style="max-width: 100px; max-height: 100px; object-fit: cover; border-radius: 4px;">`;
            const img = previewDiv.querySelector('img');
            img.onerror = () => {
                console.warn("Failed to load preview image, reverting to default.");
                img.src = defaultAvatar;
            };
        }
    }
}

function togglePassword(inputId, buttonId) {
    const input = document.getElementById(inputId);
    if (input) {
        input.type = input.type === "password" ? "text" : "password";
        const button = document.getElementById(buttonId);
        if (button)
            button.textContent = input.type === "password" ? "Show" : "Hide";
    }
}

async function postto(apiUrl = "", payload = {}) {
    try {
        const response = await fetch(apiUrl, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: payload }),
        });
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error("Request failed:", error);
        displayMessage(error.message, true);
        throw error;
    }
}

function closePanel(id) {
    const panel = document.getElementById(id);
    if (panel) panel.remove();
}

async function checkUser(userId) {
    try {
        
        const response = await fetch(host+`aui.php?action=list&id=${encodeURIComponent(userId)}`);
        const result = await response.json();
        if (result.data && result.data.length > 0) {
            return result.data[0];
        } else {
            throw new Error('User not found');
        }
    } catch (error) {
        throw new Error('Error fetching user: ' + error.message);
    }
}

async function updateUserPanel(userId = "") {
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';

    if (!userId && window.user && window.user.id) {
        userId = window.user.id;
    }

    if (!userId) {
        displayMessage("No user ID provided", true);
        return;
    }

    try {
        displayMessage("Loading user data...", false, true);
        const user = await checkUser(userId);

        const formHTML = `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                #updateImgPreview img {
                    max-width: 100px;
                    max-height: 100px;
                    object-fit: cover;
                    border-radius: 4px;
                }
            </style>
            <div class="logo-controls">
                <div id="updateImgPreview"><img src="${user.img ? decodeURI(user.img) : defaultAvatar}" alt="preview"></div>
                <div class="button-group" style="display: flex; gap: 10px; justify-content: center; margin-top: 10px;">
                    <button type="button" id="updateUploadLogoBtn">Upload logo</button>
                    <button type="button" id="updateApplyLogoBtn">Apply Logo</button>
                </div>
            </div>
            <form id="updateUserForm">
                <div><label for="updateId">ID:</label> <input type="text" id="updateId" value="${user.id}" style="width: 100%;></div>
                <div><label for="updateName">Name:</label> <input type="text" id="updateName" value="${user.name || ''}"></div>
                <div><label for="updateUsername">Username:</label> <input type="text" id="updateUsername" value="${user.username || ''}"></div>
                <div>
                    <label for="updatePassword">Password:</label>
                    <input type="password" id="updatePassword" value="${user.password || ''}">
                    <button type="button" id="toggleUpdatePasswordBtn">Show</button>
                </div>
                <div class="form-row">
                    <div>
                        <label for="updateFlag">Flag:</label>
                        <select id="updateFlag">
                            <option value="1" ${user.flag == 1 ? 'selected' : ''}>1 (Active)</option>
                            <option value="0" ${user.flag == 0 ? 'selected' : ''}>0 (Suspended)</option>
                            <option value="3" ${user.flag == 3 ? 'selected' : ''}>3 (Wait)</option>
                        </select>
                    </div>
                    <div>
                        <label for="updateLevel">Level:</label>
                        <select id="updateLevel">
                            <option value="1" ${user.level == 1 ? 'selected' : ''}>1 (User)</option>
                            <option value="2" ${user.level == 2 ? 'selected' : ''}>2 (Moderator)</option>
                            <option value="3" ${user.level == 3 ? 'selected' : ''}>3 (Admin)</option>
                        </select>
                    </div>
                </div>
                <div>
                    <label for="updateImg">Image URL:</label>
                    <input type="text" id="updateImg" value="${user.img ? (user.img) : ''}">
                </div>
                <button type="submit">Update User</button>
            </form>
            <div id="message" class="message"></div>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("updateUserPanel", "CC", "ctc", 400, 600, {
            bodyContent: formHTML,
            noFooter: true,
            title: 'Update User'
        });

        const form = document.getElementById("updateUserForm");
        if (form) {
            form.addEventListener("submit", async (e) => {
                e.preventDefault();
                const payload = {
                    id: userId,
                    name: document.getElementById("updateName").value,
                    username: document.getElementById("updateUsername").value,
                    password: document.getElementById("updatePassword").value,
                    flag: parseInt(document.getElementById("updateFlag").value),
                    level: parseInt(document.getElementById("updateLevel").value),
                    img: document.getElementById("updateImg").value ? (document.getElementById("updateImg").value) : ''
                };
                try {
                    
                    const response = await postto(host+"aui.php?action=update", payload);
                    displayMessage(response.message || `User ${userId} updated successfully`);
form.reset();
//document.getElementById("updateImgPreview").innerHTML = `<img src="${defaultAvatar}" alt="preview">`;

// Wait 2 seconds to show message before redirect
// setTimeout(() => { closePanel("updateUserPanel"); window.location.href = "index.html";}, 2000);

                } catch (error) {
                    console.error("Update user error:", error);
                    displayMessage(`Error updating user ${userId}: ${error.message}`, true);
                }
            });

            document.getElementById("toggleUpdatePasswordBtn").addEventListener("click", () =>
                togglePassword("updatePassword", "toggleUpdatePasswordBtn")
            );
            document.getElementById("updateUploadLogoBtn").addEventListener("click", () => {
                window.logomode = true;
                toggleUploadPanel(); // from upl3.js
            });
            document.getElementById("updateApplyLogoBtn").addEventListener("click", () =>
                logoset("updateImg", "updateImgPreview")
            );
        }

    } catch (error) {
        displayMessage(error.message, true);
    }
}
async function addUserPanel() {
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';

    const formHTML = `
        <style>
            .form-row { display: flex; gap: 10px; }
            .form-row > div { flex: 1; }
            #addImgPreview img {
                max-width: 100px;
                max-height: 100px;
                object-fit: cover;
                border-radius: 4px;
            }
        </style>
        <div class="logo-controls">
            <div id="addImgPreview"><img src="${defaultAvatar}" alt="preview"></div>
            <div class="button-group" style="display: flex; gap: 10px; justify-content: center; margin-top: 10px;">
                <button type="button" id="addUploadLogoBtn">Upload logo</button>
                <button type="button" id="addApplyLogoBtn">Apply Logo</button>
            </div>
        </div>
        <form id="addUserForm">
            <div><label for="addName">Name:</label> <input type="text" id="addName" required></div>
            <div><label for="addUsername">Username:</label> <input type="text" id="addUsername" required></div>
            <div>
                <label for="addPassword">Password:</label>
                <input type="password" id="addPassword" required>
                <button type="button" id="toggleAddPasswordBtn">Show</button>
            </div>
            <div class="form-row">
                <div>
                    <label for="addFlag">Flag:</label>
                    <select id="addFlag">
                        <option value="1" selected>1 (Active)</option>
                        <option value="0">0 (Suspended)</option>
                        <option value="3">3 (Wait)</option>
                    </select>
                </div>
                <div>
                    <label for="addLevel">Level:</label>
                    <select id="addLevel">
                        <option value="1" selected>1 (User)</option>
                        <option value="2">2 (Moderator)</option>
                        <option value="3">3 (Admin)</option>
                    </select>
                </div>
            </div>
            <div>
                <label for="addImg">Image URL:</label>
                <input type="text" id="addImg">
            </div>
            <button type="submit">Add User</button>
        </form>
        <div id="message" class="message"></div>
    `;

    if (typeof createPanel !== "function") {
        console.error("Missing createPanel(). Please include panel system.");
        return;
    }

    createPanel("addUserPanel", "CC", "ctc", 400, 600, {
        bodyContent: formHTML,
        noFooter: true,
        title: 'Add New User'
    });

    const form = document.getElementById("addUserForm");
    if (form) {
        form.addEventListener("submit", async (e) => {
            e.preventDefault();
            
            const username = document.getElementById("addUsername").value;
            
            // First check if username exists
            try {
                
                const checkResponse = await postto(host+"aui.php?action=check_username", { username });
                if (checkResponse.exists) {
                    displayMessage("Username already exists", true);
                    return;
                }
                
                // If username doesn't exist, proceed with adding user
                const payload = {
                    name: document.getElementById("addName").value,
                    username: username,
                    password: document.getElementById("addPassword").value,
                    flag: parseInt(document.getElementById("addFlag").value),
                    level: parseInt(document.getElementById("addLevel").value),
                    img: document.getElementById("addImg").value ? (document.getElementById("addImg").value) : ''
                };
                
                const response = await postto(host+"aui.php?action=add", payload);
                displayMessage(response.message || "User added successfully");
                form.reset();
                
                // Close panel after 2 seconds
                setTimeout(() => {
                    closePanel("addUserPanel");
                }, 2000);
                
            } catch (error) {
                console.error("Add user error:", error);
                displayMessage(`Error adding user: ${error.message}`, true);
            }
        });

        document.getElementById("toggleAddPasswordBtn").addEventListener("click", () =>
            togglePassword("addPassword", "toggleAddPasswordBtn")
        );
        document.getElementById("addUploadLogoBtn").addEventListener("click", () => {
            window.logomode = true;
            toggleUploadPanel(); // from upl3.js
        });
        document.getElementById("addApplyLogoBtn").addEventListener("click", () =>
            logoset("addImg", "addImgPreview")
        );
    }
}
async function listUsersPanel() {
    try {
        displayMessage("Loading users...", false, true);
        
        const response = await fetch(host+"aui.php?action=list");
        const result = await response.json();
        
        if (!result.data || result.data.length === 0) {
            displayMessage("No users found", true);
            return;
        }

        // Create table HTML
        let tableHTML = `
            <style>
                .user-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 10px;
                }
                .user-table th, .user-table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                }
                .user-table th {
                    background-color: #f2f2f2;
                }
                .user-table tr:nth-child(even) {
                    background-color: #f9f9f9;
                }
                .user-table tr:hover {
                    background-color: #f1f1f1;
                }
                .action-buttons {
                    display: flex;
                    gap: 5px;
                }
                .action-buttons button {
                    padding: 3px 6px;
                    font-size: 12px;
                }
                .user-avatar {
                    width: 30px;
                    height: 30px;
                    border-radius: 50%;
                    object-fit: cover;
                }
            </style>
            <table class="user-table">
                <thead>
                    <tr>
                        <th>Avatar</th>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Username</th>
                        <th>Status</th>
                        <th>Level</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
        `;

        // Add rows for each user
        result.data.forEach(user => {
            const avatarSrc = user.img ? decodeURI(user.img) : 'https://placehold.co/30x30/666/fff/png';
            const statusText = 
                user.flag == 0 ? 'Suspended' : 
                user.flag == 3 ? 'Wait' : 'Active';
            const levelText = 
                user.level == 1 ? 'User' : 
                user.level == 2 ? 'Moderator' : 'Admin';
            
            tableHTML += `
                <tr>
                    <td><img src="${user.img}" class="user-avatar" alt="avatar"></td>
                    <td>${user.id}</td>
                    <td>${user.name || '-'}</td>
                    <td>${user.username}</td>
                    <td>${statusText}</td>
                    <td>${levelText}</td>
                    <td class="action-buttons">
                        <button class="edit-btn" data-id="${user.id}">Edit</button>
                        <button class="delete-btn" data-id="${user.id}">Delete</button>
                    </td>
                </tr>
            `;
        });

        tableHTML += `
                </tbody>
            </table>
            <div id="message" class="message"></div>
        `;

        // Create panel
        createPanel("listUsersPanel", "CC", "ctc", 800, 600, {
            bodyContent: tableHTML,
            noFooter: true,
            title: 'User List'
        });

        // Add event listeners to buttons
        document.querySelectorAll('.edit-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const userId = e.target.getAttribute('data-id');
                updateUserPanel(userId);
            });
        });

        document.querySelectorAll('.delete-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                const userId = e.target.getAttribute('data-id');
                if (confirm(`Are you sure you want to delete user ${userId}?`)) {
                    try {
                        
                        const response = await postto(host+"aui.php?action=del", { id: userId });
                        displayMessage(response.message || `User ${userId} deleted successfully`);
                        // Refresh the list after deletion
                        setTimeout(() => {
                            closePanel("listUsersPanel");
                            listUsersPanel();
                        }, 1000);
                    } catch (error) {
                        displayMessage(`Error deleting user: ${error.message}`, true);
                    }
                }
            });
        });

    } catch (error) {
        displayMessage(`Error loading users: ${error.message}`, true);
    }
}
// Function to list all items
async function listItemsPanel() {
    try {
        displayMessage("Loading items...", false, true);
        
        const response = await fetch(host+"api62.php?action=list_all&fields=id,name,flag,timestamp,board");
        const result = await response.json();
        
        if (!result.data || result.data.length === 0) {
            displayMessage("No items found", true);
            return;
        }

        // Create table HTML
        let tableHTML = `
            <style>
                .items-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 10px;
                }
                .items-table th, .items-table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                }
                .items-table th {
                    background-color: #f2f2f2;
                }
                .items-table tr:nth-child(even) {
                    background-color: #f9f9f9;
                }
                .items-table tr:hover {
                    background-color: #f1f1f1;
                }
                .action-buttons {
                    display: flex;
                    gap: 5px;
                }
                .action-buttons button {
                    padding: 3px 6px;
                    font-size: 12px;
                }
            </style>
            <div style="margin-bottom: 10px;">
                <button id="addNewItemBtn">Add New Item</button>
            </div>
            <table class="items-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Bucket</th>
                        <th>Cup</th>
                        <th>Board</th>
                        <th>Keeper</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
        `;

        // Add rows for each item
        result.data.forEach(item => {
            const statusText = 
                item.flag == 0 ? 'Inactive' : 
                item.flag == 3 ? 'Deleted' : 'Active';
            
            tableHTML += `
                <tr>
                    <td>${item.id}</td>
                    <td>${item.name || '-'}</td>
                    <td>${item.bucket || '-'}</td>
                    <td>${item.cup || '-'}</td>
                    <td>${item.board || '-'}</td>
                    <td>${item.keeper || '-'}</td>
                    <td>${statusText}</td>
                    <td class="action-buttons">
                        <button class="edit-item-btn" data-id="${item.id}">Edit</button>
                        <button class="delete-item-btn" data-id="${item.id}">Delete</button>
                    </td>
                </tr>
            `;
        });

        tableHTML += `
                </tbody>
            </table>
            <div id="message" class="message"></div>
        `;

        // Create panel
        createPanel("listItemsPanel", "CC", "ctc", 900, 600, {
            bodyContent: tableHTML,
            noFooter: true,
            title: 'Items List'
        });

        // Add event listeners to buttons
        document.getElementById('addNewItemBtn').addEventListener('click', addItemPanel);

        document.querySelectorAll('.edit-item-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const itemId = e.target.getAttribute('data-id');
                updateItemPanel(itemId);
            });
        });

        document.querySelectorAll('.delete-item-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                const itemId = e.target.getAttribute('data-id');
                if (confirm(`Are you sure you want to delete item ${itemId}?`)) {
                    try {                        
                        const response = await postto(host+"api622.php", { 
                            action: 'delete',
                            id: itemId
                        });
                        displayMessage(response.message || `Item ${itemId} deleted successfully`);
                        // Refresh the list after deletion
                        setTimeout(() => {
                            closePanel("listItemsPanel");
                            listItemsPanel();
                        }, 1000);
                    } catch (error) {
                        displayMessage(`Error deleting item: ${error.message}`, true);
                    }
                }
            });
        });

    } catch (error) {
        displayMessage(`Error loading items: ${error.message}`, true);
    }
}

// Function to add a new item
async function addItemPanel() {
    const formHTML = `
        <style>
            .form-row { display: flex; gap: 10px; }
            .form-row > div { flex: 1; }
        </style>
        <form id="addItemForm">
            <div><label for="addItemId">ID:</label> <input type="text" id="addItemId" required></div>
            <div><label for="addItemName">Name:</label> <input type="text" id="addItemName" required></div>
            <div class="form-row">
                <div><label for="addItemBucket">Bucket:</label> <input type="text" id="addItemBucket"></div>
                <div><label for="addItemCup">Cup:</label> <input type="text" id="addItemCup"></div>
            </div>
            <div class="form-row">
                <div><label for="addItemBoard">Board:</label> <input type="text" id="addItemBoard"></div>
                <div><label for="addItemKeeper">Keeper:</label> <input type="text" id="addItemKeeper"></div>
            </div>
            <div>
                <label for="addItemFlag">Status:</label>
                <select id="addItemFlag">
                    <option value="1" selected>Active</option>
                    <option value="0">Inactive</option>
                </select>
            </div>
            <button type="submit">Add Item</button>
        </form>
        <div id="message" class="message"></div>
    `;

    createPanel("addItemPanel", "CC", "ctc", 500, 500, {
        bodyContent: formHTML,
        noFooter: true,
        title: 'Add New Item'
    });

    const form = document.getElementById("addItemForm");
    if (form) {
        form.addEventListener("submit", async (e) => {
            e.preventDefault();
            
            const payload = {
                id: document.getElementById("addItemId").value,
                name: document.getElementById("addItemName").value,
                bucket: document.getElementById("addItemBucket").value,
                cup: document.getElementById("addItemCup").value,
                board: document.getElementById("addItemBoard").value,
                keeper: document.getElementById("addItemKeeper").value,
                flag: parseInt(document.getElementById("addItemFlag").value)
            };
            
            try {
                const response = await postto("api622.php", payload);
                displayMessage(response.message || "Item added successfully");
                form.reset();
                
                // Close panel after 2 seconds and refresh list
                setTimeout(() => {
                    closePanel("addItemPanel");
                    closePanel("listItemsPanel");
                    listItemsPanel();
                }, 1000);
                
            } catch (error) {
                displayMessage(`Error adding item: ${error.message}`, true);
            }
        });
    }
}

// Function to update an existing item
async function updateItemPanel(itemId) {
    try {
        displayMessage("Loading item data...", false, true);        
        const response = await fetch(host+`api622.php?id=${encodeURIComponent(itemId)}`);
         const result = await response.json();
        
        if (!result.data) {
            throw new Error('Item not found');
        }

        const item = result.data;

        const formHTML = `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                .form-textarea {
                    width: 100%;
                    min-height: 80px;
                    padding: 8px;
                    border: 1px solid #ddd;
                    border-radius: 4px;
                    resize: vertical;
                }
                .form-field {
                    margin-bottom: 15px;
                }
                .form-field label {
                    display: block;
                    margin-bottom: 5px;
                    font-weight: bold;
                }
            </style>
            <form id="updateItemForm">
                <div class="form-field">
                    <label for="updateItemId">ID:</label>
                    <input type="text" id="updateItemId" value="${item.id}" readonly>
                </div>
                
                <div class="form-field">
                    <label for="updateItemName">Name:</label>
                    <textarea id="updateItemName" class="form-textarea" required>${item.name || ''}</textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-field">
                        <label for="updateItemBucket">Bucket:</label>
                        <textarea id="updateItemBucket" class="form-textarea">${item.bucket || ''}</textarea>
                    </div>
                    <div class="form-field">
                        <label for="updateItemCup">Cup:</label>
                        <textarea id="updateItemCup" class="form-textarea">${item.cup || ''}</textarea>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-field">
                        <label for="updateItemBoard">Board:</label>
                        <textarea id="updateItemBoard" class="form-textarea">${item.board || ''}</textarea>
                    </div>
                    <div class="form-field">
                        <label for="updateItemKeeper">Keeper:</label>
                        <textarea id="updateItemKeeper" class="form-textarea">${item.keeper || ''}</textarea>
                    </div>
                </div>
                
                <div class="form-field">
                    <label for="updateItemFlag">Status:</label>
                    <select id="updateItemFlag">
                        <option value="1" ${item.flag == 1 ? 'selected' : ''}>Active</option>
                        <option value="0" ${item.flag == 0 ? 'selected' : ''}>Inactive</option>
                    </select>
                </div>
                
                <button type="submit" style="margin-top: 15px;">Update Item</button>
            </form>
            <div id="message" class="message"></div>
        `;

        createPanel("updateItemPanel", "CC", "ctc", 600, 700, {  // Increased height for textareas
            bodyContent: formHTML,
            noFooter: true,
            title: 'Update Item'
        });

        const form = document.getElementById("updateItemForm");
        if (form) {
            form.addEventListener("submit", async (e) => {
                e.preventDefault();
                
                const payload = {
                    id: itemId,
                    name: document.getElementById("updateItemName").value.trim(),
                    bucket: document.getElementById("updateItemBucket").value.trim(),
                    cup: document.getElementById("updateItemCup").value.trim(),
                    board: document.getElementById("updateItemBoard").value.trim(),
                    keeper: document.getElementById("updateItemKeeper").value.trim(),
                    flag: parseInt(document.getElementById("updateItemFlag").value)
                };
                
                try {
                    const response = await postto("api622.php", payload);
                    displayMessage(response.message || "Item updated successfully");
                    
                    // Close panel after 2 seconds and refresh list
                    setTimeout(() => {
                        closePanel("updateItemPanel");
                        closePanel("listItemsPanel");
                        listItemsPanel();
                    }, 1000);
                    
                } catch (error) {
                    displayMessage(`Error updating item: ${error.message}`, true);
                }
            });
        }

    } catch (error) {
        displayMessage(error.message, true);
    }
}

async function listItemsBoard() {
    try {
        displayMessage("Loading board data...", false, true);
        
        const response = await fetch(host+"api62.php?action=list_all&fields=id,name,flag,timestamp,board");
        const result = await response.json();
        
        if (!result.data || result.data.length === 0) {
            displayMessage("No items found", true);
            return;
        }

        // Create panel with table and summary
        const panelHTML = `
            <style>
                .board-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 10px;
                    font-size: 14px;
                }
                .board-table th, .board-table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                }
                .board-table th {
                    background-color: #f2f2f2;
                    position: sticky;
                    top: 0;
                }
                .board-table tr:nth-child(even) {
                    background-color: #f9f9f9;
                }
                .board-table tr:hover {
                    background-color: #f1f1f1;
                }
                .completed {
                    background-color: #dfd !important;
                }
                .in-progress {
                    background-color: #ffd !important;
                }
                .not-started {
                    background-color: #fdd !important;
                }
                .new-member {
                    background-color: #e6f7ff !important;
                    font-weight: bold;
                }
                .existing-member {
                    background-color: #f6ffed !important;
                }
                .summary-stats {
                    margin-top: 20px;
                    padding: 15px;
                    background-color: #f8f9fa;
                    border-radius: 5px;
                }
                .summary-stats h3 {
                    margin-top: 0;
                    color: #333;
                }
                .summary-stats p {
                    margin: 5px 0;
                }
                .scroll-container {
                    max-height: 500px;
                    overflow-y: auto;
                }
            </style>
            <div class="scroll-container" id="boardDataContainer"></div>
            <div class="summary-stats" id="summaryStats"></div>
            <div id="message" class="message"></div>
        `;

        createPanel("listItemsBoardPanel", "CC", "ctc", 900, 700, {
            bodyContent: panelHTML,
            noFooter: true,
            title: 'Items Board View'
        });

        // Render the data
        renderBoardData(result);

    } catch (error) {
        displayMessage(`Error loading board data: ${error.message}`, true);
    }
}

function renderBoardData(loadedData) {
    const container = document.getElementById('boardDataContainer');
    const summaryDiv = document.getElementById('summaryStats');
    
    if (!container || !summaryDiv) return;

    container.innerHTML = '';
    summaryDiv.innerHTML = '';

    if (!loadedData || !loadedData.data) {
        container.textContent = 'No data available';
        return;
    }

    // Create table
    const table = document.createElement('table');
    table.className = 'board-table';

    // Table header
    const headerRow = table.insertRow();
    const headers = [
        'ID', 'Name', 'Flag', 'User ID', 'Username',
        'Total', 'Answered', 'Progress %',
        'Bucket Total', 'Bucket Answered', 'Bucket %',
        'Cup Total', 'Cup Answered', 'Cup %',
        'Save Status', 'Time', 'First Time'
    ];
    
    headers.forEach(headerText => {
        const th = document.createElement('th');
        th.textContent = headerText;
        headerRow.appendChild(th);
    });

    // Counters for summary
    let stats = {
        completed: 0,
        inProgress: 0,
        notStarted: 0,
        bucketCompleted: 0,
        cupCompleted: 0,
        newMembers: 0,
        existingMembers: 0,
        total: loadedData.data.length
    };

    // Add data rows
    loadedData.data.forEach(item => {
        const row = table.insertRow();
        
        try {
            const board = item.board ? JSON.parse(item.board) : {};
            
            // Determine member type
            const isNewMember = board.userid ? board.userid.startsWith('2025') : false;
            const flagText = isNewMember ? 'new' : (board.userid ? 'existing' : 'unknown');
            
            // Calculate percentages
            const totalPercent = calculatePercentage(board.answeredRequired, board.totalRequired);
            const bucketPercent = calculatePercentage(board.bucket_answeredRequired, board.bucket_totalRequired);
            const cupPercent = calculatePercentage(board.cup_answeredRequired, board.cup_totalRequired);
            
            // Update statistics
            updateStats(stats, board, isNewMember);
            
            // Add cells
            const cells = [
                item.id || '',
                item.name || '',
                flagText,
                board.userid || '',
                board.user_name || '',
                board.totalRequired || '0',
                board.answeredRequired || '0',
                totalPercent,
                board.bucket_totalRequired || '0',
                board.bucket_answeredRequired || '0',
                bucketPercent,
                board.cup_totalRequired || '0',
                board.cup_answeredRequired || '0',
                cupPercent,
                board.save || '',
                board.time || '',
                board.firstTime || ''
            ];
            
            cells.forEach((cellText, index) => {
                const td = document.createElement('td');
                td.textContent = cellText;
                
                // Apply styling
                if (headers[index].endsWith('%') && cellText !== 'N/A') {
                    const percent = parseFloat(cellText);
                    if (percent >= 100) {
                        td.classList.add('completed');
                    } else if (percent >= 50) {
                        td.classList.add('in-progress');
                    } else if (percent > 0) {
                        td.classList.add('not-started');
                    }
                }
                
                if (headers[index] === 'Flag') {
                    if (cellText === 'new') {
                        td.classList.add('new-member');
                    } else if (cellText === 'existing') {
                        td.classList.add('existing-member');
                    }
                }
                
                row.appendChild(td);
            });
        } catch (e) {
            console.error('Error parsing board data:', e);
            const errorCell = document.createElement('td');
            errorCell.colSpan = headers.length;
            errorCell.textContent = `Error: ${e.message}`;
            errorCell.style.color = 'red';
            row.appendChild(errorCell);
        }
    });

    container.appendChild(table);
    renderSummaryStats(summaryDiv, stats);
}

function calculatePercentage(answered, total) {
    if (!total || isNaN(total) || !answered || isNaN(answered)) return 'N/A';
    return (answered / total * 100).toFixed(1) + '%';
}

function updateStats(stats, board, isNewMember) {
    if (isNewMember) {
        stats.newMembers++;
    } else if (board.userid) {
        stats.existingMembers++;
    }
    
    if (board.totalRequired && board.answeredRequired) {
        if (board.answeredRequired >= board.totalRequired) {
            stats.completed++;
        } else if (board.answeredRequired > 0) {
            stats.inProgress++;
        } else {
            stats.notStarted++;
        }
    }
    
    if (board.bucket_totalRequired && board.bucket_answeredRequired && 
        board.bucket_answeredRequired >= board.bucket_totalRequired) {
        stats.bucketCompleted++;
    }
    
    if (board.cup_totalRequired && board.cup_answeredRequired && 
        board.cup_answeredRequired >= board.cup_totalRequired) {
        stats.cupCompleted++;
    }
}

function renderSummaryStats(container, stats) {
    container.innerHTML = `
        <h3>Summary Statistics</h3>
        <p><strong>Member Types:</strong></p>
        <p>New Members: ${stats.newMembers} (${calculatePercentage(stats.newMembers, stats.total)})</p>
        <p>Existing Members: ${stats.existingMembers} (${calculatePercentage(stats.existingMembers, stats.total)})</p>
        
        <p><strong>Overall Progress:</strong></p>
        <p>Total Items: ${stats.total}</p>
        <p>Completed: ${stats.completed} (${calculatePercentage(stats.completed, stats.total)})</p>
        <p>In Progress: ${stats.inProgress} (${calculatePercentage(stats.inProgress, stats.total)})</p>
        <p>Not Started: ${stats.notStarted} (${calculatePercentage(stats.notStarted, stats.total)})</p>
        
        <p><strong>Bucket Progress:</strong></p>
        <p>Completed: ${stats.bucketCompleted} (${calculatePercentage(stats.bucketCompleted, stats.total)})</p>
        
        <p><strong>Cup Progress:</strong></p>
        <p>Completed: ${stats.cupCompleted} (${calculatePercentage(stats.cupCompleted, stats.total)})</p>
    `;
}
// Global memory for users
let userMemoryCache = [];
// Global sort state
let sortColumn = null;
let sortDirection = 'asc';
async function loadUsersIntoMemory() {
    try {
        
        const response = await fetch(host+"aui.php?action=list");
        const result = await response.json();
        if (result.data) {
            userMemoryCache = result.data;
            return true;
        }
        return false;
    } catch (error) {
        console.error("Failed to load users:", error);
        return false;
    }
}

async function listMixBoard() {
    try {
        // Load users first
        if (userMemoryCache.length === 0) {
            const loaded = await loadUsersIntoMemory();
            if (!loaded) {
                throw new Error("Could not load user data");
            }
        }

        // Load board data
        displayMessage("Loading board data...", false, true);
        
        const response = await fetch(host+"api62.php?action=list_all&fields=id,name,flag,timestamp,board");
        const result = await response.json();

        if (!result.data || result.data.length === 0) {
            displayMessage("No board data found", true);
            return;
        }

        // Create panel with table and summary container
        const panelHTML = `
            <style>
                .mixboard-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 10px;
                    font-size: 14px;
                }
                .mixboard-table th, .mixboard-table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                }
                .mixboard-table th {
                    background-color: #f2f2f2;
                    position: sticky;
                    top: 0;
                }
                .scroll-container {
                    max-height: 500px;
                    overflow-y: auto;
                }
                .summary-container {
                    margin-top: 20px;
                }
            </style>
            <div class="scroll-container">
                <table class="mixboard-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Flag</th>
                            <th>User ID</th>
                            <th>Username</th>
                            <th>Total</th>
                            <th>Answered</th>
                            <th>Progress %</th>
                            <th>Bucket Total</th>
                            <th>Bucket Answered</th>
                            <th>Bucket %</th>
                            <th>Cup Total</th>
                            <th>Cup Answered</th>
                            <th>Cup %</th>
                            <th>Save Status</th>
                            <th>Time</th>
                            <th>First Time</th>
                        </tr>
                    </thead>
                    <tbody id="mixboardDataBody"></tbody>
                </table>
            </div>
            <div id="summaryContainer" class="summary-container"></div>
            <div id="message" class="message"></div>
        `;

        createPanel("listMixBoardPanel", "CC", "ctc", 1200, 700, {
            bodyContent: panelHTML,
            noFooter: true,
            title: 'Mixed Board View'
        });

        // Render data with user matching
        renderMixBoardData(result);

    } catch (error) {
        displayMessage(`Error: ${error.message}`, true);
    }
}

function renderMixBoardData(loadedData) {
    const tbody = document.getElementById('mixboardDataBody');
    const summaryContainer = document.getElementById('summaryContainer');
    if (!tbody || !summaryContainer) return;

    tbody.innerHTML = '';

    if (!loadedData || !loadedData.data) {
        tbody.textContent = 'No data available';
        return;
    }

    // Add data rows
    loadedData.data.forEach(item => {
        const row = document.createElement('tr');

        try {
            const board = item.board ? JSON.parse(item.board) : {};
            const matchedUser = userMemoryCache.find(user => user.id === item.name) || {};

            // Determine flag
            const flagFromBoard = board.userid ? 
                (board.userid.startsWith('2025') ? 'new' : 'existing') : 
                'unknown';

            // Calculate percentages
            const totalPercent = calculatePercentage(board.answeredRequired, board.totalRequired);
            const bucketPercent = calculatePercentage(board.bucket_answeredRequired, board.bucket_totalRequired);
            const cupPercent = calculatePercentage(board.cup_answeredRequired, board.cup_totalRequired);

            // Create cells
            const cells = [
                item.id || '',
                item.name || '',
                flagFromBoard,
                matchedUser.id || board.userid || 'N/A',
                matchedUser.username || board.user_name || 'N/A',
                board.totalRequired || '0',
                board.answeredRequired || '0',
                totalPercent,
                board.bucket_totalRequired || '0',
                board.bucket_answeredRequired || '0',
                bucketPercent,
                board.cup_totalRequired || '0',
                board.cup_answeredRequired || '0',
                cupPercent,
                board.save || '',
                board.time || '',
                board.firstTime || ''
            ];

            cells.forEach((cellText, index) => {
                const td = document.createElement('td');
                td.textContent = cellText;
                td.style.border = '1px solid #ddd';
                td.style.padding = '8px';

                // Apply color coding for percentage columns
                if (['Progress %', 'Bucket %', 'Cup %'].includes(document.querySelector('thead tr').children[index].textContent) && cellText !== 'N/A') {
                    const percent = parseFloat(cellText);
                    if (percent >= 100) {
                        td.style.backgroundColor = '#dfd';
                    } else if (percent >= 50) {
                        td.style.backgroundColor = '#ffd';
                    } else {
                        td.style.backgroundColor = '#fdd';
                    }
                }

                // Apply styling for flag column
                if (document.querySelector('thead tr').children[index].textContent === 'Flag') {
                    if (cellText === 'new') {
                        td.style.backgroundColor = '#e6f7ff';
                        td.style.fontWeight = 'bold';
                    } else if (cellText === 'existing') {
                        td.style.backgroundColor = '#f6ffed';
                    }
                }

                // Highlight unmatched user info
                if (['User ID', 'Username'].includes(document.querySelector('thead tr').children[index].textContent) && cellText === 'N/A') {
                    td.style.backgroundColor = '#ffeeee';
                }

                row.appendChild(td);
            });

            tbody.appendChild(row);

        } catch (e) {
            console.error('Error rendering item:', item, e);
            const errorCell = document.createElement('td');
            errorCell.colSpan = 17;
            errorCell.textContent = `Error: ${e.message}`;
            errorCell.style.color = 'red';
            row.appendChild(errorCell);
            tbody.appendChild(row);
        }
    });

    // Add summary statistics
    if (loadedData.data.length > 0) {
        let completed = 0, inProgress = 0, notStarted = 0;
        let bucketCompleted = 0, cupCompleted = 0;
        let newMembers = 0, existingMembers = 0;

        loadedData.data.forEach(item => {
            try {
                const board = item.board ? JSON.parse(item.board) : {};

                // Count member types
                if (board.userid) {
                    if (board.userid.startsWith('2025')) {
                        newMembers++;
                    } else {
                        existingMembers++;
                    }
                }

                // Main progress
                if (board.totalRequired && board.answeredRequired) {
                    if (board.answeredRequired >= board.totalRequired) {
                        completed++;
                    } else if (board.answeredRequired > 0) {
                        inProgress++;
                    } else {
                        notStarted++;
                    }
                }

                // Bucket progress
                if (board.bucket_totalRequired && board.bucket_answeredRequired) {
                    if (board.bucket_answeredRequired >= board.bucket_totalRequired) {
                        bucketCompleted++;
                    }
                }

                // Cup progress
                if (board.cup_totalRequired && board.cup_answeredRequired) {
                    if (board.cup_answeredRequired >= board.cup_totalRequired) {
                        cupCompleted++;
                    }
                }
            } catch (e) {
                console.error('Error parsing board for statistics:', e);
            }
        });

        summaryContainer.innerHTML = `
            <h3>Summary Statistics</h3>
            <p><strong>Member Types:</strong></p>
            <p>New Members: ${newMembers} (${Math.round(newMembers / loadedData.data.length * 100)}%)</p>
            <p>Existing Members: ${existingMembers} (${Math.round(existingMembers / loadedData.data.length * 100)}%)</p>
            <p><strong>Overall Progress:</strong></p>
            <p>Total Items: ${loadedData.data.length}</p>
            <p>Completed: ${completed} (${Math.round(completed / loadedData.data.length * 100)}%)</p>
            <p>In Progress: ${inProgress} (${Math.round(inProgress / loadedData.data.length * 100)}%)</p>
            <p>Not Started: ${notStarted} (${Math.round(notStarted / loadedData.data.length * 100)}%)</p>
            <p><strong>Bucket Progress:</strong></p>
            <p>Completed: ${bucketCompleted} (${Math.round(bucketCompleted / loadedData.data.length * 100)}%)</p>
            <p><strong>Cup Progress:</strong></p>
            <p>Completed: ${cupCompleted} (${Math.round(cupCompleted / loadedData.data.length * 100)}%)</p>
        `;
    }
}
async function listMixBoardSort() {
    try {
        // Load users first
        if (userMemoryCache.length === 0) {
            const loaded = await loadUsersIntoMemory();
            if (!loaded) {
                throw new Error("Could not load user data");
            }
        }

        // Load board data
        displayMessage("Loading board data...", false, true);
        
        const response = await fetch(host+"api62.php?action=list_all&fields=id,name,flag,timestamp,board");
        const result = await response.json();

        if (!result.data || result.data.length === 0) {
            displayMessage("No board data found", true);
            return;
        }

        // Create panel with table and summary container
        const panelHTML = `
            <style>
                .mixboard-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 10px;
                    font-size: 14px;
                }
                .mixboard-table th, .mixboard-table td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: left;
                }
                .mixboard-table th {
                    background-color: #f2f2f2;
                    position: sticky;
                    top: 0;
                }
                .mixboard-table th.sortable {
                    cursor: pointer;
                }
                .mixboard-table th.sortable:hover {
                    background-color: #e0e0e0;
                }
                .sort-asc::after {
                    content: ' ↑';
                }
                .sort-desc::after {
                    content: ' ↓';
                }
                .scroll-container {
                    max-height: 500px;
                    overflow-y: auto;
                }
                .summary-container {
                    margin-top: 20px;
                }
            </style>
            <div class="scroll-container">
                <table class="mixboard-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Flag</th>
                            <th class="sortable" data-column="userId">User ID</th>
                            <th class="sortable" data-column="userName">User Name</th>
                            <th class="sortable" data-column="username">Username</th>
                            <th>Total</th>
                            <th>Answered</th>
                            <th class="sortable" data-column="progressPercent">Progress %</th>
                            <th>Bucket Total</th>
                            <th>Bucket Answered</th>
                            <th class="sortable" data-column="bucketPercent">Bucket %</th>
                            <th>Cup Total</th>
                            <th>Cup Answered</th>
                            <th class="sortable" data-column="cupPercent">Cup %</th>
                            <th>Save Status</th>
                            <th class="sortable" data-column="time">Time</th>
                            <th>First Time</th>
                        </tr>
                    </thead>
                    <tbody id="mixboardDataBody"></tbody>
                </table>
            </div>
            <div id="summaryContainer" class="summary-container"></div>
            <div id="message" class="message"></div>
        `;

        createPanel("listMixBoardSortPanel", "CC", "ctc", 1200, 700, {
            bodyContent: panelHTML,
            noFooter: true,
            title: 'Mixed Board View with Sorting'
        });

        // Render initial data
        renderMixBoardDataSort(result);

        // Add click event listeners for sortable columns
        const sortableHeaders = document.querySelectorAll('.mixboard-table th.sortable');
        sortableHeaders.forEach(header => {
            header.addEventListener('click', () => {
                const column = header.getAttribute('data-column');
                // Toggle sort direction if clicking the same column
                if (sortColumn === column) {
                    sortDirection = sortDirection === 'asc' ? 'desc' : 'asc';
                } else {
                    sortColumn = column;
                    sortDirection = 'asc';
                }
                // Update sort indicators
                sortableHeaders.forEach(h => {
                    h.classList.remove('sort-asc', 'sort-desc');
                    if (h.getAttribute('data-column') === sortColumn) {
                        h.classList.add(`sort-${sortDirection}`);
                    }
                });
                // Re-render data with new sort
                renderMixBoardDataSort(result);
            });
        });

    } catch (error) {
        displayMessage(`Error: ${error.message}`, true);
    }
}

function renderMixBoardDataSort(loadedData) {
    const tbody = document.getElementById('mixboardDataBody');
    const summaryContainer = document.getElementById('summaryContainer');
    if (!tbody || !summaryContainer) return;

    tbody.innerHTML = '';

    if (!loadedData || !loadedData.data) {
        tbody.textContent = 'No data available';
        return;
    }

    // Sort data if a sort column is selected
    let sortedData = [...loadedData.data];
    if (sortColumn) {
        sortedData.sort((a, b) => {
             let boardA = {}, boardB = {}, matchedUserA = {}, matchedUserB = {};
            try {
                boardA = a.board ? JSON.parse(a.board) : {};
                boardB = b.board ? JSON.parse(b.board) : {};
                matchedUserA = userMemoryCache.find(user => user.id === a.name) || {};
                matchedUserB = userMemoryCache.find(user => user.id === b.name) || {};
            } catch (e) {
                console.error('Error parsing board during sort:', e);
                // Return 0 to maintain relative order for invalid items
                return 0;
            }

            let valueA, valueB;
            switch (sortColumn) {
                case 'userId':
                    valueA = matchedUserA.id || boardA.userid || '';
                    valueB = matchedUserB.id || boardB.userid || '';
                    break;
                case 'userName':
                    valueA = matchedUserA.name || boardA.user_name || '';
                    valueB = matchedUserB.name || boardB.user_name || '';
                    break;
                case 'username':
                    valueA = matchedUserA.username || boardA.user_name || '';
                    valueB = matchedUserB.username || boardB.user_name || '';
                    break;
                case 'progressPercent':
                    valueA = boardA.totalRequired && boardA.answeredRequired
                        ? boardA.answeredRequired / boardA.totalRequired * 100
                        : 0;
                    valueB = boardB.totalRequired && boardB.answeredRequired
                        ? boardB.answeredRequired / boardB.totalRequired * 100
                        : 0;
                    break;
                 case 'bucketPercent':
                    valueA = boardA.bucket_totalRequired && boardA.bucket_answeredRequired
                        ? boardA.bucket_answeredRequired / boardA.bucket_totalRequired * 100
                        : 0;
                    valueB = boardB.bucket_totalRequired && boardB.bucket_answeredRequired
                        ? boardB.bucket_answeredRequired / boardB.bucket_totalRequired * 100
                        : 0;
                    break;
                case 'cupPercent':
                    valueA = boardA.cup_totalRequired && boardA.cup_answeredRequired
                        ? boardA.cup_answeredRequired / boardA.cup_totalRequired * 100
                        : 0;
                    valueB = boardB.cup_totalRequired && boardB.cup_answeredRequired
                        ? boardB.cup_answeredRequired / boardB.cup_totalRequired * 100
                        : 0;
                    break;                    
                case 'time':
                    valueA = boardA.time || '';
                    valueB = boardB.time || '';
                    break;
                default:
                    return 0;
            }

            // Handle numeric sorting for progressPercent
            if (sortColumn === 'progressPercent') {
                return sortDirection === 'asc' ? valueA - valueB : valueB - valueA;
            }

            // String sorting (case-insensitive)
            valueA = valueA.toString().toLowerCase();
            valueB = valueB.toString().toLowerCase();
            if (valueA < valueB) return sortDirection === 'asc' ? -1 : 1;
            if (valueA > valueB) return sortDirection === 'asc' ? 1 : -1;
            return 0;
        });
    }

    // Add data rows
    sortedData.forEach(item => {
        const row = document.createElement('tr');

        try {
            const board = item.board ? JSON.parse(item.board) : {};
            const matchedUser = userMemoryCache.find(user => user.id === item.name) || {};

            // Determine flag
            const flagFromBoard = board.userid
                ? (board.userid.startsWith('2025') ? 'new' : 'existing')
                : 'unknown';

            // Calculate percentages
            const totalPercent = calculatePercentage(board.answeredRequired, board.totalRequired);
            const bucketPercent = calculatePercentage(board.bucket_answeredRequired, board.bucket_totalRequired);
            const cupPercent = calculatePercentage(board.cup_answeredRequired, board.cup_totalRequired);

            // Create cells
            const cells = [
                item.id || '',
                item.name || '',
                flagFromBoard,
                matchedUser.id || board.userid || 'N/A',
                matchedUser.name || board.user_name || 'N/A',
                matchedUser.username || board.user_name || 'N/A',
                board.totalRequired || '0',
                board.answeredRequired || '0',
                totalPercent,
                board.bucket_totalRequired || '0',
                board.bucket_answeredRequired || '0',
                bucketPercent,
                board.cup_totalRequired || '0',
                board.cup_answeredRequired || '0',
                cupPercent,
                board.save || '',
                board.time || '',
                board.firstTime || ''
            ];

            cells.forEach((cellText, index) => {
                const td = document.createElement('td');
                td.textContent = cellText;
                td.style.border = '1px solid #ddd';
                td.style.padding = '8px';

                // Apply color coding for percentage columns
                if (['Progress %', 'Bucket %', 'Cup %'].includes(document.querySelector('thead tr').children[index].textContent) && cellText !== 'N/A') {
                    const percent = parseFloat(cellText);
                    if (percent >= 100) {
                        td.style.backgroundColor = '#dfd';
                    } else if (percent >= 50) {
                        td.style.backgroundColor = '#ffd';
                    } else {
                        td.style.backgroundColor = '#fdd';
                    }
                }

                // Apply styling for flag column
                if (document.querySelector('thead tr').children[index].textContent === 'Flag') {
                    if (cellText === 'new') {
                        td.style.backgroundColor = '#e6f7ff';
                        td.style.fontWeight = 'bold';
                    } else if (cellText === 'existing') {
                        td.style.backgroundColor = '#f6ffed';
                    }
                }

                // Highlight unmatched user info
                if (['User ID', 'User Name', 'Username'].includes(document.querySelector('thead tr').children[index].textContent) && cellText === 'N/A') {
                    td.style.backgroundColor = '#ffeeee';
                }

                row.appendChild(td);
            });

            tbody.appendChild(row);

        } catch (e) {
            console.error('Error rendering item:', item, e);
            const errorCell = document.createElement('td');
            errorCell.colSpan = 18;
            errorCell.textContent = `Error: ${e.message}`;
            errorCell.style.color = 'red';
            row.appendChild(errorCell);
            tbody.appendChild(row);
        }
    });

    // Add summary statistics
    if (loadedData.data.length > 0) {
        let completed = 0, inProgress = 0, notStarted = 0;
        let bucketCompleted = 0, cupCompleted = 0;
        let newMembers = 0, existingMembers = 0;

        loadedData.data.forEach(item => {
            try {
                const board = item.board ? JSON.parse(item.board) : {};

                // Count member types
                if (board.userid) {
                    if (board.userid.startsWith('2025')) {
                        newMembers++;
                    } else {
                        existingMembers++;
                    }
                }

                // Main progress
                if (board.totalRequired && board.answeredRequired) {
                    if (board.answeredRequired >= board.totalRequired) {
                        completed++;
                    } else if (board.answeredRequired > 0) {
                        inProgress++;
                    } else {
                        notStarted++;
                    }
                }

                // Bucket progress
                if (board.bucket_totalRequired && board.bucket_answeredRequired) {
                    if (board.bucket_answeredRequired >= board.bucket_totalRequired) {
                        bucketCompleted++;
                    }
                }

                // Cup progress
                if (board.cup_totalRequired && board.cup_answeredRequired) {
                    if (board.cup_answeredRequired >= board.cup_totalRequired) {
                        cupCompleted++;
                    }
                }
            } catch (e) {
                console.error('Error parsing board for statistics:', e);
            }
        });

        summaryContainer.innerHTML = `
            <h3>Summary Statistics</h3>
            <p><strong>Member Types:</strong></p>
            <p>New Members: ${newMembers} (${Math.round(newMembers / loadedData.data.length * 100)}%)</p>
            <p>Existing Members: ${existingMembers} (${Math.round(existingMembers / loadedData.data.length * 100)}%)</p>
            <p><strong>Overall Progress:</strong></p>
            <p>Total Items: ${loadedData.data.length}</p>
            <p>Completed: ${completed} (${Math.round(completed / loadedData.data.length * 100)}%)</p>
            <p>In Progress: ${inProgress} (${Math.round(inProgress / loadedData.data.length * 100)}%)</p>
            <p>Not Started: ${notStarted} (${Math.round(notStarted / loadedData.data.length * 100)}%)</p>
            <p><strong>Bucket Progress:</strong></p>
            <p>Completed: ${bucketCompleted} (${Math.round(bucketCompleted / loadedData.data.length * 100)}%)</p>
            <p><strong>Cup Progress:</strong></p>
            <p>Completed: ${cupCompleted} (${Math.round(cupCompleted / loadedData.data.length * 100)}%)</p>
        `;
    }
}
function calculatePercentage(answered, total) {
    if (!total || isNaN(total) || !answered || isNaN(answered)) return 'N/A';
    return (answered / total * 100).toFixed(1) + '%';
}