window.debug = false;
window.logomode = true;

function displayMessage(message, isError = false, isInfo = false) {
    let msgDiv = document.getElementById("message");
    if (!msgDiv) {
        msgDiv = document.createElement("div");
        msgDiv.id = "message";
        document.body.insertBefore(msgDiv, document.body.firstChild);
    }
    msgDiv.textContent = message;
    msgDiv.className = "message " + (isError ? "error" : isInfo ? "info" : "success");
    if (isInfo) {
        setTimeout(() => {
            if (msgDiv.className.includes("info")) {
                msgDiv.textContent = "";
                msgDiv.className = "message";
            }
        }, 3000);
    }
}

function logoset(targetId, previewId = "updateImgPreview") {
    const stringText = window.imgsrc;
    const targetInput = document.getElementById(targetId);
    const previewDiv = document.getElementById(previewId);
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';
    if (targetInput && typeof stringText !== "undefined") {
        targetInput.value = stringText;
        if (previewDiv) {
            previewDiv.innerHTML = `<img src="${stringText}" alt="preview" style="max-width: 100px; max-height: 100px; object-fit: cover; border-radius: 4px;">`;
            const img = previewDiv.querySelector('img');
            img.onerror = () => {
                console.warn("Failed to load preview image, reverting to default.");
                img.src = defaultAvatar;
            };
        }
    }
}

function togglePassword(inputId, buttonId) {
    const input = document.getElementById(inputId);
    const button = document.getElementById(buttonId);
    if (input && button) {
        input.type = input.type === "password" ? "text" : "password";
        button.textContent = input.type === "password" ? "Show" : "Hide";
    } else {
        console.error(`Input (${inputId}) or Button (${buttonId}) not found`);
    }
}

async function postto(apiUrl = "", payload = {}) {
    try {
        const response = await fetch(apiUrl, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: payload }),
        });
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error("Request failed:", error);
        displayMessage(error.message, true);
        throw error;
    }
}

function closePanel(id) {
    const panel = document.getElementById(id);
    if (panel) panel.remove();
}

async function checkUser(userId) {
    try {
        const response = await fetch(`https://j.kunok.com/aui.php?action=list&id=${encodeURIComponent(userId)}`);
        const result = await response.json();
        if (result.data && result.data.length > 0) {
            return result.data[0];
        } else {
            throw new Error('User not found');
        }
    } catch (error) {
        throw new Error('Error fetching user: ' + error.message);
    }
}

async function addUserPanel() {
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';

    const formHTML = `
        <style>
            .form-row { display: flex; gap: 10px; }
            .form-row > div { flex: 1; }
            #addImgPreview img {
                max-width: 100px;
                max-height: 100px;
                object-fit: cover;
                border-radius: 4px;
            }
        </style>
        <div class="logo-controls">
            <div id="addImgPreview"><img src="${defaultAvatar}" alt="preview"></div>
            <div class="button-group" style="display: flex; gap: 10px; justify-content: center; margin-top: 10px;">
                <button type="button" id="addUploadLogoBtn">Upload logo</button>
                <button type="button" id="addApplyLogoBtn">Apply Logo</button>
            </div>
        </div>
        <form id="addUserForm">
            <div><label for="addName">Name:</label> <input type="text" id="addName" value=""></div>
            <div><label for="addUsername">Username:</label> <input type="text" id="addUsername" value=""></div>
            <div>
                <label for="addPassword">Password:</label>
                <input type="password" id="addPassword" value="">
                <button type="button" id="toggleAddPasswordBtn">Show</button>
            </div>
            <div class="form-row">
                <div>
                    <label for="addFlag">Flag:</label>
                    <select id="addFlag">
                        <option value="1">1 (Active)</option>
                        <option value="0">0 (Suspended)</option>
                        <option value="3">3 (Wait)</option>
                    </select>
                </div>
                <div>
                    <label for="addLevel">Level:</label>
                    <select id="addLevel">
                        <option value="1">1 (User)</option>
                        <option value="2">2 (Moderator)</option>
                        <option value="3">3 (Admin)</option>
                    </select>
                </div>
            </div>
            <div>
                <label for="addImg">Image URL:</label>
                <input type="text" id="addImg" value="">
            </div>
            <button type="submit">Add User</button>
        </form>
        <div id="message" class="message"></div>
    `;

    if (typeof createPanel !== "function") {
        console.error("Missing createPanel(). Please include panel system.");
        return;
    }

    createPanel("addUserPanel", "CC", "ctc", 400, 600, {
        bodyContent: formHTML,
        noFooter: true,
        title: 'Add User'
    });

    const form = document.getElementById("addUserForm");
    if (form) {
        form.addEventListener("submit", async (e) => {
            e.preventDefault();
            const payload = {
                name: document.getElementById("addName").value,
                username: document.getElementById("addUsername").value,
                password: document.getElementById("addPassword").value,
                flag: parseInt(document.getElementById("addFlag").value),
                level: parseInt(document.getElementById("addLevel").value),
                img: document.getElementById("addImg").value ? btoa(document.getElementById("addImg").value) : ''
            };
            try {
                const response = await postto("https://j.kunok.com/aui.php?action=add", payload);
                displayMessage(response.message || `User created successfully`);
                form.reset();
                document.getElementById("addImgPreview").innerHTML = `<img src="${defaultAvatar}" alt="preview">`;
            } catch (error) {
                console.error("Add user error:", error);
                displayMessage(`Error adding user: ${error.message}`, true);
            }
        });

        document.getElementById("toggleAddPasswordBtn").addEventListener("click", () =>
            togglePassword("addPassword", "toggleAddPasswordBtn")
        );
        document.getElementById("addUploadLogoBtn").addEventListener("click", () => {
            if (filesToUpload.length > 0) {
                if (!confirm("Files already exist in the upload panel. Do you want to add a new logo?")) {
                    return;
                }
                filesToUpload = []; // Clear existing files if user confirms
            }
            window.logomode = true;
            toggleUploadPanel(); // from upl23.js
        });
        document.getElementById("addApplyLogoBtn").addEventListener("click", () =>
            logoset("addImg", "addImgPreview")
        );
    }
}

async function updateUserPanel(userId = "") {
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';

    if (!userId && window.user && window.user.id) {
        userId = window.user.id;
    }

    if (!userId) {
        displayMessage("No user ID provided", true);
        return;
    }

    try {
        displayMessage("Loading user data...", false, true);
        const user = await checkUser(userId);

        const formHTML = `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                #updateImgPreview img {
                    max-width: 100px;
                    max-height: 100px;
                    object-fit: cover;
                    border-radius: 4px;
                }
            </style>
            <div class="logo-controls">
                <div id="updateImgPreview"><img src="${user.img ? atob(user.img) : defaultAvatar}" alt="preview"></div>
                <div class="button-group" style="display: flex; gap: 10px; justify-content: center; margin-top: 10px;">
                    <button type="button" id="updateUploadLogoBtn">Upload logo</button>
                    <button type="button" id="updateApplyLogoBtn">Apply Logo</button>
                </div>
            </div>
            <form id="updateUserForm">
                <div><label for="updateId">ID:</label> <input type="text" id="updateId" value="${user.id}" style="width: 100%;" readonly></div>
                <div><label for="updateName">Name:</label> <input type="text" id="updateName" value="${user.name || ''}"></div>
                <div><label for="updateUsername">Username:</label> <input type="text" id="updateUsername" value="${user.username || ''}"></div>
                <div>
                    <label for="updatePassword">Password:</label>
                    <input type="password" id="updatePassword" value="${user.password || ''}">
                    <button type="button" id="toggleUpdatePasswordBtn">Show</button>
                </div>
                <div class="form-row">
                    <div>
                        <label for="updateFlag">Flag:</label>
                        <select id="updateFlag">
                            <option value="1" ${user.flag == 1 ? 'selected' : ''}>1 (Active)</option>
                            <option value="0" ${user.flag == 0 ? 'selected' : ''}>0 (Suspended)</option>
                            <option value="3" ${user.flag == 3 ? 'selected' : ''}>3 (Wait)</option>
                        </select>
                    </div>
                    <div>
                        <label for="updateLevel">Level:</label>
                        <select id="updateLevel">
                            <option value="1" ${user.level == 1 ? 'selected' : ''}>1 (User)</option>
                            <option value="2" ${user.level == 2 ? 'selected' : ''}>2 (Moderator)</option>
                            <option value="3" ${user.level == 3 ? 'selected' : ''}>3 (Admin)</option>
                        </select>
                    </div>
                </div>
                <div>
                    <label for="updateImg">Image URL:</label>
                    <input type="text" id="updateImg" value="${user.img ? atob(user.img) : ''}">
                </div>
                <button type="submit">Update User</button>
            </form>
            <div id="message" class="message"></div>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("updateUserPanel", "CC", "ctc", 400, 600, {
            bodyContent: formHTML,
            noFooter: true,
            title: 'Update User'
        });

        const form = document.getElementById("updateUserForm");
        if (form) {
            form.addEventListener("submit", async (e) => {
                e.preventDefault();
                const payload = {
                    id: userId,
                    name: document.getElementById("updateName").value,
                    username: document.getElementById("updateUsername").value,
                    password: document.getElementById("updatePassword").value,
                    flag: parseInt(document.getElementById("updateFlag").value),
                    level: parseInt(document.getElementById("updateLevel").value),
                    img: document.getElementById("updateImg").value ? btoa(document.getElementById("updateImg").value) : ''
                };
                try {
                    const response = await postto("https://j.kunok.com/aui.php?action=update", payload);
                    displayMessage(response.message || `User ${userId} updated successfully`);
                    form.reset();
                    document.getElementById("updateImgPreview").innerHTML = `<img src="${defaultAvatar}" alt="preview">`;
                } catch (error) {
                    console.error("Update user error:", error);
                    displayMessage(`Error updating user ${userId}: ${error.message}`, true);
                }
            });

            document.getElementById("toggleUpdatePasswordBtn").addEventListener("click", () =>
                togglePassword("updatePassword", "toggleUpdatePasswordBtn")
            );
            document.getElementById("updateUploadLogoBtn").addEventListener("click", () => {
                if (filesToUpload.length > 0) {
                    if (!confirm("Files already exist in the upload panel. Do you want to add a new logo?")) {
                        return;
                    }
                    filesToUpload = []; // Clear existing files if user confirms
                }
                window.logomode = true;
                toggleUploadPanel(); // from upl23.js
            });
            document.getElementById("updateApplyLogoBtn").addEventListener("click", () =>
                logoset("updateImg", "updateImgPreview")
            );
        }

    } catch (error) {
        displayMessage(error.message, true);
    }
}

async function listUsersPanel() {
    const defaultAvatar = 'https://placehold.co/100x100/666/fff/png';

    try {
        displayMessage("Loading users...", false, true);
        const response = await fetch('https://j.kunok.com/aui.php?action=list');
        const result = await response.json();
        const users = result.data || [];

        if (users.length === 0) {
            displayMessage("No users found", false, true);
            return;
        }

        let tableHTML = `
            <style>
                table { width: 100%; border-collapse: collapse; }
                th, td { padding: 8px; border: 1px solid #ddd; text-align: left; }
                th { background-color: #f2f2f2; }
                .user-img { max-width: 50px; max-height: 50px; object-fit: cover; border-radius: 4px; }
                .action-btn { padding: 4px 8px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; }
                .edit-btn { background-color: #4CAF50; color: white; }
                .delete-btn { background-color: #f44336; color: white; }
            </style>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Username</th>
                        <th>Flag</th>
                        <th>Level</th>
                        <th>Image</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
        `;

        users.forEach(user => {
            const imgSrc = user.img ? atob(user.img) : defaultAvatar;
            const flagText = user.flag == 1 ? 'Active' : user.flag == 0 ? 'Suspended' : user.flag == 3 ? 'Wait' : user.flag;
            const levelText = user.level == 1 ? 'User' : user.level == 2 ? 'Moderator' : user.level == 3 ? 'Admin' : user.level;

            tableHTML += `
                <tr>
                    <td>${user.id}</td>
                    <td>${user.name || ''}</td>
                    <td>${user.username || ''}</td>
                    <td>${flagText}</td>
                    <td>${levelText}</td>
                    <td><img src="${imgSrc}" alt="user image" class="user-img" onerror="this.src='${defaultAvatar}'"></td>
                    <td>
                        <button class="action-btn edit-btn" onclick="updateUserPanel('${user.id}')">Edit</button>
                        <button class="action-btn delete-btn" onclick="deleteUser('${user.id}')">Delete</button>
                    </td>
                </tr>
            `;
        });

        tableHTML += `
                </tbody>
            </table>
        `;

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel(). Please include panel system.");
            return;
        }

        createPanel("listUsersPanel", "CC", "ctc", 800, 600, {
            bodyContent: tableHTML,
            noFooter: true,
            title: 'List Users'
        });

    } catch (error) {
        displayMessage(error.message, true);
    }
}

async function deleteUser(userId) {
    if (!confirm(`Are you sure you want to delete user ${userId}?`)) {
        return;
    }

    try {
        const response = await postto("https://j.kunok.com/aui.php?action=del", { id: userId });
        displayMessage(response.message || `User ${userId} deleted successfully`);
        closePanel("listUsersPanel");
        listUsersPanel();
    } catch (error) {
        console.error("Delete user error:", error);
        displayMessage(`Error deleting user ${userId}: ${error.message}`, true);
    }
}