function getGroupingData(schema) {
  const defaultResult = { 
    segments: [], 
    summary: { 
      segments: {}, 
      totals: {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredRequiredFields: []
      }
    } 
  };
  if (schema.grouping && typeof schema.grouping === "object") {
    return schema.grouping;
  }
  console.warn("Invalid data format: grouping is missing");
  return null;
}
function QSummary_01(data) {
  if (!data || !data.grouping || !data.grouping.name) {
    console.warn("Invalid data format: grouping is missing");
    return { segments: [], summary: { segments: {}, totals: {} } };
  }

  const result = { segments: [] };
  const summary = {
    segments: {},
    totals: {
      allFields: 0,
      requiredFields: 0,
      answeredFields: 0,
      answeredRequiredFields: 0,
      totalScore: 0,
      totalScoreWeighted: 0,
      maxPossibleScore: 0,
      maxPossibleScoreWeighted: 0,
      totalFiles: 0,
      unansweredRequiredFields: [],
    },
  };

  const segmentMap = {};

  Object.entries(data).forEach(([key, item]) => {
    if (key === "grouping" || item.type === "section") return;

    const segment = item.segment?.[0] || data.grouping?.name[0] || "default";
    const section = item.section;
    if (!segment || !section || !data[section]) {
      console.warn(
        `Invalid field ${key}: missing segment or section, or section not found in schema`,
      );
      return;
    }

    // Initialize segment/section if not exists
    if (!summary.segments[segment]) {
      const segmentIndex = data.grouping.name.indexOf(segment);
      summary.segments[segment] = {
        label: data.grouping.label?.en[segmentIndex] || segment,
        sections: {},
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [],
        },
      };
    }
    if (!summary.segments[segment].sections[section]) {
      summary.segments[segment].sections[section] = {
        label: data[section]?.label?.en || section,
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [],
        },
      };
    }

    const currentScore = calculateCurrentScore(item);
    const maxScore = calculateMaxScore(item);
    const isAnswered = isFieldAnswered(item);

    // Handle file counting for all field types
    let fileCount = 0;
    if (item.data_type === "attachment") {
      fileCount = countFiles(item);
    } else if (item.data_type === "radiotextatt") {
      // Create a temporary field object just for counting
      fileCount = countFiles({
        value: item.value?.att || "",
        data_type: "attachment",
      });
    } else if (item.data_type === "radiotextatt" && item.value?.att) {
      fileCount = countFiles({
        value: item.value.att,
        data_type: "attachment",
      });
    } else if (item.data_type === "radioatt" && item.value?.att) {
      // Added for radioatt
      fileCount = countFiles({
        value: item.value.att,
        data_type: "attachment",
      });
    }

    // Update totals at all levels
    const sectionTotals = summary.segments[segment].sections[section].totals;
    const segmentTotals = summary.segments[segment].totals;
    const globalTotals = summary.totals;

    // Section level
    sectionTotals.allFields++;
    if (item.required) sectionTotals.requiredFields++;
    if (isAnswered) sectionTotals.answeredFields++;
    if (item.required && isAnswered) sectionTotals.answeredRequiredFields++;
    sectionTotals.totalScore += isAnswered ? item.score || 1 : 0;
    sectionTotals.totalScoreWeighted += currentScore;
    sectionTotals.maxPossibleScore += item.score || 1;
    sectionTotals.maxPossibleScoreWeighted += maxScore;
    sectionTotals.totalFiles += fileCount;

    // Segment level (aggregate from sections)
    segmentTotals.allFields++;
    if (item.required) segmentTotals.requiredFields++;
    if (isAnswered) segmentTotals.answeredFields++;
    if (item.required && isAnswered) segmentTotals.answeredRequiredFields++;
    segmentTotals.totalScore += isAnswered ? item.score || 1 : 0;
    segmentTotals.totalScoreWeighted += currentScore;
    segmentTotals.maxPossibleScore += item.score || 1;
    segmentTotals.maxPossibleScoreWeighted += maxScore;
    segmentTotals.totalFiles += fileCount;

    // Global level (aggregate from segments)
    globalTotals.allFields++;
    if (item.required) globalTotals.requiredFields++;
    if (isAnswered) globalTotals.answeredFields++;
    if (item.required && isAnswered) globalTotals.answeredRequiredFields++;
    globalTotals.totalScore += isAnswered ? item.score || 1 : 0;
    globalTotals.totalScoreWeighted += currentScore;
    globalTotals.maxPossibleScore += item.score || 1;
    globalTotals.maxPossibleScoreWeighted += maxScore;
    globalTotals.totalFiles += fileCount;

    // Track unanswered required fields at all levels
    if (item.required && !isAnswered) {
      const fieldInfo = {
        key,
        label: getLabel(item, window.lang || "en") || key,
        segment,
        section,
      };

      sectionTotals.unansweredRequiredFields.push(fieldInfo);
      segmentTotals.unansweredRequiredFields.push(fieldInfo);
      globalTotals.unansweredRequiredFields.push(fieldInfo);
    }
  });

  // Build the result.segments structure (if needed for other purposes)
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    const segmentEntry = {
      name: segmentKey,
      label: segment.label,
      sections: [],
    };

    Object.entries(segment.sections).forEach(([sectionKey, section]) => {
      segmentEntry.sections.push({
        name: sectionKey,
        label: section.label,
        totals: section.totals,
      });
    });

    result.segments.push(segmentEntry);
  });

  return { result, summary };
}
function QSummary(data) {
  // Create a default empty result structure
  const defaultResult = { 
    segments: [], 
    summary: { 
      segments: {}, 
      totals: {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredRequiredFields: []
      }
    } 
  };
if (!data || typeof data !== 'object') {
    console.warn("Invalid data format: data is missing or not an object");
    return { segments: [], summary: { segments: {}, totals: {} } };
  }
// Handle profile-style grouping (where grouping is an object with arrays)
  if (data.grouping && typeof data.grouping === 'object' && !Array.isArray(data.grouping)) {
    // Normalize profile grouping structure to match gen structure
    const grouping = {
      name: Array.isArray(data.grouping.name) ? data.grouping.name : [data.grouping.name || "default"],
      type: Array.isArray(data.grouping.type) ? data.grouping.type : [data.grouping.type || "segment"],
      label: {
        en: Array.isArray(data.grouping.label?.en) ? data.grouping.label.en : [data.grouping.label?.en || "Default"],
        th: Array.isArray(data.grouping.label?.th) ? data.grouping.label.th : [data.grouping.label?.th || "ค่าเริ่มต้น"]
      }
    };
    data.grouping = grouping;
  }
  // If no grouping exists at all, create a default one
  else if (!data.grouping) {
    data.grouping = {
      name: ["default"],
      type: ["segment"],
      label: { 
        en: ["Default"], 
        th: ["ค่าเริ่มต้น"] 
      }
    };
  }
  // Initialize with default grouping if missing
  if (!data.grouping) {
    data.grouping = {
      name: ["default"],
      type: ["segment"],
      label: { en: ["Default"], th: ["ค่าเริ่มต้น"] }
    };
  }

  const result = { segments: [] };
  const summary = {
    segments: {},
    totals: {
      allFields: 0,
      requiredFields: 0,
      answeredFields: 0,
      answeredRequiredFields: 0,
      totalScore: 0,
      totalScoreWeighted: 0,
      maxPossibleScore: 0,
      maxPossibleScoreWeighted: 0,
      totalFiles: 0,
      unansweredRequiredFields: [],
    },
  };

  const segmentMap = {};

  Object.entries(data).forEach(([key, item]) => {
    if (key === "grouping" || item.type === "section") return;

    const segment = item.segment?.[0] || data.grouping?.name[0] || "default";
    const section = item.section;
    if (!segment || !section || !data[section]) {
      console.warn(
        `Invalid field ${key}: missing segment or section, or section not found in schema`,
      );
      return;
    }

    // Initialize segment/section if not exists
    if (!summary.segments[segment]) {
      const segmentIndex = data.grouping.name.indexOf(segment);
      summary.segments[segment] = {
        label: data.grouping.label?.en[segmentIndex] || segment,
        sections: {},
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [],
        },
      };
    }
    if (!summary.segments[segment].sections[section]) {
      summary.segments[segment].sections[section] = {
        label: data[section]?.label?.en || section,
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [],
        },
      };
    }

    const currentScore = calculateCurrentScore(item);
    const maxScore = calculateMaxScore(item);
    const isAnswered = isFieldAnswered(item);

    // Handle file counting for all field types
    let fileCount = 0;
    if (item.data_type === "attachment") {
      fileCount = countFiles(item);
    } else if (item.data_type === "radiotextatt") {
      // Create a temporary field object just for counting
      fileCount = countFiles({
        value: item.value?.att || "",
        data_type: "attachment",
      });
    } else if (item.data_type === "radiotextatt" && item.value?.att) {
      fileCount = countFiles({
        value: item.value.att,
        data_type: "attachment",
      });
    } else if (item.data_type === "radioatt" && item.value?.att) {
      // Added for radioatt
      fileCount = countFiles({
        value: item.value.att,
        data_type: "attachment",
      });
    }

    // Update totals at all levels
    const sectionTotals = summary.segments[segment].sections[section].totals;
    const segmentTotals = summary.segments[segment].totals;
    const globalTotals = summary.totals;

    // Section level
    sectionTotals.allFields++;
    if (item.required) sectionTotals.requiredFields++;
    if (isAnswered) sectionTotals.answeredFields++;
    if (item.required && isAnswered) sectionTotals.answeredRequiredFields++;
    sectionTotals.totalScore += isAnswered ? item.score || 1 : 0;
    sectionTotals.totalScoreWeighted += currentScore;
    sectionTotals.maxPossibleScore += item.score || 1;
    sectionTotals.maxPossibleScoreWeighted += maxScore;
    sectionTotals.totalFiles += fileCount;

    // Segment level (aggregate from sections)
    segmentTotals.allFields++;
    if (item.required) segmentTotals.requiredFields++;
    if (isAnswered) segmentTotals.answeredFields++;
    if (item.required && isAnswered) segmentTotals.answeredRequiredFields++;
    segmentTotals.totalScore += isAnswered ? item.score || 1 : 0;
    segmentTotals.totalScoreWeighted += currentScore;
    segmentTotals.maxPossibleScore += item.score || 1;
    segmentTotals.maxPossibleScoreWeighted += maxScore;
    segmentTotals.totalFiles += fileCount;

    // Global level (aggregate from segments)
    globalTotals.allFields++;
    if (item.required) globalTotals.requiredFields++;
    if (isAnswered) globalTotals.answeredFields++;
    if (item.required && isAnswered) globalTotals.answeredRequiredFields++;
    globalTotals.totalScore += isAnswered ? item.score || 1 : 0;
    globalTotals.totalScoreWeighted += currentScore;
    globalTotals.maxPossibleScore += item.score || 1;
    globalTotals.maxPossibleScoreWeighted += maxScore;
    globalTotals.totalFiles += fileCount;

    // Track unanswered required fields at all levels
    if (item.required && !isAnswered) {
      const fieldInfo = {
        key,
        label: getLabel(item, window.lang || "en") || key,
        segment,
        section,
      };

      sectionTotals.unansweredRequiredFields.push(fieldInfo);
      segmentTotals.unansweredRequiredFields.push(fieldInfo);
      globalTotals.unansweredRequiredFields.push(fieldInfo);
    }
  });

  // Build the result.segments structure (if needed for other purposes)
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    const segmentEntry = {
      name: segmentKey,
      label: segment.label,
      sections: [],
    };

    Object.entries(segment.sections).forEach(([sectionKey, section]) => {
      segmentEntry.sections.push({
        name: sectionKey,
        label: section.label,
        totals: section.totals,
      });
    });

    result.segments.push(segmentEntry);
  });

  return { result, summary };
}
function updateJumperHelperWithUnanswered(
  summary,
  data,
  targetDivId = "jumperhelper",
) {
  const lang = window.lang || "en";
  if (!summary || !summary.segments || !summary.totals) {
    console.error("Invalid summary object:", summary);
    return "";
  }

  const panelStack = window.panelStack || [];
  const formId =
    panelStack.length > 0 ? panelStack[panelStack.length - 1] : "gform";
  //const activeFormId =    formId === "gen" ? "gform" : formId === "profile" ? "pro" : formId;

  if (formId === "gen") window.activeFormId = "gform";
  else if (formId === "profile") window.activeFormId = "pro";

  // Enhanced getLabel function with better fallbacks
  const getLabel = (item) => {
    if (!item) return "";
    if (typeof item.label === "string") return item.label;
    if (item.label && (item.label[lang] || item.label.en)) {
      return item.label[lang] || item.label.en;
    }
    if (item.labels && (item.labels[lang] || item.labels.en)) {
      return item.labels[lang] || item.labels.en;
    }
    return item.key || "";
  };

  // Helper functions
  const generateId = (prefix) =>
    `${prefix}-${Math.random().toString(36).substr(2, 9)}`;
  const generateScrollLink = (key) =>
    `javascript:scrollToField('${key}', '${activeFormId}')`;

  // Generate all toggle IDs in advance
  const globalUnansweredToggleId = generateId("global-unanswered-toggle");
  const collapseAllId = generateId("collapse-all");
  const sectionToggleIds = {};
  const segmentToggleIds = {};
  const progressCircleId = generateId("progress-circle");

  // Pre-process to collect all toggle IDs
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    segmentToggleIds[segmentKey] = generateId(`segment-${segmentKey}-toggle`);
    Object.entries(segment.sections || {}).forEach(([sectionKey]) => {
      sectionToggleIds[sectionKey] = generateId(`section-${sectionKey}-toggle`);
    });
  });

  let html = `<div style='font-family: Arial, sans-serif;'>`;

  // Enhanced Collapse/Expand All controls
  html += `
    <div style='margin-bottom: 1px; display: flex; gap: 10px;'>
      <button id="${collapseAllId}"
              onclick="
                const isCollapsed = this.textContent.includes('${lang === "th" ? "Expand" : "Expand"}');
                document.querySelectorAll('[id^=segment-][id$=-toggle], [id^=section-][id$=-toggle]').forEach(el => {
                  el.style.display = isCollapsed ? 'block' : 'none';
                });
                document.querySelectorAll('[id^=segment-][id$=-toggle] + div, [id^=section-][id$=-toggle] + div').forEach(el => {
                  el.style.display = isCollapsed ? 'block' : 'none';
                });
                this.textContent = isCollapsed ?
                  '▼ ${lang === "th" ? "Collapse" : "Collapse All"}' :
                  '▶ ${lang === "th" ? "Expand" : "Expand All"}';
                document.getElementById('${globalUnansweredToggleId}').style.display = isCollapsed ? 'block' : 'none';
              "
              style="
                padding: 5px 10px;
                background: #4a90e2;
                color: white;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                flex: 1;
              ">
        ▼ ${lang === "th" ? "Collapse All" : "Collapse All"}
      </button>


    </div>
  `;

  // Grand Summary - unchanged from previous version
  const globalTotals = summary.totals;
  const fieldsPercent =
    globalTotals.allFields > 0
      ? Math.round((globalTotals.answeredFields / globalTotals.allFields) * 100)
      : 100;
  const requiredPercent =
    globalTotals.requiredFields > 0
      ? Math.round(
          (globalTotals.answeredRequiredFields / globalTotals.requiredFields) *
            100,
        )
      : 100;
  // Calculate progress values
  //globalTotals = summary.totals;
  const answeredRequired = globalTotals.answeredRequiredFields || 0;
  const totalRequired = globalTotals.requiredFields || 1; // Avoid division by zero
  const progressPercent = Math.round((answeredRequired / totalRequired) * 100);
  const progressColor =
    progressPercent === 100
      ? "#4CAF50"
      : progressPercent >= 50
        ? "#FFC107"
        : "#F44336";
  html += `
     <div style="display: flex; flex-direction: column; align-items: center; margin-bottom: 20px;">
       <svg width="120" height="120" id="${progressCircleId}-svg">
         <circle cx="60" cy="60" r="50" stroke="#eee" stroke-width="12" fill="none" />
         <circle id="${progressCircleId}" cx="60" cy="60" r="50"
                 stroke="${progressColor}" stroke-width="12" fill="none" stroke-linecap="round"
                 transform="rotate(-90 60 60)"
                 stroke-dasharray="314" stroke-dashoffset="${314 - (314 * progressPercent) / 100}" />
       </svg>
       <div id="${progressCircleId}-label" style="width: 160px; text-align: center; margin-top: 15px; font-size: 1.2em; font-weight: bold;">
         <div>${getLabel({ label: { en: "Progress", th: "Progress" } })}: <span id="${progressCircleId}-answered">${answeredRequired}</span>/<span id="${progressCircleId}-total">${totalRequired}</span></div>
         <div id="${progressCircleId}-percent">${progressPercent}%</div>
       </div>
     </div>
   `;
  html += `
    <div style='border: 2px solid #4a90e2; padding: 10px; margin-bottom: 20px; border-radius: 5px;'>
      <h4 style='margin: 0 0 5px 0;'>${getLabel({ label: { en: "Grand Summary", th: "Summary" } })}</h4>
      <div style='margin-bottom: 5px;'>
        ${getLabel({ label: { en: "Total Fields", th: "Total Fields" } })}: ${globalTotals.answeredFields}/${globalTotals.allFields}
        <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
          <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
        </div>
      </div>
      <div style='margin-bottom: 5px;'>
        ${getLabel({ label: { en: "Required", th: "Required" } })}: ${globalTotals.answeredRequiredFields}/${globalTotals.requiredFields}
        <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
          <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
        </div>
      </div>
      <div style='margin-bottom: 5px;'>
        ${getLabel({ label: { en: "Total Files", th: "Total Files" } })}: <b>${globalTotals.totalFiles}</b>
      </div>
  `;

  // Global unanswered fields with toggle
  if (globalTotals.unansweredRequiredFields?.length > 0) {
    html += `
      <div style='margin-top: 10px; color: orange;'>
        <a href="javascript:void(0)"
           onclick="document.getElementById('${globalUnansweredToggleId}').style.display =
                   document.getElementById('${globalUnansweredToggleId}').style.display === 'none' ? 'block' : 'none';
                   this.innerHTML = document.getElementById('${globalUnansweredToggleId}').style.display === 'none' ?
                   '${getLabel({ label: { en: "Unanswered Required Fields: ▼", th: "ฟิลด์ที่จำเป็นยังไม่ได้ตอบ: ▼" } })}' :
                   '${getLabel({ label: { en: "Unanswered Required Fields: ▶", th: "ฟิลด์ที่จำเป็นยังไม่ได้ตอบ: ▶" } })}';"
           style="text-decoration: none; color: orange;">
          ${getLabel({ label: { en: "Unanswered Required Fields: ▼", th: "ฟิลด์ที่จำเป็นยังไม่ได้ตอบ: ▼" } })}
        </a>
        <div id="${globalUnansweredToggleId}">
          <ul style='margin: 5px 0; padding-left: 20px;'>
            ${globalTotals.unansweredRequiredFields
              .map((field) => {
                const fieldData = data[field.key] || {};
                return `
              <li style='max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
              <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${activeFormId}')"
                       style='text-decoration: none; color: red;'
                       title="${getLabel(fieldData)}">
                      ${getLabel(fieldData)} (${field.section})
                    </a>
              </li>
            `;
              })
              .join("")}
          </ul>
        </div>
      </div>
    `;
  }

  html += `</div>`;

  // Segments with their sections - now with toggle support for segments
  html += "<div style='display: flex; flex-wrap: wrap; gap: 10px;'>";

  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    const segmentTotals = segment.totals || {};
    const segmentData = data[segmentKey] || {};
    const segmentToggleId = segmentToggleIds[segmentKey];
    const fieldsPercent =
      segmentTotals.allFields > 0
        ? Math.round(
            (segmentTotals.answeredFields / segmentTotals.allFields) * 100,
          )
        : 100;
    const requiredPercent =
      segmentTotals.requiredFields > 0
        ? Math.round(
            (segmentTotals.answeredRequiredFields /
              segmentTotals.requiredFields) *
              100,
          )
        : 100;

    const segmentLabel = getLabel(segmentData) || segmentKey;

    html += `
      <div style='border: 1px solid #ccc; padding: 10px; border-radius: 5px; min-width: 200px;'>
        <h4 style='margin: 0 0 5px 0; cursor: pointer;'
            onclick="
              const segToggle = document.getElementById('${segmentToggleId}');
              segToggle.style.display = segToggle.style.display === 'none' ? 'block' : 'none';
              this.querySelector('span').textContent = segToggle.style.display === 'none' ? '▶' : '▼';
            ">
          <span>▼</span> ${segmentLabel}
        </h4>
        <div id="${segmentToggleId}">
          <div style='margin-bottom: 5px;'>
            ${getLabel({ label: { en: "Fields", th: "ฟิลด์" } })}: ${segmentTotals.answeredFields}/${segmentTotals.allFields}
            <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
              <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
            </div>
          </div>
          <div style='margin-bottom: 5px;'>
            ${getLabel({ label: { en: "Required", th: "จำเป็นต้องตอบ" } })}: ${segmentTotals.answeredRequiredFields}/${segmentTotals.requiredFields}
            <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
              <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
            </div>
          </div>
    `;

    // Group unanswered fields by section
    const unansweredBySection = {};
    if (segmentTotals.unansweredRequiredFields?.length > 0) {
      segmentTotals.unansweredRequiredFields.forEach((field) => {
        if (!unansweredBySection[field.section]) {
          unansweredBySection[field.section] = [];
        }
        unansweredBySection[field.section].push(field);
      });
    }

    // Sections within this segment
    html += `<div style='margin-top: 10px;'>`;

    Object.entries(segment.sections || {}).forEach(([sectionKey, section]) => {
      const sectionData = data[sectionKey] || {};
      const sectionTotals = section.totals || {};
      const sectionToggleId = sectionToggleIds[sectionKey];
      const hasUnanswered = unansweredBySection[sectionKey]?.length > 0;

      const sectionLabel = getLabel(sectionData) || sectionKey;

      html += `
        <div style='margin-bottom: 8px;'>
          <div style='display: flex; justify-content: space-between; align-items: center; cursor: pointer;'
               onclick="
                 const secToggle = document.getElementById('${sectionToggleId}');
                 if (secToggle) {
                   secToggle.style.display = secToggle.style.display === 'none' ? 'block' : 'none';
                   this.querySelector('span').textContent = secToggle.style.display === 'none' ? '▶' : '▼';
                 }
               ">
            <div style='flex-grow: 1;'>
              ${sectionLabel} (${sectionTotals.answeredFields}/${sectionTotals.allFields})
            </div>
            ${hasUnanswered ? `<span>▼</span>` : ""}
          </div>
          ${
            hasUnanswered
              ? `
            <div id="${sectionToggleId}" style="display: block; margin-top: 4px;">
              <ul style='margin: 3px 0; padding-left: 15px; font-size: 0.9em;'>
                ${unansweredBySection[sectionKey]
                  .map((field) => {
                    const fieldData = data[field.key] || {};
                    return `
                  <li style='max-width: 180px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
                    <a href="${generateScrollLink(field.key)}"
                       style='text-decoration: none; color: red;'
                       title="${getLabel(fieldData)}">
                      ${getLabel(fieldData)}
                    </a>
                  </li>
                `;
                  })
                  .join("")}
              </ul>
            </div>
          `
              : ""
          }
        </div>
      `;
    });

    html += `</div></div></div>`;
  });

  html += `</div></div>`;

  // Add CSS styles
  html += `
    <style>
      [id$="-toggle"] {
        transition: all 0.3s ease;
      }
      a[onclick], [onclick] {
        cursor: pointer;
        user-select: none;
      }
      a[onclick]:hover {
        text-decoration: underline !important;
      }
      .section-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
      }
      #${collapseAllId}:hover {
        background: #3a80d2;
      }
      h4[onclick]:hover, [onclick]:hover {
        background-color: #f5f5f5;
      }
    </style>
  `;

  const jumperDiv = document.getElementById(targetDivId);
  if (jumperDiv) {
    jumperDiv.innerHTML = html;
  } else {
    console.error(`Target div ${targetDivId} not found`);
  }

  return html;
}
// Helper function to animate the progress circle
function animateProgressCircle(circleId, answered, total) {
  const circle = document.getElementById(circleId);
  const percentEl = document.getElementById(`${circleId}-percent`);
  const answeredEl = document.getElementById(`${circleId}-answered`);
  const totalEl = document.getElementById(`${circleId}-total`);
  const label = document.getElementById(`${circleId}-label`);

  if (!circle || !percentEl) {
    console.error("Progress elements not found");
    return;
  }

  // Ensure we have valid numbers
  answered = answered || 0;
  total = total || 1; // Avoid division by zero
  const percent = Math.round((answered / total) * 100);
  const color =
    percent === 100 ? "#4CAF50" : percent >= 50 ? "#FFC107" : "#F44336";

  // Update counts immediately
  if (answeredEl) answeredEl.textContent = answered;
  if (totalEl) totalEl.textContent = total;
  if (label) label.style.color = color;

  // Animate progress
  const radius = 50;
  const circumference = 2 * Math.PI * radius;
  const duration = 1000;
  const start = performance.now();

  function animate(time) {
    const elapsed = time - start;
    const progress = Math.min(elapsed / duration, 1);
    const value = percent * progress;
    const offset = circumference - (value / 100) * circumference;
    circle.style.strokeDashoffset = offset;
    circle.style.stroke = color;
    if (percentEl) percentEl.textContent = `${Math.round(value)}%`;
    if (progress < 1) {
      requestAnimationFrame(animate);
    }
  }

  requestAnimationFrame(animate);
}
function createResetButton_01(divId, key, field, input, lang, schema, fieldDiv) {
  const inputType = field.data_type || field.type;
  const resetBtn = document.createElement("button");
  resetBtn.textContent = getLabel(
    { label: { en: "reset", th: "reset" } },
    lang,
  );
  resetBtn.style.cssText = ` margin-left: 10px; padding: 4px 8px; cursor: pointer;
    background: #f44336;
           color: #ddd;
           border: none;
           border-radius: 10px;
           width: 38px;
           height: 15px;
           font-size: 12px;
           line-height: 1;
           cursor: pointer;
           margin-left: 5px;
           padding: 0px 15px;
           display: inline-flex;
           align-items: center;
           justify-content: center;
           transition: all 0.2s ease;
    `;
  resetBtn.onmouseover = function () {
    this.style.background = "#d32f2f";
    this.style.transform = "scale(1.1)";
  };
  resetBtn.onmouseout = function () {
    this.style.background = "#f44336";
    this.style.transform = "scale(1)";
  };
  resetBtn.onclick = () => {
    if (inputType === "radio") {
      schema[key].value = null;
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}"]`)
        .forEach((el) => {
          el.checked = false;
          fieldDiv.style.border = field.required ? "2px solid red" : "";
        });
    } else if (inputType === "radioatt") {
      // Reset radio and att components
      schema[key].value = { radio: null, att: "" };
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`)
        .forEach((el) => {
          el.checked = false;
        });
      const attInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-att`,
      );
      if (attInput) {
        attInput.value = "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    } else if (inputType === "checkbox") {
      schema[key].value = [];
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`)
        .forEach((el) => {
          el.checked = false;
          fieldDiv.style.border = field.required ? "2px solid red" : "";
        });
    } else if (inputType === "select") {
      schema[key].value = "";
      const inputEl = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      inputEl.value = "";
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    } else if (inputType === "attachment") {
      schema[key].value = "";
      const textInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      textInput.value = "";
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      document.getElementById(
        `${divId}-filecount-${encodeFieldKey(key)}`,
      ).textContent = `Files: 0`;
    } else if (inputType === "radiotext" || inputType === "radiotextatt") {
      // Reset both radio, text, and att components
      schema[key].value = { radio: null, text: "", att: "" };
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`)
        .forEach((el) => {
          el.checked = false;
        });
      const textInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-text`,
      );
      if (textInput) {
        textInput.value = "";
      }
      const attInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-att`,
      );
      if (attInput) {
        attInput.value = "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      //end radiotext case
    } else  if (inputType === "checktext" || inputType === "checktextatt") {
      // 1. Reset the data model - this is what gets saved
      field.value = { check: [], text: "", att: "" };
      
      // 2. Force uncheck ALL checkboxes in this field
      const checkboxes = fieldDiv.querySelectorAll(`input[type="checkbox"][name*="${encodeFieldKey(key)}"]`);
      checkboxes.forEach(cb => {
        cb.checked = false;
        // Force change event to update any listeners
        cb.dispatchEvent(new Event('change'));
      });
      
      // 3. Clear and hide text input
      const textInput = fieldDiv.querySelector(`#${divId}-${encodeFieldKey(key)}-text`);
      if (textInput) {
        textInput.value = "";
        textInput.style.display = "none";
        textInput.dispatchEvent(new Event('input'));
      }
      
      // 4. Clear attachments if present
      if (inputType === "checktextatt") {
        const attInput = fieldDiv.querySelector(`#${divId}-${encodeFieldKey(key)}-att`);
        if (attInput) {
          attInput.value = "";
          document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
          attInput.dispatchEvent(new Event('input'));
        }
      }
    } else {
      // Handle string, number, textarea types
      schema[key].value = inputType === "number" ? 0 : "";
      const inputEl = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      if (inputEl) {
        inputEl.value = inputType === "number" ? "0" : "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    }

    // Update score bar
    const scoreBar = fieldDiv.querySelector(".score-bar");
    const currentScore = calculateCurrentScore(field);
    const maxScore = calculateMaxScore(field);
    const weight = field.weights ? field.weights.join(", ") : "1";
    const fileCount =
      field.data_type === "attachment" ||
      field.data_type === "radiotextatt" ||
      field.data_type === "checktextatt"
        ? countFiles(field)
        : 0;
    let valueText = "";
    if (typeof field.value === "object" && field.value !== null) {
      try {
        valueText = JSON.stringify(field.value);
      } catch (e) {
        valueText = "[object]";
      }
    } else {
      valueText = field.value || "";
    }
    if (scoreBar) {
      if (window.debug)
        scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
      else
        scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }

    // Update jumper helper and section summaries
    if (document.getElementById("jumperhelper")) {
      const summary = QSummary(schema).summary;
      updateJumperHelperWithUnanswered(summary, schema);
    }
    updateSectionSummaries(divId, schema, lang);
  };

  return resetBtn;
}
function createResetButton02(divId, key, field, input, lang, schema, fieldDiv) {
  const inputType = field.data_type || field.type;
  const resetBtn = document.createElement("button");
  resetBtn.textContent = getLabel({ label: { en: "reset", th: "reset" } }, lang);
  resetBtn.style.cssText = `margin-left: 10px; padding: 4px 8px; cursor: pointer;
    background: #f44336;
    color: #ddd;
    border: none;
    border-radius: 10px;
    width: 38px;
    height: 15px;
    font-size: 12px;
    line-height: 1;
    cursor: pointer;
    margin-left: 5px;
    padding: 0px 15px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;`;

  // Button hover effects
  resetBtn.onmouseover = function() {
    this.style.background = "#d32f2f";
    this.style.transform = "scale(1.1)";
  };
  resetBtn.onmouseout = function() {
    this.style.background = "#f44336";
    this.style.transform = "scale(1)";
  };

  resetBtn.onclick = () => {
    // Common reset actions
    const resetCommon = () => {
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      updateFieldDisplay(fieldDiv, field);
      updateSectionSummaries(divId, schema, lang);
    };

    // Handle different field types
    switch(inputType) {
      case "radio":
        schema[key].value = null;
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}"]`).forEach(el => {
          el.checked = false;
        });
        break;

      case "radioatt":
        schema[key].value = { radio: null, att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        break;

      case "checkbox":
        schema[key].value = [];
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        break;

      case "select":
        schema[key].value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = "";
        break;

      case "attachment":
        schema[key].value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        break;

      case "radiotext":
      case "radiotextatt":
        schema[key].value = { radio: null, text: "", att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        if (inputType === "radiotextatt") {
          document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
          document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        }
        break;

      case "checktext":
      case "checktextatt":
        schema[key].value = { check: [], text: "", att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        if (inputType === "checktextatt") {
          document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
          document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        }
        break;

      default: // string, number, textarea
        schema[key].value = inputType === "number" ? 0 : "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = inputType === "number" ? "0" : "";
    }

    // Update score bar with current file count
    const scoreBar = fieldDiv.querySelector(".score-bar");
    if (scoreBar) {
      const currentScore = calculateCurrentScore(field);
      const maxScore = calculateMaxScore(field);
      const weight = field.weights ? field.weights.join(", ") : "1";
      const fileCount = countFiles(field); // This will now be accurate after reset
      
      let valueText = "";
      if (typeof field.value === "object" && field.value !== null) {
        try {
          valueText = JSON.stringify(field.value);
        } catch (e) {
          valueText = "[object]";
        }
      } else {
        valueText = field.value || "";
      }
      
      scoreBar.textContent = window.debug
        ? `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`
        : `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }

    // Update UI and summaries
    resetCommon();

    // Update jumper helper if present
    if (document.getElementById("jumperhelper")) {
      const summary = QSummary(schema).summary;
      updateJumperHelperWithUnanswered(summary, schema);
    }
  };

  return resetBtn;
}
function createResetButton03(divId, key, field, input, lang, schema, fieldDiv) {
  const inputType = field.data_type || field.type;
  const resetBtn = document.createElement("button");
  resetBtn.textContent = getLabel({ label: { en: "reset", th: "reset" } }, lang);
  resetBtn.style.cssText = `margin-left: 10px; padding: 4px 8px; cursor: pointer;
    background: #f44336;
    color: #ddd;
    border: none;
    border-radius: 10px;
    width: 38px;
    height: 15px;
    font-size: 12px;
    line-height: 1;
    cursor: pointer;
    margin-left: 5px;
    padding: 0px 15px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;`;

  resetBtn.onmouseover = function() {
    this.style.background = "#d32f2f";
    this.style.transform = "scale(1.1)";
  };
  resetBtn.onmouseout = function() {
    this.style.background = "#f44336";
    this.style.transform = "scale(1)";
  };

  resetBtn.onclick = () => {
    // Common reset actions
    const resetCommon = () => {
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      updateFieldDisplay(fieldDiv, field);
      updateSectionSummaries(divId, schema, lang);
    };

    // Handle different field types
    switch(inputType) {
      case "radio":
        schema[key].value = null;
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}"]`).forEach(el => {
          el.checked = false;
        });
        break;

      case "radioatt":
        schema[key].value = { radio: null, att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        break;

      case "checkbox":
        schema[key].value = [];
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        break;

      case "select":
        schema[key].value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = "";
        break;

      case "attachment":
        schema[key].value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        break;

      case "radiotext":
        schema[key].value = { radio: null, text: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block"; // Always show
        break;

      case "radiotextatt":
        schema[key].value = { radio: null, text: "", att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block"; // Always show
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).style.display = "block"; // Always show
        break;

      case "checktext":
        schema[key].value = { check: [], text: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block"; // Always show
        break;

      case "checktextatt":
        schema[key].value = { check: [], text: "", att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block"; // Always show
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).style.display = "block"; // Always show
        break;

      default: // string, number, textarea
        schema[key].value = inputType === "number" ? 0 : "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = inputType === "number" ? "0" : "";
    }

    // Update score bar with current file count
    const scoreBar = fieldDiv.querySelector(".score-bar");
    if (scoreBar) {
      const currentScore = calculateCurrentScore(field);
      const maxScore = calculateMaxScore(field);
      const weight = field.weights ? field.weights.join(", ") : "1";
      const fileCount = countFiles(field);
      
      let valueText = "";
      if (typeof field.value === "object" && field.value !== null) {
        try {
          valueText = JSON.stringify(field.value);
        } catch (e) {
          valueText = "[object]";
        }
      } else {
        valueText = field.value || "";
      }
      
      scoreBar.textContent = window.debug
        ? `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`
        : `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }

    // Update UI and summaries
    resetCommon();

    // Update jumper helper if present
    if (document.getElementById("jumperhelper")) {
      const summary = QSummary(schema).summary;
      updateJumperHelperWithUnanswered(summary, schema);
    }
  };

  return resetBtn;
}
function createResetButton(divId, key, field, input, lang, schema, fieldDiv) {
  const inputType = field.data_type || field.type;
  const resetBtn = document.createElement("button");
  resetBtn.textContent = getLabel({ label: { en: "reset", th: "reset" } }, lang);
  resetBtn.style.cssText = `margin-left: 10px; padding: 4px 8px; cursor: pointer;
    background: #f44336;
    color: #ddd;
    border: none;
    border-radius: 10px;
    width: 38px;
    height: 15px;
    font-size: 12px;
    line-height: 1;
    cursor: pointer;
    margin-left: 5px;
    padding: 0px 15px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s ease;`;

  // Button hover effects
  resetBtn.onmouseover = function() {
    this.style.background = "#d32f2f";
    this.style.transform = "scale(1.1)";
  };
  resetBtn.onmouseout = function() {
    this.style.background = "#f44336";
    this.style.transform = "scale(1)";
  };

  resetBtn.onclick = () => {
    // Common reset actions
    const resetCommon = () => {
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      updateFieldDisplay(fieldDiv, field);
      updateSectionSummaries(divId, schema, lang);
      
      // Force update jumper helper if present
      if (document.getElementById("jumperhelper")) {
        const summary = QSummary(schema).summary;
        updateJumperHelperWithUnanswered(summary, schema);
      }
      
      // Force update summan if present
      if (document.getElementById("summan")) {
        renderSchemaStructureToSumman(schema, divId, lang);
      }
    };

    // Handle different field types
    switch(inputType) {
      case "radio":
        schema[key].value = null;
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}"]`).forEach(el => {
          el.checked = false;
        });
        break;

      case "radioatt":
        schema[key].value = { radio: null, att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        break;

      case "checkbox":
        schema[key].value = [];
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        break;

      case "select":
        schema[key].value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = "";
        break;

      case "attachment":
        schema[key].value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        break;

      case "radiotext":
        schema[key].value = { radio: null, text: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block";
        break;

      case "radiotextatt":
        schema[key].value = { radio: null, text: "", att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}-radio"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).style.display = "block";
        break;

      case "checktext":
        schema[key].value = { check: [], text: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block";
        break;

      case "checktextatt":
        schema[key].value = { check: [], text: "", att: "" };
        document.querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`).forEach(el => {
          el.checked = false;
        });
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).value = "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).value = "";
        document.getElementById(`${divId}-filecount-${encodeFieldKey(key)}`).textContent = "Files: 0";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-text`).style.display = "block";
        document.getElementById(`${divId}-${encodeFieldKey(key)}-att`).style.display = "block";
        break;

      default: // string, number, textarea
        schema[key].value = inputType === "number" ? 0 : "";
        document.getElementById(`${divId}-${encodeFieldKey(key)}`).value = inputType === "number" ? "0" : "";
    }

    // Update score bar with current file count
    const scoreBar = fieldDiv.querySelector(".score-bar");
    if (scoreBar) {
      const currentScore = calculateCurrentScore(field);
      const maxScore = calculateMaxScore(field);
      const weight = field.weights ? field.weights.join(", ") : "1";
      const fileCount = countFiles(field);
      
      let valueText = "";
      if (typeof field.value === "object" && field.value !== null) {
        try {
          valueText = JSON.stringify(field.value);
        } catch (e) {
          valueText = "[object]";
        }
      } else {
        valueText = field.value || "";
      }
      
      scoreBar.textContent = window.debug
        ? `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`
        : `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }

    // Update UI and summaries
    resetCommon();
  };

  return resetBtn;
}
function updateFieldDisplay(fieldDiv, field) {
  // Update border based on required status and whether the field is answered
  fieldDiv.style.border =
    field.required && !isFieldAnswered(field) ? "2px solid red" : "";

  // Update score bar
  const scoreBar = fieldDiv.querySelector(".score-bar");
  if (scoreBar) {
    const currentScore = calculateCurrentScore(field);
    const maxScore = calculateMaxScore(field);
    const scoreAchieved = calculateScoreAchieved(field);

    const weight = field.weights ? field.weights.join(", ") : "1";

    // Get file count based on type
    //let fileCount = 0;
    fileCount = countFiles(field);
    let valueDisplay = "";

    if (field.data_type === "attachment") {
      fileCount = countFiles(field);
    } else if (field.data_type === "radiotextatt" && field.value?.att) {
      fileCount = countFiles(field);
    }

    let valueText = "";
    if (typeof field.value === "object" && field.value !== null) {
      try {
        valueText = JSON.stringify(field.value);
      } catch (e) {
        valueText = "[object]";
      }
    } else {
      valueText = field.value || "";
    }
    if (field.data_type === "radioatt") {
      valueDisplay = `Selected: ${field.value?.radio ?? "none"}`;
      if (fileCount > 0) {
        valueDisplay += `, Files: ${fileCount}`;
      }
    }
    if (window.debug) {
      scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, score achieved: ${scoreAchieved.toFixed(1)}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    } else {
      scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }
  }
}
function updateSectionSummaries(divId, schema, lang) {
  // Accumulate global totals from all segments and sections
  let globalTotalFields = 0;
  let globalAnsweredFields = 0;
  globalRequiredFields = 0;
  let globalAnsweredRequiredFields = 0;
  const segments = new Map();
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  } else {
    console.warn("No grouping defined in schema, using default segment");
    segments.set("default", { label: "Default", sections: new Map() });
  }

  // Map fields to segments and sections
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];
    const segment = field.segment?.[0] || schema.grouping?.name[0] || "default";
    if (!segments.has(segment)) {
      console.log(`Creating default segment for ${segment}`);
      segments.set(segment, { label: segment, sections: new Map() });
    }
    if (field.type === "section") {
      segments.get(segment).sections.set(key, getLabel(field, lang) || key);
    } else if (field.section) {
      if (!segments.get(segment).sections.has(field.section)) {
        if (schema[field.section]) {
          segments
            .get(segment)
            .sections.set(
              field.section,
              getLabel({ label: schema[field.section]?.label }, lang) ||
                field.section,
            );
        } else {
          console.warn(
            `Section ${field.section} for field ${key} not found in schema`,
          );
        }
      }
    } else {
      console.warn(`Field ${key} has no section assigned`);
    }
  }

  segments.forEach((segmentData, segmentId) => {
    console.log(`Processing segment: ${segmentId}`);
    segmentData.sections.forEach((sectionLabel, sectionId) => {
      console.log(`Processing section: ${sectionId} in segment: ${segmentId}`);
      let sectionTotals = {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredFields: [],
      };

      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (
          field.section !== sectionId ||
          !field.segment?.includes(segmentId)
        ) {
          console.log(
            `Skipping field ${key} in render: section=${field.section}, segment=${field.segment}, expected section=${sectionId}, segment=${segmentId}`,
          );
          continue;
        }

        console.log(
          `Rendering field: ${key}, type: ${field.data_type || field.type}`,
        );
        hasField = true;
        const fieldDiv = document.createElement("div");
        fieldDiv.className = "form-field";
        const encodedKey = encodeFieldKey(key);
        fieldDiv.id = `${divId}-field-${encodedKey}`;
        fieldDiv.style.cssText =
          "background-color: #f9f9f9; padding: 10px; margin-bottom: 10px; border-radius: 4px;";
        if (field.required && !isFieldAnswered(field)) {
          fieldDiv.style.border = "2px solid red";
        }
        if (!field.data_type && !field.type) {
          console.warn(`Field ${key} has no data_type or type defined`);
          continue;
        }
        globalTotalFields += sectionTotals.allFields;

        console.log(
          `Processing field: ${key}, type: ${field.data_type || field.type}`,
        );
        let fileCount = 0;

        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const scoreAchieved = calculateScoreAchieved(field);

        const isAnswered = isFieldAnswered(field);
        const isFile =
          field.data_type === "attachment" || field.type === "attachment";
        fileCount = isFile ? countFiles(field) : 0;
        if (field.data_type === "attachment") {
          fileCount = countFiles(field);
        } else if (field.data_type === "radiotextatt" && field.value?.att) {
          fileCount = countFiles({
            value: field.value.att,
            data_type: "attachment", // Treat it as attachment for counting
          });
        } else if (field.data_type === "radioatt" && field.value?.att) {
          // ADDED THIS CASE
          fileCount = countFiles({
            value: field.value.att,
            data_type: "attachment",
          });
        }
        sectionTotals.allFields++;
        if (field.required) sectionTotals.requiredFields++;
        if (isAnswered) sectionTotals.answeredFields++;
        if (field.required && isAnswered)
          sectionTotals.answeredRequiredFields++;
        sectionTotals.totalScore += isAnswered ? field.score || 1 : 0;
        sectionTotals.totalScoreWeighted += currentScore;
        sectionTotals.maxPossibleScore += field.score || 1;
        sectionTotals.maxPossibleScoreWeighted += maxScore;
        sectionTotals.totalFiles += fileCount;

        if (isFile) {
          console.log(
            `Attachment field ${key}: value="${field.value}", files=${fileCount}`,
          );
        }

        if (field.required && !isFieldAnswered(field)) {
          const encodedKey = encodeFieldKey(key);
          const fieldElement = document.getElementById(
            `${divId}-field-${encodedKey}`,
          );
          if (fieldElement) {
            sectionTotals.unansweredFields.push({
              key,
              label: getLabel(field, lang) || key,
              elementId: `${divId}-field-${encodedKey}`,
            });
          } else {
            console.warn(
              `Field ${key} not rendered in DOM (ID: ${divId}-field-${encodedKey}), skipping from unanswered list`,
            );
          }
        }
      }

      console.log(
        `Section ${sectionId} totals: ${JSON.stringify(sectionTotals, null, 2)}`,
      );

      const sectionSummary = document.getElementById(
        `${divId}-summary-${encodeFieldKey(segmentId)}-${encodeFieldKey(sectionId)}`,
      );
      if (sectionSummary) {
        //window.debug=true;
        const unansweredCount =
          sectionTotals.requiredFields - sectionTotals.answeredRequiredFields;
        // In the section summary HTML generation
        // In the section summary HTML generation

        a = window.debug
          ? `[field total: ${sectionTotals.allFields}, score: ${sectionTotals.totalScore}, achieved score: ${sectionTotals.totalScoreWeighted}, maxscore: ${sectionTotals.maxPossibleScoreWeighted}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]`
          : `[field total: ${sectionTotals.allFields}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]`;
        sectionSummary.innerHTML =
          `
          <summary style="cursor: pointer;">UnAnswered/Total/Required (${unansweredCount}/${sectionTotals.allFields}/${sectionTotals.requiredFields})</summary>
          <div style="padding: 8px;">
            ` +
          a +
          `${
            sectionTotals.unansweredFields.length > 0
              ? `
              <div style="margin-top: 10px; color: red;">
                <strong>Unanswered Required Fields:</strong>
                <ul style="margin: 5px 0; padding-left: 20px;">
                  ${sectionTotals.unansweredFields
                    .map(
                      (field) => `
                    <li style=" white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                      <a href="javascript:scrollToField('${encodeFieldKey(
                        field.key,
                      )}', '${divId}')"
                         style="text-decoration: none; color: red;"
                         title="${field.label}">
                        ${field.label}
                      </a>
                    </li>
                  `,
                    )
                    .join("")}
                </ul>
              </div>
            `
              : ""
          }
          </div>
        `;
        if (unansweredCount > 0) {
          sectionSummary.classList.add("unanswered");
        } else {
          sectionSummary.classList.remove("unanswered");
        }
      }

      const sectionHeader = document.getElementById(
        `${divId}-section-header-${encodeFieldKey(sectionId)}`,
      );
      if (sectionHeader) {
        const progressPercent =
          sectionTotals.requiredFields > 0
            ? Math.round(
                (sectionTotals.answeredRequiredFields /
                  sectionTotals.requiredFields) *
                  100,
              )
            : 100;
        globalAnsweredFields += sectionTotals.answeredRequiredFields;
        globalAnsweredRequiredFields += sectionTotals.answeredRequiredFields;
        globalRequiredFields += sectionTotals.requiredFields;
        sectionHeader.innerHTML = `
         
 
            [Progress: ${progressPercent}% (${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields})]
            <div style="width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;">
              <div style="width: ${progressPercent}%; background-color: #4caf50; height: 100%;"></div>
            </div>
         
        `;
      }
    });
  });
  // Update your summary div by id 'summan'
  const summanDiv = document.getElementById("summan");
  if (summanDiv) {
    summanDiv.textContent = `Total fields: ${globalTotalFields} | ${globalRequiredFields} Answered required fields: ${globalAnsweredFields}`;
    renderSchemaStructureToSumman(schema, divId, lang);
  }
}
function renderSchemaStructureToSumman(schema, divId, lang = "en") {
  const segments = new Map();
  let globalTotalFields = 0;
  let globalRequiredFields = 0;
  let globalAnsweredFields = 0;
  let globalAnsweredRequiredFields = 0;
  let globalunAnsweredRequiredFields = 0;
  let globalTotalFiles = 0; // Add global file counter

  // 1. Prepare segment structure
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  } else {
    segments.set("default", { label: "Default", sections: new Map() });
  }

  // 2. Organize fields into segments/sections and count
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];

    const segment =
      field.segment?.[0] || schema.grouping?.name?.[0] || "default";
    if (!segments.has(segment)) {
      segments.set(segment, { label: segment, sections: new Map() });
    }

    if (field.type === "section") {
      segments.get(segment).sections.set(key, {
        label: getLabel(field, lang) || key,
        fields: [],
      });
    } else if (field.section) {
      const section = field.section;
      if (!segments.get(segment).sections.has(section)) {
        if (schema[section]) {
          segments.get(segment).sections.set(section, {
            label: getLabel({ label: schema[section].label }, lang) || section,
            fields: [],
          });
        }
      }

      const sectionObj = segments.get(segment).sections.get(section);
      if (sectionObj) {
        sectionObj.fields.push({
          key,
          label: getLabel(field, lang) || key,
        });
      }

      // Count for global progress
      globalTotalFields++;
      const answered = isFieldAnswered(field);
      if (answered) globalAnsweredFields++;
      if (field.required) {
        globalRequiredFields++;
        if (answered) globalAnsweredRequiredFields++;
        else globalunAnsweredRequiredFields++;
      }
      // Count files for all field types that can have them
      // Add this file counting block:
      if (field.data_type === "attachment") {
        globalTotalFiles += countFiles(field);
      } else if (field.data_type === "radiotextatt" && field.value?.att) {
        globalTotalFiles += countFiles({
          value: field.value.att,
          data_type: "attachment",
        });
      } else if (field.data_type === "radioatt" && field.value?.att) {
        // Add this case
        globalTotalFiles += countFiles({
          value: field.value.att,
          data_type: "attachment",
        });
      }
    }
  }

  // 3. Calculate percentages and set window.allowsubmit
  const percentAnsweredAll =
    globalTotalFields > 0
      ? Math.round((globalAnsweredFields / globalTotalFields) * 100)
      : 100;

  const percentAnsweredRequired =
    globalRequiredFields > 0
      ? Math.round((globalAnsweredRequiredFields / globalRequiredFields) * 100)
      : 100;

  let status = "Not Active";
  window.allowsubmit = false;

  if (globalAnsweredRequiredFields === 0) {
    status = "⛔ Not Active";
  } else if (globalAnsweredRequiredFields < globalRequiredFields) {
    status = "🔄 In Progress";
  } else {
    status = "✅ READY";
    window.allowsubmit = true;
  }

  // 4. Build HTML 📊
  let html = `<div style="font-size: 0.95em; font-family: sans-serif;">
    <div style="margin-bottom: 15px;">
      <strong>Summary ${status}</strong><br><hr>
      Total fields: <b>${globalTotalFields}</b><br>
      Total required: <b>${globalRequiredFields}</b><br>
      Answered fields: <b>${globalAnsweredFields}</b><br>
      Answered required: <b>${globalAnsweredRequiredFields}</b><br>
        UnAnswered required: <b style="color: red">${globalunAnsweredRequiredFields}</b><br>

      Total files: <b>${globalTotalFiles}</b><br><br>  <!-- Add files count -->

      <div style="margin-bottom: 10px;">
        <div>[Answered: ${percentAnsweredAll}%]</div>
        <div style="background: #eee; height: 6px; border-radius: 5px;">
          <div style="width: ${percentAnsweredAll}%; background: #2196f3; height: 6px; border-radius: 5px;"></div>
        </div>
      </div>

      <div>
        <div>[Answered Required: ${percentAnsweredRequired}%]</div>
        <div style="background: #eee; height: 6px; border-radius: 5px;">
          <div style="width: ${percentAnsweredRequired}%; background: #4caf50; height: 6px; border-radius: 5px;"></div>
        </div>
      </div>
    </div>`;

  // 5. Segment → Section → Fields  📁 color: #2a5; 📂 color: #06c; 🔑 color: #c00;  ${field.key}:
  // Helper function to count required and answered fields in a section
  const countSectionStats = (fields) => {
    let required = 0;
    let answeredRequired = 0;
    let unAnsweredRequired = 0;

    fields.forEach((field) => {
      const fieldObj = schema[field.key];
      if (fieldObj?.required) {
        required++;
        if (isFieldAnswered(fieldObj)) {
          answeredRequired++;
        } else unAnsweredRequired++;
      }
    });

    return { required, answeredRequired, unAnsweredRequired };
  };

  // 5. Segment → Section → Fields with expand/collapse
  segments.forEach((seg, segId) => {
    // Count segment stats
    let segRequired = 0;
    let segAnsweredRequired = 0;
    let segTotalFields = 0;
    let segAnsweredFields = 0;
    let segUnanswered = 0;

    seg.sections.forEach((section) => {
      section.fields.forEach((field) => {
        const fieldObj = schema[field.key];
        segTotalFields++;
        if (isFieldAnswered(fieldObj)) segAnsweredFields++;
        if (fieldObj?.required) {
          segRequired++;
          if (isFieldAnswered(fieldObj)) segAnsweredRequired++;
          else segUnanswered++;
        }
      });
    });

    const segPercent =
      segRequired > 0
        ? Math.round((segAnsweredRequired / segRequired) * 100)
        : 100;
    const jumpsg = `${divId}-segment-${encodeFieldKey(segId)}`;
    html += `<div style="margin-bottom: 10px;">
    <div style="display: flex; align-items: center; gap: 4px;">
      <div style="text-decoration: none; flex-grow: 1;font-size: 14px; cursor: pointer;" onclick="toggleSection(this, 'seg_${segId}')"><b>${seg.label}</b> (${segAnsweredRequired}/${segRequired}, ${segPercent}%,Unanswered: <b style="color: red">${segUnanswered}</b>)
     </div> <a href="javascript:scrollTo('${jumpsg}', '${divId}')"
       style="margin-left: 10px; font-size: 0.8em; text-decoration: none;">
       <]
    </a>
    </div>
      <div id="seg_${segId}" style=" margin-left: 1px;">`;

    seg.sections.forEach((section, secId) => {
      // Count section stats
      const stats = countSectionStats(section.fields);
      const secPercent =
        stats.required > 0
          ? Math.round((stats.answeredRequired / stats.required) * 100)
          : 100;
      const jumpsc = `${divId}-section-${encodeFieldKey(secId)}`;

      html += `<div style="margin-left: 0px;">
      <div style="display: flex; align-items: center; gap: 1px;windows:400px;">
        <div style="font-size: 13px; flex-grow: 0;white-space: wrap;cursor: pointer;" onclick="toggleSection(this, 'sec_${segId}_${secId}')">
          <b>${section.label}</b> (${stats.answeredRequired}/${stats.required}, ${secPercent}%,: UnAnswered:<b style="color: red">${stats.unAnsweredRequired}</b>)
         </div><a href="javascript:scrollTo('${jumpsc}', '${divId}')"
       style="margin-left: 10px; font-size: 0.8em; text-decoration: none;">
       <]
    </a>
    </div>
        <div id="sec_${segId}_${secId}" style=" margin-left: 0px;">`;

      section.fields.forEach((field) => {
        const fieldObj = schema[field.key];
        const isAnswered = fieldObj ? isFieldAnswered(fieldObj) : false;
        const dotColor = isAnswered ? "green" : "red";
        const isRequired = fieldObj?.required ? " (Required)" : "";

        html += `<div style="margin-left: 0px; display: flex; align-items: center; min-height: 20px;">
        <span style="
            display: inline-block;
            min-width: 8px;
            min-height: 8px;
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: ${dotColor};
            margin-right: 4px;
            flex-shrink: 0;
        "></span>
          <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${divId}')"
             style="
             font-size: 14px;
               text-decoration: none;
               color: inherit;
               white-space: nowrap;
               overflow: hidden;
               text-overflow: ellipsis;
               max-width: 90%;
           "
             title="${field.label}">
              ${field.label}${isRequired}
          </a>
        </div>`;
      });

      html += `</div></div>`;
    });

    html += `</div></div>`;
  });

  html += `</div>`;

  // Add the toggle function to the window object
  window.toggleSection = function (element, id) {
    const content = document.getElementById(id);
    if (content.style.display === "none") {
      content.style.display = "block";
      element.innerHTML = element.innerHTML.replace("▶", "▼");
    } else {
      content.style.display = "none";
      element.innerHTML = element.innerHTML.replace("▼", "▶");
    }
  };

  // 6. Output to #summan
  const summaryDiv = document.getElementById("summan");
  if (summaryDiv) {
    summaryDiv.innerHTML = html;
  } else {
    console.warn("Div with id 'summan' not found");
  }
}
function scrollToField(key, formId = "gform") {
  const id = `${formId}-field-${encodeURIComponent(key)}`;
  const el = document.getElementById(id);
  if (el) {
    el.scrollIntoView({ behavior: "smooth", block: "center" });
    el.style.background = "rgba(255, 165, 0, 0.3)";
    setTimeout(() => (el.style.background = ""), 2000);
  } else {
    console.warn("Scroll target not found:", id);
  }
}
function scrollTo(key, formId = "gform") {
  const id = `${encodeURIComponent(key)}`;
  const el = document.getElementById(id);
  if (el) {
    el.scrollIntoView({ behavior: "smooth", block: "start" });
    el.style.border = "3px solid #ffa500";
    setTimeout(() => (el.style.border = ""), 2000);
  } else {
    console.warn("Scroll target not found:", id);
  }
}