(function () {
  function createDraggablePanel(options = {}) {
    const {
      bottom = 100,
      left = 10,
      width = 200,
      collapsedHeight = 70,
      expandedHeight = 200,
      avatars = [], // array of image URLs
      showExternalAvatar = true,
      externalAvatarURL = 'https://i.pravatar.cc/41',
    } = options;

    // Create panel
    const panel = document.createElement('div');
    panel.id = 'miniPanel_' + Math.random().toString(36).substr(2, 5);
    panel.style.cssText = `
      position: absolute;
      bottom: ${bottom}px;
      left: ${left}px;
      width: ${width}px;
      height: ${collapsedHeight}px;
      background: #f0f0f0;
      border: 1px solid #aaa;
      border-radius: 8px 8px 0 0;
      padding: 10px;
      cursor: move;
      transition: height 0.3s;
      display: flex;
      align-items: flex-end;
      flex-direction: row;
      gap: 10px;
      z-index: 1000;
    `;

    // Add avatars to panel
    avatars.forEach((url, i) => {
      const avatar = document.createElement('div');
      avatar.className = 'avatar';
      avatar.title = `Avatar ${i + 1}`;
      avatar.style.cssText = `
        width: 40px;
        height: 40px;
        background: url('${url}') no-repeat center/cover;
        border-radius: 50%;
        cursor: grab;
      `;
      avatar.addEventListener('mousedown', (e) => {
        panel.dispatchEvent(new MouseEvent('mousedown', e));
      });
      panel.appendChild(avatar);
    });

    // Add toggle button
    const btn = document.createElement('button');
    btn.innerText = 'Expand';
    btn.className = 'toggle-btn';
    btn.style.cssText = `
      margin-left: auto;
      align-self: flex-end;
      cursor: pointer;
      background: #ccc;
      padding: 5px 10px;
      border: none;
      border-radius: 5px;
    `;
    btn.onclick = () => {
      const isExpanded = panel.classList.toggle('expanded');
      panel.style.height = isExpanded ? expandedHeight + 'px' : collapsedHeight + 'px';
    };
    panel.appendChild(btn);

    document.body.appendChild(panel);

    // Draggable function
    function makeDraggableBottomAnchor(el) {
      let startX = 0, startY = 0, startLeft = 0, startBottom = 0, isDragging = false;

      el.addEventListener('mousedown', (e) => {
        isDragging = true;
        startX = e.clientX;
        startY = e.clientY;
        const style = window.getComputedStyle(el);
        startLeft = parseInt(style.left, 10);
        startBottom = parseInt(style.bottom, 10);
        el.style.cursor = 'grabbing';
        e.preventDefault();
      });

      document.addEventListener('mousemove', (e) => {
        if (isDragging) {
          const dx = e.clientX - startX;
          const dy = e.clientY - startY;
          el.style.left = (startLeft + dx) + 'px';
          el.style.bottom = (startBottom - dy) + 'px';
        }
      });

      document.addEventListener('mouseup', () => {
        isDragging = false;
        el.style.cursor = 'grab';
      });
    }

    makeDraggableBottomAnchor(panel);

    // Optional external avatar
    if (showExternalAvatar) {
      const ext = document.createElement('div');
      ext.className = 'external-avatar';
      ext.style.cssText = `
        position: absolute;
        left: ${left + width + 10}px;
        bottom: ${bottom}px;
        width: 40px;
        height: 40px;
        background: url('${externalAvatarURL}') no-repeat center/cover;
        border-radius: 50%;
        cursor: pointer;
      `;
      ext.title = 'Double-click me!';
      ext.ondblclick = () => alert('hi');
      makeDraggableBottomAnchor(ext);
      document.body.appendChild(ext);
    }

    return panel.id; // Optional: return ID for further usage
  }

  // Expose globally
  window.createDraggablePanel = createDraggablePanel;
})();
