function genform(divId, data) {
  const container = document.getElementById(divId);
  if (!container) {
    console.error(`Div with ID ${divId} not found`);
    return;
  }

  // Store reference to original data (no deep copy)
  container._formData = data;
  const lang = window.lang || "en";
// Normalize radiotext values on load
  for (const key in data) {
    if (key === "grouping" || data[key].type === "section") continue;
    const field = data[key];
    if (field.data_type === "radiotext" && field.value) {
      if (
        field.value.radio === "" ||
        field.value.text === "Others (please specify)/อื่น ๆ (โปรดระบุ)"
      ) {
        field.value = { radio: null, text: "" };
      }
    }
  }
  // Initialize container HTML
  container.innerHTML = `
        <div style="font-family: Arial, sans-serif;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="font-size: 0.7em;">${getLabel({ label: { en: "Form in", th: "FORM" } }, lang)} ${divId}</h3>
            </div>
            <div id="${divId}-fields"></div>
            <div style="display:none;">
                <textarea id="${divId}-input" placeholder="Enter JSON schema"></textarea>
                <button id="${divId}-copy-btn">Copy to Schema</button>
                <textarea id="${divId}-output" readonly></textarea>
            </div>
        </div>
    `;

  const fieldsDiv = document.getElementById(`${divId}-fields`);
  const schema = container._formData;

  // Organize segments and sections
  const segments = new Map();
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  }

  // Map fields to segments and sections
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];
    const segment = field.segment?.[0] || schema.grouping?.name[0] || "default";
    if (!segments.has(segment)) {
      segments.set(segment, { label: segment, sections: new Map() });
    }
    if (field.type === "section") {
      segments.get(segment).sections.set(key, getLabel(field, lang) || key);
    } else if (field.section) {
      if (!segments.get(segment).sections.has(field.section)) {
        if (schema[field.section]) {
          segments
            .get(segment)
            .sections.set(
              field.section,
              getLabel({ label: schema[field.section]?.label }, lang) ||
                field.section,
            );
        } else {
          console.warn(
            `Section ${field.section} for field ${key} not found in schema`,
          );
        }
      }
    } else {
      console.warn(`Field ${key} has no section assigned`);
    }
  }

  // Generate form structure
  segments.forEach((segmentData, segmentId) => {
    const segDetails = document.createElement("details");
    segDetails.className = "segment";
    segDetails.style.cssText =
      "border-radius: 5px;background-color: rgba(255,255,255,.1);backdrop-filter: blur(5px);border: 2px solid #4a90e2; border-radius: 8px; margin-bottom: 10px; padding: 10px;";
    segDetails.id = `${divId}-segment-${encodeFieldKey(segmentId)}`;
    segDetails.open = true;
    segDetails.innerHTML = `<summary style="font-weight: bold; cursor: pointer;">${getLabel({ label: { en: "Segment", th: "ส่วน" } }, lang)}: ${segmentData.label}</summary>`;

    const segDiv = document.createElement("div");
    segDiv.style.padding = "10px";

    segmentData.sections.forEach((sectionLabel, sectionId) => {
      const secDetails = document.createElement("details");
      secDetails.className = "section";
      secDetails.id = `${divId}-section-${encodeFieldKey(sectionId)}`;
      secDetails.style.cssText =
        "box-shadow: 0 14px 12px rgba(0, 0, 0, 0.2);border-radius: 5px;background-color: rgba(215,255,255,.2);backdrop-filter: blur(20px);border: 1px solid green; border-radius: 5px; margin: 8px 0; padding: 8px;";
      secDetails.open = true;

      // Calculate section totals for progress bar
      let sectionTotals = {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredFields: [],
      };

      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (field.section !== sectionId || !field.segment?.includes(segmentId))
          continue;
        if (!field.data_type && !field.type) {
          console.warn(`Field ${key} has no data_type or type defined`);
          continue;
        }

        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const isAnswered = isFieldAnswered(field);
        const isFile = field.data_type === "attachment";
        const fileCount = isFile ? countFiles(field) : 0;

        sectionTotals.allFields++;
        if (field.required) sectionTotals.requiredFields++;
        if (isAnswered) sectionTotals.answeredFields++;
        if (field.required && isAnswered)
          sectionTotals.answeredRequiredFields++;
        sectionTotals.totalScore += isAnswered ? field.score || 1 : 0;
        sectionTotals.totalScoreWeighted += currentScore;
        sectionTotals.maxPossibleScore += field.score || 1;
        sectionTotals.maxPossibleScoreWeighted += maxScore;
        sectionTotals.totalFiles += fileCount;
        if (field.required && !isFieldAnswered(field)) {
          if (
            document.getElementById(`${divId}-field-${encodeFieldKey(key)}`)
          ) {
            sectionTotals.unansweredFields.push({
              key,
              label: getLabel(field, lang) || key,
            });
          } else {
            console.warn(
              `Field ${key} not rendered in DOM, skipping from unanswered list`,
            );
          }
        }
      }

      // Section title with progress bar, styled like section-summary
      const progressPercent =
        sectionTotals.requiredFields > 0
          ? Math.round(
              (sectionTotals.answeredRequiredFields /
                sectionTotals.requiredFields) *
                100,
            )
          : 100;
      secDetails.innerHTML = `
        <summary style="cursor: pointer; margin-top: 10px; padding: 8px; background-color: rgba(200,190,200,.1); border-radius: 4px; font-size: 0.9em;">
          <span style="font-weight: bold;">${sectionLabel}</span>
        </summary>
        <div id="${divId}-section-header-${encodeFieldKey(sectionId)}" style="margin-bottom: 8px; padding: 8px; background-color:rgba(200,200,200,.1);backdrop-filter: blur(10px); ">
          <div style="font-weight: bold;"></div>
          <div style="margin-top: 5px;  background-color:rgb(163, 61, 61);border-radius: 4px;">
            [Progress: ${progressPercent}% (${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields})]
            <div style="width: 100%;  height: 15px; border-radius: 5px; overflow: hidden;">
              <div style="width: ${progressPercent}%; background-color: #4caf50; height: 100%;"></div>
            </div>
          </div>
        </div>
      `;

      const secDiv = document.createElement("div");
      secDiv.style.padding = "8px";

      let hasField = false;
      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (field.section !== sectionId || !field.segment?.includes(segmentId))
          continue;

        hasField = true;
        const fieldDiv = document.createElement("div");
        fieldDiv.className = "form-field";
        fieldDiv.id = `${divId}-field-${encodeFieldKey(key)}`;
        fieldDiv.style.cssText =
          "border-radius: 5px;background-color: rgba(255,255,255,.3);backdrop-filter: blur(10px); padding: 10px; margin-bottom: 10px; border-radius: 4px;";
        if (field.required && !isFieldAnswered(field)) {
          fieldDiv.style.border = "2px solid red";
        }

        const label = document.createElement("label");

        label.style.cssText =
          "margin:10px 0px;box-shadow: 0 14px 12px rgba(0, 0, 0, 0.1);padding:10px 5px;border-radius: 5px;background-color: rgba(255,255,255,.1);backdrop-filter: blur(5px);font-weight: 300; margin-right: 6px; display: block; padding-bottom: 5px;";
        if (field.required) label.classList.add("required-field");
        label.innerHTML =
          (getLabel(field, lang) || key) + (field.required ? " *" : "");
        label.htmlFor = `${divId}-${encodeFieldKey(key)}`;
        if (field.score !== undefined || field.weights) {
          const scoreDisplay = document.createElement("span");
          scoreDisplay.className = "score-display";
          const maxScore = calculateMaxScore(field);
          const currentScore = calculateCurrentScore(field);
          if (typeof window.score !== "undefined" && window.score) {
            scoreDisplay.textContent = `(Score: ${currentScore}/${maxScore})`;
          }
          label.appendChild(scoreDisplay);
        }
        fieldDiv.appendChild(label);

        const input = createInputElement(divId, key, field, lang, schema);
        fieldDiv.appendChild(input);
        fieldDiv.appendChild(
          createResetButton(divId, key, field, input, lang, schema, fieldDiv),
        );

        // Add score bar area
        const scoreBar = document.createElement("div");
        scoreBar.className = "score-bar";
        scoreBar.style.cssText =
          "margin-top: 5px; font-size: 0.9em; color: #555;";
        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const weight = field.weights ? field.weights.join(", ") : "1";
        const fileCount =
          field.data_type === "attachment" ? countFiles(field) : 0;
        //const value = field.value || "";
        //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
        let valueText = "";
if (typeof field.value === "object" && field.value !== null) {
  try {
    valueText = JSON.stringify(field.value);
  } catch (e) {
    valueText = "[object]";
  }
} else {
  valueText = field.value || "";
}
if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
        fieldDiv.appendChild(scoreBar);

        secDiv.appendChild(fieldDiv);
      }

      if (hasField) {
        // Add section summary
        const sectionSummary = document.createElement("details");
        sectionSummary.className = "section-summary";
        sectionSummary.id = `${divId}-summary-${encodeFieldKey(segmentId)}-${encodeFieldKey(sectionId)}`;
        sectionSummary.style.cssText =
          "margin-top: 10px; padding: 8px; background-color: #f0f0f0; border-radius: 4px; font-size: 0.9em;";
        sectionSummary.open = true;
        const unansweredCount =
          sectionTotals.requiredFields - sectionTotals.answeredRequiredFields;
        sectionSummary.innerHTML = `
          <summary style="cursor: pointer;">UnAnswered/Total/TotalRequired (${unansweredCount}/${sectionTotals.allFields}/${sectionTotals.requiredFields})</summary>
          <div style="padding: 8px;">
            [field total: ${sectionTotals.allFields}, score: ${sectionTotals.totalScore}, achieved score: ${sectionTotals.totalScoreWeighted}, maxscore: ${sectionTotals.maxPossibleScoreWeighted}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]
            ${
              sectionTotals.unansweredFields.length > 0
                ? `
              <div style="margin-top: 10px; color: red;">
                <strong>Unanswered Required Fields:</strong>
                <ul style="margin: 5px 0; padding-left: 20px;">
                  ${sectionTotals.unansweredFields
                    .filter((field) =>
                      document.getElementById(
                        `${divId}-field-${encodeFieldKey(field.key)}`,
                      ),
                    )
                    .map(
                      (field) => `
                      <li style="max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                        <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${divId}')"
                           style="text-decoration: none; color: red;"
                           title="${field.label}">
                          ${field.label}
                        </a>
                      </li>
                    `,
                    )
                    .join("")}
                </ul>
              </div>
            `
                : ""
            }
          </div>
        `;
        secDiv.appendChild(sectionSummary);
        segDiv.appendChild(secDetails);
        secDetails.appendChild(secDiv);
      }
    });

    segDetails.appendChild(segDiv);
    fieldsDiv.appendChild(segDetails);
  });

  // Handle schema copy button
  const copyBtn = document.getElementById(`${divId}-copy-btn`);
  copyBtn.addEventListener("click", () => {
    const inputEl = document.getElementById(`${divId}-input`);
    try {
      const newSchema = JSON.parse(inputEl.value);
      genform(divId, newSchema);
    } catch (e) {
      console.error(`Invalid JSON in ${divId}-input: ${e.message}`);
      alert(
        getLabel(
          {
            label: {
              en: `Invalid JSON: ${e.message}`,
              th: `JSON ไม่ถูกต้อง: ${e.message}`,
            },
          },
          lang,
        ),
      );
    }
  });

  listenform(divId);
}
//

// Helper function to check if a field is answered
function isFieldAnsweredO(field) {
  const inputType = field.data_type || field.type;
  console.log(
    `isFieldAnswered: field=${field.key || "unknown"}, type=${inputType}, value=${
      field.value
    }, answered=${!!field.value}`,
  );
  if (inputType === "checkbox") {
    return Array.isArray(field.value) && field.value.length > 0;
  } else if (inputType === "radio" || inputType === "select") {
    return field.value !== null && field.value !== "";
  } else if (inputType === "attachment" || inputType === "file") {
    return field.value && countFiles(field) > 0;
  } else if (inputType === "radiotext") {
  const isValidRadio = field.value?.radio !== null && field.value?.radio !== undefined && field.value?.radio !== "";
  const isValidText = field.value?.text && field.value.text.trim() !== "" && 
                      field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
  return isValidRadio && isValidText;
}
 else {
    return field.value !== null && field.value !== "";
  }
}
function isFieldAnsweredAA(field) {
  const inputType = field.data_type || field.type;
  const fieldKey = field.key || "unknown";

  if (inputType === "checkbox") {
    return Array.isArray(field.value) && field.value.length > 0;
  } else if (inputType === "radio" || inputType === "select") {
    return field.value !== null && field.value !== "";
  } else if (inputType === "attachment" || inputType === "file") {
    return field.value && countFiles(field) > 0;
  } else if (inputType === "radiotext") {
    const isValidRadio = field.value?.radio !== null &&
                         field.value?.radio !== undefined &&
                         field.value?.radio !== "";
    const isValidText = field.value?.text &&
                        field.value.text.trim() !== "" &&
                        field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    return isValidRadio && isValidText;
  } else {
    return field.value !== null && field.value !== "";
  }
}
function isFieldAnswered(field) {
  const inputType = field.data_type || field.type;

  if (inputType === "checkbox") {
    return Array.isArray(field.value) && field.value.length > 0;

  } else if (inputType === "radio" || inputType === "select") {
    return field.value !== null && field.value !== "";

  } else if (inputType === "attachment" || inputType === "file") {
    return field.value && countFiles(field) > 0;

  } else if (inputType === "radiotext") {
    const hasValidRadio = field.value?.radio !== null && field.value?.radio !== undefined && field.value?.radio !== "";
    const hasValidText = field.value?.text && field.value.text.trim() !== "" && 
                         field.value.text !== "Others (please specify)/อื่น ๆ (โปรดระบุ)";
    return hasValidRadio || hasValidText;
  } else {
    return field.value !== null && field.value !== "";
  }
}

// Helper function to count files
function countFiles(field) {
  if (!field || !field.value || typeof field.value !== "string") {
    console.log(
      `countFiles: Invalid field or value for ${field?.key || "unknown"}, value=${
        field?.value
      }, returning 0`,
    );
    return 0;
  }
  const files = field.value.split(",").filter((f) => f.trim()).length;
  console.log(
    `countFiles: field=${field.key || "unknown"}, value="${
      field.value
    }", count=${files}`,
  );
  return files;
}

// Helper function to create input elements
function createInputElement(divId, key, field, lang, schema) {
  const inputType = field.data_type || field.type;
  let input;

  if (inputType === "textarea") {
    input = document.createElement("textarea");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.value = field.value || "";
    input.style.width = "260px";
  } else if (inputType === "radio") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}" value="${val}" ${field.value === val ? "checked" : ""}> ${opt}`;
      input.appendChild(radioLabel);
    });
  } else if (inputType === "checkbox") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";
    getOptions(field, lang).forEach((opt, idx) => {
      const checkLabel = document.createElement("label");
      checkLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      checkLabel.innerHTML = `<input type="checkbox" name="${divId}-${encodeFieldKey(key)}[]" value="${val}" ${field.value && field.value.includes(val) ? "checked" : ""}> ${opt}`;
      input.appendChild(checkLabel);
    });
  } else if (inputType === "select") {
    input = document.createElement("select");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.style.width = "260px";
    getOptions(field, lang).forEach((opt, idx) => {
      const option = document.createElement("option");
      const value = idx === 0 ? "" : opt;
      option.value = value;
      option.textContent = opt;
      option.selected = field.value === value;
      input.appendChild(option);
    });
  } else if (inputType === "attachment") {
    input = document.createElement("div");
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.id = `${divId}-${encodeFieldKey(key)}`;
    textInput.value = field.value || "";
    textInput.placeholder = getLabel(
      {
        label: {
          en: "Enter file names (comma-separated)",
          th: "ป้อนชื่อไฟล์ (คั่นด้วยเครื่องหมายจุลภาค)",
        },
      },
      lang,
    );
    textInput.style.width = "260px";
    input.appendChild(textInput);
    const fileCount = document.createElement("div");
    fileCount.id = `${divId}-filecount-${encodeFieldKey(key)}`;
    fileCount.style.marginTop = "5px";
    fileCount.textContent = `Files: ${countFiles(field)}`;
    input.appendChild(fileCount);
    const buttonDiv = document.createElement("div");
    buttonDiv.style.marginTop = "5px";
    const addFileBtn = document.createElement("button");
    addFileBtn.textContent = getLabel(
      { label: { en: "Add file", th: "เพิ่มไฟล์" } },
      lang,
    );
    addFileBtn.style.cssText =
      "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
    addFileBtn.onclick = () => {
      toggleUploadPanel();
      updateFileListFromInput(`${divId}-${encodeFieldKey(key)}`);
      const count = countFiles(schema[key]);
      fileCount.textContent = `Files: ${count}`;
      schema[key].value = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      ).value;
      const fieldDiv = document.getElementById(
        `${divId}-field-${encodeFieldKey(key)}`,
      );
      if (fieldDiv) {
        fieldDiv.style.border =
          field.required && !isFieldAnswered(field) ? "2px solid red" : "";
      }
      const scoreBar = fieldDiv?.querySelector(".score-bar");
      const currentScore = calculateCurrentScore(field);
      const maxScore = calculateMaxScore(field);
      const weight = field.weights ? field.weights.join(", ") : "1";
      const fileCountVal = countFiles(field);
      const value = field.value || "";
      if (scoreBar) {
        //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCountVal}, value: ${value}]`;
        if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
      }
      if (document.getElementById("jumperhelper")) {
        const summary = QSummary(schema).summary;
        updateJumperHelperWithUnanswered(summary, schema);
      }
      updateSectionSummaries(divId, schema, lang);
    };
    buttonDiv.appendChild(addFileBtn);
    const countFilesBtn = document.createElement("button");
    countFilesBtn.textContent = getLabel(
      { label: { en: "Save Attachment", th: "นับไฟล์" } },
      lang,
    );
    countFilesBtn.style.cssText =
      "margin-left: 10px; padding: 4px 8px; cursor: pointer;";
    countFilesBtn.onclick = () => {
      schema[key].value = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      ).value;
      fileCount.textContent = `Files: ${countFiles(schema[key])}`;
      const fieldDiv = document.getElementById(
        `${divId}-field-${encodeFieldKey(key)}`,
      );
      if (fieldDiv) {
        fieldDiv.style.border =
          field.required && !isFieldAnswered(field) ? "2px solid red" : "";
      }
      const scoreBar = fieldDiv?.querySelector(".score-bar");
      const currentScore = calculateCurrentScore(field);
      const maxScore = calculateMaxScore(field);
      const weight = field.weights ? field.weights.join(", ") : "1";
      const fileCountVal = countFiles(field);
      const value = field.value || "";
      if (scoreBar) {
        //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCountVal}, value: ${value}]`;
        if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
      }
      if (document.getElementById("jumperhelper")) {
        const summary = QSummary(schema).summary;
        updateJumperHelperWithUnanswered(summary, schema);
      }
      updateSectionSummaries(divId, schema, lang);
    };
    buttonDiv.appendChild(countFilesBtn);
    input.appendChild(buttonDiv);
  } else if (inputType === "mixed") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";

    // Create a container for the mixed inputs
    const mixedContainer = document.createElement("div");
    mixedContainer.style.display = "flex";
    mixedContainer.style.gap = "10px";
    mixedContainer.style.alignItems = "center";

    // Add radio/select/checkbox options
    if (field.options) {
      const optionType = field.optionType || "radio"; // default to radio

      if (optionType === "radio") {
        getOptions(field, lang).forEach((opt, idx) => {
          const radioLabel = document.createElement("label");
          radioLabel.style.cssText =
            "margin-right: 8px; display: flex; align-items: center;";
          const val = idx;
          radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}-option" value="${val}" ${field.value?.option === val ? "checked" : ""}> ${opt}`;
          mixedContainer.appendChild(radioLabel);
        });
      } else if (optionType === "select") {
        const select = document.createElement("select");
        select.name = `${divId}-${encodeFieldKey(key)}-option`;
        getOptions(field, lang).forEach((opt, idx) => {
          const option = document.createElement("option");
          option.value = idx;
          option.textContent = opt;
          option.selected = field.value?.option === idx;
          select.appendChild(option);
        });
        mixedContainer.appendChild(select);
      } else if (optionType === "checkbox") {
        getOptions(field, lang).forEach((opt, idx) => {
          const checkLabel = document.createElement("label");
          checkLabel.style.cssText =
            "margin-right: 8px; display: flex; align-items: center;";
          const val = idx;
          checkLabel.innerHTML = `<input type="checkbox" name="${divId}-${encodeFieldKey(key)}-option[]" value="${val}" ${field.value?.options?.includes(val) ? "checked" : ""}> ${opt}`;
          mixedContainer.appendChild(checkLabel);
        });
      }
    }

    // Add text input
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.name = `${divId}-${encodeFieldKey(key)}-text`;
    textInput.value = field.value?.text || "";
    textInput.placeholder = field.textPlaceholder || "Enter text";
    textInput.style.flexGrow = "1";

    mixedContainer.appendChild(textInput);
    input.appendChild(mixedContainer);

    // Add file input if needed
    if (field.allowFile) {
      const fileInput = document.createElement("input");
      fileInput.type = "file";
      fileInput.name = `${divId}-${encodeFieldKey(key)}-file`;
      fileInput.style.marginTop = "5px";
      input.appendChild(fileInput);
    }

    return input;
  } else if (inputType === "radiotext") {
    input = document.createElement("div");
    input.style.cssText = "display: flex; flex-direction: column; gap: 5px;";

    // Create radio buttons container
    const radioContainer = document.createElement("div");
    radioContainer.style.cssText =
      "display: flex; flex-direction: column; gap: 5px;";

    // Add radio options
    getOptions(field, lang).forEach((opt, idx) => {
      const radioLabel = document.createElement("label");
      radioLabel.style.cssText =
        "margin-right: 8px; display: flex; align-items: center;";
      const val = idx;
      radioLabel.innerHTML = `<input type="radio" name="${divId}-${encodeFieldKey(key)}-radio" value="${val}" ${
        field.value && field.value.radio === val ? "checked" : ""
      }> ${opt}`;
      radioContainer.appendChild(radioLabel);
    });

    input.appendChild(radioContainer);

    // Add text input
    const textInput = document.createElement("input");
    textInput.type = "text";
    textInput.id = `${divId}-${encodeFieldKey(key)}-text`;
    textInput.value = (field.value && field.value.text) || "";
    textInput.placeholder = getLabel(
      { label: { en: "Enter text", th: "ป้อนข้อความ" } },
      lang,
    );
    textInput.style.width = "260px";
    textInput.style.marginTop = "5px";
    input.appendChild(textInput);
  } else {
    input = document.createElement("input");
    input.id = `${divId}-${encodeFieldKey(key)}`;
    input.type = inputType === "string" ? "text" : inputType || "text";
    input.value = field.value || (inputType === "number" ? 0 : "");
    input.style.width = "260px";
  }

  return input;
}
// Helper function to get options (assumed to exist in original code)
function getOptions(field, lang) {
  return field.options?.[lang] || field.options?.en || [];
}
function getOptions(field, lang) {
  if (Array.isArray(field.options)) {
    return field.options; // Handle non-bilingual options
  }
  return field.options?.[lang] || field.options?.en || [];
}

// Helper function to get label (assumed to exist in original code)
function getLabel(field, lang) {
  return field.label?.[lang] || field.label?.en || "";
}

// Helper function to encode field key (assumed to exist in original code)
function encodeFieldKey(key) {
  return encodeURIComponent(key).replace(/'/g, "&#39;");
}
// Helper function to create reset button
function createResetButton(divId, key, field, input, lang, schema, fieldDiv) {
  const inputType = field.data_type || field.type;
  const resetBtn = document.createElement("button");
  resetBtn.textContent = getLabel(
    { label: { en: "Reset", th: "รีเซ็ต" } },
    lang,
  );
  resetBtn.style.cssText =
    "margin-left: 10px; padding: 4px 8px; cursor: pointer;";

  resetBtn.onclick = () => {
    if (inputType === "radio") {
      schema[key].value = null;
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}"]`)
        .forEach((el) => {
          el.checked = false;
          fieldDiv.style.border = field.required ? "2px solid red" : "";
        });
    } else if (inputType === "checkbox") {
      schema[key].value = [];
      document
        .querySelectorAll(`input[name="${divId}-${encodeFieldKey(key)}[]"]`)
        .forEach((el) => {
          el.checked = false;
          fieldDiv.style.border = field.required ? "2px solid red" : "";
        });
    } else if (inputType === "select") {
      schema[key].value = "";
      const inputEl = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      inputEl.value = "";
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    } else if (inputType === "attachment") {
      schema[key].value = "";
      const textInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      textInput.value = "";
      fieldDiv.style.border = field.required ? "2px solid red" : "";
      document.getElementById(
        `${divId}-filecount-${encodeFieldKey(key)}`,
      ).textContent = `Files: 0`;
    } else if (inputType === "radiotext") {
      // Reset both radio and text components
      schema[key].value = { radio: null, text: "" };
      document
        .querySelectorAll(
          `input[name="${divId}-${encodeFieldKey(key)}-radio"]`,
        )
        .forEach((el) => {
          el.checked = false;
        });
      const textInput = document.getElementById(
        `${divId}-${encodeFieldKey(key)}-text`,
      );
      if (textInput) {
        textInput.value = "";
      }
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    } else {
      schema[key].value = inputType === "number" ? 0 : "";
      const inputEl = document.getElementById(
        `${divId}-${encodeFieldKey(key)}`,
      );
      inputEl.value = inputType === "number" ? 0 : "";
      fieldDiv.style.border = field.required ? "2px solid red" : "";
    }

    // Update score bar
    const scoreBar = fieldDiv.querySelector(".score-bar");
    const currentScore = calculateCurrentScore(field);
    const maxScore = calculateMaxScore(field);
    const weight = field.weights ? field.weights.join(", ") : "1";
    const fileCount = field.data_type === "attachment" ? countFiles(field) : 0;
    let valueText = "";
    if (typeof field.value === "object" && field.value !== null) {
      try {
        valueText = JSON.stringify(field.value);
      } catch (e) {
        valueText = "[object]";
      }
    } else {
      valueText = field.value || "";
    }
    if (scoreBar) {
      //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
      if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    }

    // Update jumper helper and section summaries
    if (document.getElementById("jumperhelper")) {
      const summary = QSummary(schema).summary;
      updateJumperHelperWithUnanswered(summary, schema);
    }
    updateSectionSummaries(divId, schema, lang);
  };

  return resetBtn;
}

// Form input event listener
function listenform(divId) {
  const schema = document.getElementById(divId)._formData;
  const lang = window.lang || "en";

  for (const key in schema) {
    if (key === "grouping" || schema[key].type === "section") continue;
    const field = schema[key];
    const inputType = field.data_type || field.type;
    const inputId = `${divId}-${encodeFieldKey(key)}`;
    const fieldDiv = document.getElementById(
      `${divId}-field-${encodeFieldKey(key)}`,
    );

    if (["string", "textarea", "text", "number"].includes(inputType)) {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("input", () => {
          schema[key].value =
            inputType === "number"
              ? parseFloat(inputEl.value) || 0
              : inputEl.value.trim();
          fieldDiv.style.border =
            field.required && !isFieldAnswered(field) ? "2px solid red" : "";
          // Update score bar
          const scoreBar = fieldDiv.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCount =
            field.data_type === "attachment" ? countFiles(field) : 0;
          const value = field.value || "";
          //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
          // Update jumper helper and section summaries
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "radio") {
      document.querySelectorAll(`input[name="${inputId}"]`).forEach((radio) => {
        radio.addEventListener("change", () => {
          schema[key].value = parseInt(radio.value);
          fieldDiv.style.border =
            field.required && !isFieldAnswered(field) ? "2px solid red" : "";
          // Update score bar
          const scoreBar = fieldDiv.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCount =
            field.data_type === "attachment" ? countFiles(field) : 0;
          const value = field.value || "";
          // scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          // Update jumper helper and section summaries
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      });
    } else if (inputType === "checkbox") {
      document
        .querySelectorAll(`input[name="${inputId}[]"]`)
        .forEach((checkbox) => {
          checkbox.addEventListener("change", () => {
            const checked = Array.from(
              document.querySelectorAll(`input[name="${inputId}[]"]:checked`),
            ).map((el) => parseInt(el.value));
            schema[key].value = checked;
            fieldDiv.style.border =
              field.required && !isFieldAnswered(field) ? "2px solid red" : "";
            // Update score bar
            const scoreBar = fieldDiv.querySelector(".score-bar");
            const currentScore = calculateCurrentScore(field);
            const maxScore = calculateMaxScore(field);
            const weight = field.weights ? field.weights.join(", ") : "1";
            const fileCount =
              field.data_type === "attachment" ? countFiles(field) : 0;
            const value = field.value || "";
            scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
            // Update jumper helper and section summaries
            if (document.getElementById("jumperhelper")) {
              const summary = QSummary(schema).summary;
              updateJumperHelperWithUnanswered(summary, schema);
            }
            updateSectionSummaries(divId, schema, lang);
          });
        });
    } else if (inputType === "select") {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("change", () => {
          schema[key].value = inputEl.value;
          fieldDiv.style.border =
            field.required && !isFieldAnswered(field) ? "2px solid red" : "";
          // Update score bar
          const scoreBar = fieldDiv.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCount =
            field.data_type === "attachment" ? countFiles(field) : 0;
          const value = field.value || "";
          scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          // Update jumper helper and section summaries
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "attachment") {
      const inputEl = document.getElementById(inputId);
      if (inputEl) {
        inputEl.addEventListener("input", () => {
          schema[key].value = inputEl.value;
          const fileCount = document.getElementById(
            `${divId}-filecount-${encodeFieldKey(key)}`,
          );
          fileCount.textContent = `Files: ${countFiles(schema[key])}`;
          const fieldDiv = document.getElementById(
            `${divId}-field-${encodeFieldKey(key)}`,
          );
          if (fieldDiv) {
            fieldDiv.style.border =
              field.required && !isFieldAnswered(field) ? "2px solid red" : "";
          }
          const scoreBar = fieldDiv?.querySelector(".score-bar");
          const currentScore = calculateCurrentScore(field);
          const maxScore = calculateMaxScore(field);
          const weight = field.weights ? field.weights.join(", ") : "1";
          const fileCountVal = countFiles(field);
          const value = field.value || "";
          if (scoreBar) {
            // scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCountVal}, value: ${value}]`;
            if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${value}]`;
          }
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    } else if (inputType === "radiotext") {
      // Handle radio button changes
      document
        .querySelectorAll(`input[name="${inputId}-radio"]`)
        .forEach((radio) => {
          radio.addEventListener("change", () => {
            if (!schema[key].value) schema[key].value = { radio: null, text: "" };
            schema[key].value.radio = parseInt(radio.value);
            updateFieldDisplay(fieldDiv, field);
            if (document.getElementById("jumperhelper")) {
              const summary = QSummary(schema).summary;
              updateJumperHelperWithUnanswered(summary, schema);
            }
            updateSectionSummaries(divId, schema, lang);
          });
        });

      // Handle text input changes
      const textInput = document.getElementById(`${inputId}-text`);
      if (textInput) {
        textInput.addEventListener("input", () => {
if (!schema[key].value) schema[key].value = { radio: null, text: "" };
          schema[key].value.text = textInput.value.trim();
          updateFieldDisplay(fieldDiv, field);
          if (document.getElementById("jumperhelper")) {
            const summary = QSummary(schema).summary;
            updateJumperHelperWithUnanswered(summary, schema);
          }
          updateSectionSummaries(divId, schema, lang);
        });
      }
    }
  }

  // Initial update of jumper helper and section summaries
  if (document.getElementById("jumperhelper")) {
    const summary = QSummary(schema).summary;
    updateJumperHelperWithUnanswered(summary, schema);
  }
  updateSectionSummaries(divId, schema, lang);
}
function renderSchemaStructureToSumman(schema, divId, lang = "en") {
  const segments = new Map();

  // 1. Prepare segment structure
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  } else {
    segments.set("default", { label: "Default", sections: new Map() });
  }

  // 2. Organize fields into segments/sections
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];
    const segment = field.segment?.[0] || schema.grouping?.name[0] || "default";

    if (!segments.has(segment)) {
      segments.set(segment, { label: segment, sections: new Map() });
    }

    if (field.type === "section") {
      segments.get(segment).sections.set(key, {
        label: getLabel(field, lang) || key,
        fields: [],
      });
    } else if (field.section) {
      const section = field.section;
      if (!segments.get(segment).sections.has(section)) {
        if (schema[section]) {
          segments.get(segment).sections.set(section, {
            label: getLabel({ label: schema[section].label }, lang) || section,
            fields: [],
          });
        }
      }
      const sectionObj = segments.get(segment).sections.get(section);
      if (sectionObj) {
        sectionObj.fields.push({
          key,
          label: getLabel(field, lang) || key,
        });
      }
    } else {
      console.warn(`Field "${key}" has no section`);
    }
  }

  // 3. Build HTML content //📁color: #2a5; color: #06c; 🔑color: #c00;
  let html = `<div style="font-size: 0.95em; font-family: sans-serif;">`;
  segments.forEach((seg, segId) => {
    html += `<div style="margin-bottom: 10px;">
      <div style="font-weight: bold; "> Segment: ${seg.label}</div>`;

    seg.sections.forEach((section, secId) => {
      html += `<div style="margin-left: 10px; ">
        <div style="font-weight: bold;"> Section: ${section.label}</div>`;

      section.fields.forEach((field) => {
        html += `<div style="margin-left: 5px;">
           <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${divId}')"
               style="text-decoration: none; "
               title="${field.label}">
            ${field.key}: ${field.label}
          </a>
        </div>`;
      });

      html += `</div>`; // close section
    });

    html += `</div>`; // close segment
  });
  html += `</div>`;

  // 4. Insert into the #summan div
  const summaryDiv = document.getElementById("summan");
  if (summaryDiv) {
    summaryDiv.innerHTML += html;
  } else {
    console.warn("Div with id 'summan' not found");
  }
}
function renderSchemaStructureToSumman(schema, divId, lang = "en") {
  const segments = new Map();
  let globalTotalFields = 0;
  let globalRequiredFields = 0;
  let globalAnsweredFields = 0;
  let globalAnsweredRequiredFields = 0;

  // 1. Prepare segment structure
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  } else {
    segments.set("default", { label: "Default", sections: new Map() });
  }

  // 2. Organize fields into segments/sections and count
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];

    const segment =
      field.segment?.[0] || schema.grouping?.name?.[0] || "default";
    if (!segments.has(segment)) {
      segments.set(segment, { label: segment, sections: new Map() });
    }

    if (field.type === "section") {
      segments.get(segment).sections.set(key, {
        label: getLabel(field, lang) || key,
        fields: [],
      });
    } else if (field.section) {
      const section = field.section;
      if (!segments.get(segment).sections.has(section)) {
        if (schema[section]) {
          segments.get(segment).sections.set(section, {
            label: getLabel({ label: schema[section].label }, lang) || section,
            fields: [],
          });
        }
      }

      const sectionObj = segments.get(segment).sections.get(section);
      if (sectionObj) {
        sectionObj.fields.push({
          key,
          label: getLabel(field, lang) || key,
        });
      }

      // Count for global progress
      globalTotalFields++;
      const answered = isFieldAnswered(field);
      if (answered) globalAnsweredFields++;
      if (field.required) {
        globalRequiredFields++;
        if (answered) globalAnsweredRequiredFields++;
      }
    }
  }

  // 3. Calculate percentages and set window.allowsubmit
  const percentAnsweredAll =
    globalTotalFields > 0
      ? Math.round((globalAnsweredFields / globalTotalFields) * 100)
      : 100;

  const percentAnsweredRequired =
    globalRequiredFields > 0
      ? Math.round((globalAnsweredRequiredFields / globalRequiredFields) * 100)
      : 100;

  let status = "Not Active";
  window.allowsubmit = false;

  if (globalAnsweredRequiredFields === 0) {
    status = "⛔ Not Active";
  } else if (globalAnsweredRequiredFields < globalRequiredFields) {
    status = "🔄 In Progress";
  } else {
    status = "✅ READY";
    window.allowsubmit = true;
  }

  // 4. Build HTML 📊
  let html = `<div style="font-size: 0.95em; font-family: sans-serif;">
    <div style="margin-bottom: 15px;">
      <strong>Summary ${status}</strong><br><hr>
      Total fields: <b>${globalTotalFields}</b><br>
      Total required: <b>${globalRequiredFields}</b><br>
      Answered fields: <b>${globalAnsweredFields}</b><br>
      Answered required: <b>${globalAnsweredRequiredFields}</b><br><br>

      <div style="margin-bottom: 10px;">
        <div>[Answered: ${percentAnsweredAll}%]</div>
        <div style="background: #eee; height: 6px; border-radius: 5px;">
          <div style="width: ${percentAnsweredAll}%; background: #2196f3; height: 6px; border-radius: 5px;"></div>
        </div>
      </div>

      <div>
        <div>[Answered Required: ${percentAnsweredRequired}%]</div>
        <div style="background: #eee; height: 6px; border-radius: 5px;">
          <div style="width: ${percentAnsweredRequired}%; background: #4caf50; height: 6px; border-radius: 5px;"></div>
        </div>
      </div>
    </div>`;

  // 5. Segment → Section → Fields  📁 color: #2a5; 📂 color: #06c; 🔑 color: #c00;  ${field.key}:
  segments.forEach((seg, segId) => {
    html += `<div style="margin-bottom: 10px;">
      <div style="font-weight: normal; "> Segment: <b>${seg.label}</b></div>`;
    seg.sections.forEach((section, secId) => {
      html += `<div style="margin-left: 0px; ">
        <div style="font-weight: normal;"> Section: <b>${section.label}</b></div>`;
      section.fields.forEach((field) => {
        html += `<div style="margin-left: 5px;">
           <a href="javascript:scrollToField('${encodeFieldKey(field.key)}', '${divId}')"
               style="text-decoration: none;color: #222;  "
               title="${field.label}">
            ${field.label}
          </a>
        </div>`;
      });
      html += `</div>`;
    });
    html += `</div>`;
  });

  html += `</div>`;

  // 6. Output to #summan
  const summaryDiv = document.getElementById("summan");
  if (summaryDiv) {
    summaryDiv.innerHTML = html;
  } else {
    console.warn("Div with id 'summan' not found");
  }
}
function updateFieldDisplay(fieldDiv, field) {
  // Update border based on required status and whether the field is answered
  fieldDiv.style.border =
    field.required && !isFieldAnswered(field) ? "2px solid red" : "";

  // Update score bar
  const scoreBar = fieldDiv.querySelector(".score-bar");
  if (scoreBar) {
    const currentScore = calculateCurrentScore(field);
    const maxScore = calculateMaxScore(field);
    const weight = field.weights ? field.weights.join(", ") : "1";
    const fileCount = field.data_type === "attachment" ? countFiles(field) : 0;
    let valueText = "";
    if (typeof field.value === "object" && field.value !== null) {
      try {
        valueText = JSON.stringify(field.value);
      } catch (e) {
        valueText = "[object]";
      }
    } else {
      valueText = field.value || "";
    }
    //scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
    if (window.debug) scoreBar.textContent = `[score: ${currentScore}, weight: ${weight}, maxscore: ${maxScore}, required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`; else 
scoreBar.textContent = `[Required: ${field.required ? "yes" : "no"}, Files: ${fileCount}, value: ${valueText}]`;
  }
}

function updateSumman(schema) {
  if (!schema) return;
  let totalFields = 0;
  let answeredFields = 0;

  for (const key in schema) {
    if (key === "grouping" || schema[key].type === "section") continue;
    totalFields++;
    if (isFieldAnswered(schema[key])) {
      answeredFields++;
    }
  }

  const summanDiv = document.getElementById("summan");
  if (summanDiv) {
    summanDiv.textContent = `Total fields: ${totalFields} | Answered: ${answeredFields}`;
  }
}

// Helper function to update section summaries
function updateSectionSummaries(divId, schema, lang) {
  // Accumulate global totals from all segments and sections
  let globalTotalFields = 0;
  let globalAnsweredFields = 0;
  globalRequiredFields = 0;
  let globalAnsweredRequiredFields = 0;
  const segments = new Map();
  if (schema.grouping) {
    schema.grouping.name.forEach((name, i) => {
      if (schema.grouping.type[i] === "segment") {
        segments.set(name, {
          label: getLabel({ label: schema.grouping.label }, lang)[i] || name,
          sections: new Map(),
        });
      }
    });
  } else {
    console.warn("No grouping defined in schema, using default segment");
    segments.set("default", { label: "Default", sections: new Map() });
  }

  // Map fields to segments and sections
  for (const key in schema) {
    if (key === "grouping") continue;
    const field = schema[key];
    const segment = field.segment?.[0] || schema.grouping?.name[0] || "default";
    if (!segments.has(segment)) {
      console.log(`Creating default segment for ${segment}`);
      segments.set(segment, { label: segment, sections: new Map() });
    }
    if (field.type === "section") {
      segments.get(segment).sections.set(key, getLabel(field, lang) || key);
    } else if (field.section) {
      if (!segments.get(segment).sections.has(field.section)) {
        if (schema[field.section]) {
          segments
            .get(segment)
            .sections.set(
              field.section,
              getLabel({ label: schema[field.section]?.label }, lang) ||
                field.section,
            );
        } else {
          console.warn(
            `Section ${field.section} for field ${key} not found in schema`,
          );
        }
      }
    } else {
      console.warn(`Field ${key} has no section assigned`);
    }
  }

  segments.forEach((segmentData, segmentId) => {
    console.log(`Processing segment: ${segmentId}`);
    segmentData.sections.forEach((sectionLabel, sectionId) => {
      console.log(`Processing section: ${sectionId} in segment: ${segmentId}`);
      let sectionTotals = {
        allFields: 0,
        requiredFields: 0,
        answeredFields: 0,
        answeredRequiredFields: 0,
        totalScore: 0,
        totalScoreWeighted: 0,
        maxPossibleScore: 0,
        maxPossibleScoreWeighted: 0,
        totalFiles: 0,
        unansweredFields: [],
      };

      for (const key in schema) {
        if (key === "grouping" || schema[key].type === "section") continue;
        const field = schema[key];
        if (
          field.section !== sectionId ||
          !field.segment?.includes(segmentId)
        ) {
          console.log(
            `Skipping field ${key} in render: section=${field.section}, segment=${field.segment}, expected section=${sectionId}, segment=${segmentId}`,
          );
          continue;
        }

        console.log(
          `Rendering field: ${key}, type: ${field.data_type || field.type}`,
        );
        hasField = true;
        const fieldDiv = document.createElement("div");
        fieldDiv.className = "form-field";
        const encodedKey = encodeFieldKey(key);
        fieldDiv.id = `${divId}-field-${encodedKey}`;
        fieldDiv.style.cssText =
          "background-color: #f9f9f9; padding: 10px; margin-bottom: 10px; border-radius: 4px;";
        if (field.required && !isFieldAnswered(field)) {
          fieldDiv.style.border = "2px solid red";
        }
        if (!field.data_type && !field.type) {
          console.warn(`Field ${key} has no data_type or type defined`);
          continue;
        }
        globalTotalFields += sectionTotals.allFields;

        console.log(
          `Processing field: ${key}, type: ${field.data_type || field.type}`,
        );
        const currentScore = calculateCurrentScore(field);
        const maxScore = calculateMaxScore(field);
        const isAnswered = isFieldAnswered(field);
        const isFile =
          field.data_type === "attachment" || field.type === "attachment";
        const fileCount = isFile ? countFiles(field) : 0;

        sectionTotals.allFields++;
        if (field.required) sectionTotals.requiredFields++;
        if (isAnswered) sectionTotals.answeredFields++;
        if (field.required && isAnswered)
          sectionTotals.answeredRequiredFields++;
        sectionTotals.totalScore += isAnswered ? field.score || 1 : 0;
        sectionTotals.totalScoreWeighted += currentScore;
        sectionTotals.maxPossibleScore += field.score || 1;
        sectionTotals.maxPossibleScoreWeighted += maxScore;
        sectionTotals.totalFiles += fileCount;

        if (isFile) {
          console.log(
            `Attachment field ${key}: value="${field.value}", files=${fileCount}`,
          );
        }

        if (field.required && !isFieldAnswered(field)) {
          const encodedKey = encodeFieldKey(key);
          const fieldElement = document.getElementById(
            `${divId}-field-${encodedKey}`,
          );
          if (fieldElement) {
            sectionTotals.unansweredFields.push({
              key,
              label: getLabel(field, lang) || key,
                            elementId: `${divId}-field-${encodedKey}`

            });
          } else {
            console.warn(
              `Field ${key} not rendered in DOM (ID: ${divId}-field-${encodedKey}), skipping from unanswered list`,
            );
          }
        }
      }

      console.log(
        `Section ${sectionId} totals: ${JSON.stringify(sectionTotals, null, 2)}`,
      );

      const sectionSummary = document.getElementById(
        `${divId}-summary-${encodeFieldKey(segmentId)}-${encodeFieldKey(sectionId)}`,
      );
      if (sectionSummary) {
        //window.debug=true;
        const unansweredCount =
          sectionTotals.requiredFields - sectionTotals.answeredRequiredFields;
a=window.debug ?`[field total: ${sectionTotals.allFields}, score: ${sectionTotals.totalScore}, achieved score: ${sectionTotals.totalScoreWeighted}, maxscore: ${sectionTotals.maxPossibleScoreWeighted}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]`:`[field total: ${sectionTotals.allFields}, total required: ${sectionTotals.requiredFields}, answered required/total: ${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields}, unanswered required/total: ${unansweredCount}/${sectionTotals.requiredFields}, Files: ${sectionTotals.totalFiles}]`;
          sectionSummary.innerHTML = `
          <summary style="cursor: pointer;">UnAnswered/Total/Required (${unansweredCount}/${sectionTotals.allFields}/${sectionTotals.requiredFields})</summary>
          <div style="padding: 8px;">
            `+a+`${
              sectionTotals.unansweredFields.length > 0
                ? `
              <div style="margin-top: 10px; color: red;">
                <strong>Unanswered Required Fields:</strong>
                <ul style="margin: 5px 0; padding-left: 20px;">
                  ${sectionTotals.unansweredFields
                    .map(
                      (field) => `
                    <li style="max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                      <a href="javascript:scrollToField('${encodeFieldKey(
                        field.key,
                      )}', '${divId}')"
                         style="text-decoration: none; color: red;"
                         title="${field.label}">
                        ${field.label}
                      </a>
                    </li>
                  `,
                    )
                    .join("")}
                </ul>
              </div>
            `
                : ""
            }
          </div>
        `;
          if (unansweredCount > 0) {
    sectionSummary.classList.add("unanswered");
  } else {
    sectionSummary.classList.remove("unanswered");
  }
      }

      const sectionHeader = document.getElementById(
        `${divId}-section-header-${encodeFieldKey(sectionId)}`,
      );
      if (sectionHeader) {
        const progressPercent =
          sectionTotals.requiredFields > 0
            ? Math.round(
                (sectionTotals.answeredRequiredFields /
                  sectionTotals.requiredFields) *
                  100,
              )
            : 100;
        globalAnsweredFields += sectionTotals.answeredRequiredFields;
        globalAnsweredRequiredFields += sectionTotals.answeredRequiredFields;
        globalRequiredFields += sectionTotals.requiredFields;
        sectionHeader.innerHTML = `
          <div style="font-weight: bold;">${sectionLabel}</div>
          <div style="margin-top: 5px;">
            [Progress: ${progressPercent}% (${sectionTotals.answeredRequiredFields}/${sectionTotals.requiredFields})]
            <div style="width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;">
              <div style="width: ${progressPercent}%; background-color: #4caf50; height: 100%;"></div>
            </div>
          </div>
        `;
      }
    });
  });
  // Update your summary div by id 'summan'
  const summanDiv = document.getElementById("summan");
  if (summanDiv) {
    summanDiv.textContent = `Total fields: ${globalTotalFields} | ${globalRequiredFields} Answered required fields: ${globalAnsweredFields}`;
    renderSchemaStructureToSumman(schema, divId, lang);
  }
}


function updateJumperHelperWithUnanswered(summary, data, targetDivId = "jumperhelper") {
  const lang = window.lang || "en";
  if (!summary || !summary.segments || !summary.totals) {
    console.error("Invalid summary object:", summary);
    return "";
  }
    const panelStack = window.panelStack || [];
      const formId = panelStack.length > 0 ? panelStack[panelStack.length - 1] : "gform";


      
     if (formId === "gen")  window.activeFormId ="gform";
        else if (formId === "profile")  window.activeFormId ="pro";

  // Remove the generateScrollFunction and use scrollToField consistently
  const generateScrollLink = (key) => {
    return `javascript:scrollToField('${key}', '${activeFormId}')`;
  };
  let html = "<div style='font-family: Arial, sans-serif;'>";

  // Grand Summary with unanswered fields
  const globalTotals = summary.totals;
  const fieldsPercent = globalTotals.allFields > 0 
    ? Math.round((globalTotals.answeredFields / globalTotals.allFields) * 100) 
    : 100;
  const requiredPercent = globalTotals.requiredFields > 0 
    ? Math.round((globalTotals.answeredRequiredFields / globalTotals.requiredFields) * 100) 
    : 100;

  html += `<div style='border: 2px solid #4a90e2; padding: 10px; margin-bottom: 20px; border-radius: 5px;'>
            <h4 style='margin: 0 0 5px 0;'>Grand Summary</h4>
            <div style='margin-bottom: 5px;'>
              Total Fields: ${globalTotals.answeredFields}/${globalTotals.allFields}
              <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
              </div>
            </div>
            <div style='margin-bottom: 5px;'>
              Required: ${globalTotals.answeredRequiredFields}/${globalTotals.requiredFields}
              <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
              </div>
            </div>`;

  // Show unanswered required fields at global level
  if (globalTotals.unansweredRequiredFields?.length > 0) {
    html += `<div style='margin-top: 10px; color: orange;'>
              Unanswered Required Fields:
              <ul style='margin: 5px 0; padding-left: 20px;'>`;
    
    globalTotals.unansweredRequiredFields.forEach(field => {
      html += `<li style='max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
                <a href="${generateScrollLink(field.key)}"
                   style='text-decoration: none; color: red;'
                   title="${field.label}">
                  ${field.label}
                </a>
              </li>`;
    });
    
    html += `</ul></div>`;
  }

  html += `</div>`;

  // Segments with their unanswered fields
  html += "<div style='display: flex; flex-wrap: wrap; gap: 10px;'>";
  Object.entries(summary.segments).forEach(([segmentKey, segment]) => {
    const segmentTotals = segment.totals || {};
    const segmentId = `gform-segment-${encodeFieldKey(segmentKey)}`;
    const fieldsPercent = segmentTotals.allFields > 0 
      ? Math.round((segmentTotals.answeredFields / segmentTotals.allFields) * 100) 
      : 100;
    const requiredPercent = segmentTotals.requiredFields > 0 
      ? Math.round((segmentTotals.answeredRequiredFields / segmentTotals.requiredFields) * 100) 
      : 100;

    html += `<div style='border: 1px solid #ccc; padding: 10px; border-radius: 5px; min-width: 200px;'>
              <h4 style='margin: 0 0 5px 0;'>
                  <a href="${generateScrollLink(segmentKey)}" 
                     style='text-decoration: none; color: #333;'
                     title="Jump to ${segment.label || segmentKey}">
                      ${segment.label || segmentKey}
                  </a>
              </h4>
              <div style='margin-bottom: 5px;'>
                Fields: ${segmentTotals.answeredFields}/${segmentTotals.allFields}
                <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                  <div style='width: ${fieldsPercent}%; background-color: #4caf50; height: 100%;'></div>
                </div>
              </div>
              <div style='margin-bottom: 5px;'>
                Required: ${segmentTotals.answeredRequiredFields}/${segmentTotals.requiredFields}
                <div style='width: 100%; background-color: #e0e0e0; height: 10px; border-radius: 5px; overflow: hidden;'>
                  <div style='width: ${requiredPercent}%; background-color: #4caf50; height: 100%;'></div>
                </div>
              </div>`;

    // Show segment's unanswered required fields
    if (segmentTotals.unansweredRequiredFields?.length > 0) {
      html += `<div style='margin-top: 5px; color: orange; font-size: 0.9em;'>
                Unanswered:
                <ul style='margin: 3px 0; padding-left: 15px;'>`;
      
      segmentTotals.unansweredRequiredFields.forEach(field => {
        const fieldId = `gform-field-${encodeFieldKey(field.key)}`;
        html += `<li style='max-width: 180px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;'>
                  <a href="${generateScrollLink(field.key)}"
                     style='text-decoration: none; color: red;'
                     title="${field.label}">
                    ${field.label}
                  </a>
                </li>`;
      });
      
      html += `</ul></div>`;
    }

    // Sections within this segment
    html += `<div style='margin-top: 10px;'>`;
    Object.entries(segment.sections || {}).forEach(([sectionKey, section]) => {
      const sectionId = `gform-section-${encodeFieldKey(sectionKey)}`;
      const sectionTotals = section.totals || {};
      
      html += `<div style='margin-bottom: 5px;'>
                <a href="${generateScrollLink(sectionKey)}"
                   style='text-decoration: none; color: #666;'
                   title="Jump to ${section.label || sectionKey}">
                  ${section.label || sectionKey} (${sectionTotals.answeredFields}/${sectionTotals.allFields})
                </a>
              </div>`;
    });
    html += `</div>`;

    html += `</div>`;
  });
  html += `</div></div>`;

  const jumperDiv = document.getElementById(targetDivId);
  if (jumperDiv) {
    jumperDiv.innerHTML = html;
  } else {
    console.error(`Target div ${targetDivId} not found`);
  }

  return html;
}

// Summary calculation function
function QSummary(data) {
  if (!data || !data.grouping || !data.grouping.name) {
    console.warn("Invalid data format: grouping is missing");
    return { segments: [], summary: { segments: {}, totals: {} } };
  }

  const result = { segments: [] };
  const summary = {
    segments: {},
    totals: {
      allFields: 0,
      requiredFields: 0,
      answeredFields: 0,
      answeredRequiredFields: 0,
      totalScore: 0,
      totalScoreWeighted: 0,
      maxPossibleScore: 0,
      maxPossibleScoreWeighted: 0,
      totalFiles: 0,
      unansweredRequiredFields: [] // Added this
    },
  };

  const segmentMap = {};

  Object.entries(data).forEach(([key, item]) => {
    if (key === "grouping" || item.type === "section") return;

    const segment = item.segment?.[0] || data.grouping?.name[0] || "default";
    const section = item.section;
    if (!segment || !section || !data[section]) {
      console.warn(
        `Invalid field ${key}: missing segment or section, or section not found in schema`
      );
      return;
    }

    // Initialize segment/section if not exists
    if (!summary.segments[segment]) {
      const segmentIndex = data.grouping.name.indexOf(segment);
      summary.segments[segment] = {
        label: data.grouping.label?.en[segmentIndex] || segment,
        sections: {},
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [] // Added this
        },
      };
    }
    if (!summary.segments[segment].sections[section]) {
      summary.segments[segment].sections[section] = {
        label: data[section]?.label?.en || section,
        totals: {
          allFields: 0,
          requiredFields: 0,
          answeredFields: 0,
          answeredRequiredFields: 0,
          totalScore: 0,
          totalScoreWeighted: 0,
          maxPossibleScore: 0,
          maxPossibleScoreWeighted: 0,
          totalFiles: 0,
          unansweredRequiredFields: [] // Added this
        },
      };
    }

    const currentScore = calculateCurrentScore(item);
    const maxScore = calculateMaxScore(item);
    const isAnswered = isFieldAnswered(item);
    const isFile = item.data_type === "attachment" || item.type === "attachment";
    const fileCount = isFile ? countFiles(item) : 0;

    // Update totals
    const sectionTotals = summary.segments[segment].sections[section].totals;
    const segmentTotals = summary.segments[segment].totals;
    const globalTotals = summary.totals;

    sectionTotals.allFields++;
    if (item.required) sectionTotals.requiredFields++;
    if (isAnswered) sectionTotals.answeredFields++;
    if (item.required && isAnswered) sectionTotals.answeredRequiredFields++;
    sectionTotals.totalScore += isAnswered ? item.score || 1 : 0;
    sectionTotals.totalScoreWeighted += currentScore;
    sectionTotals.maxPossibleScore += item.score || 1;
    sectionTotals.maxPossibleScoreWeighted += maxScore;
    sectionTotals.totalFiles += fileCount;

    // Track unanswered required fields
    if (item.required && !isAnswered) {
      sectionTotals.unansweredRequiredFields.push({
        key,
        label: getLabel(item, window.lang || "en") || key,
        segment,
        section
      });
    }

    // Update segment totals (same logic as section)
    segmentTotals.allFields++;
    if (item.required) segmentTotals.requiredFields++;
    if (isAnswered) segmentTotals.answeredFields++;
    if (item.required && isAnswered) segmentTotals.answeredRequiredFields++;
    segmentTotals.totalScore += isAnswered ? item.score || 1 : 0;
    segmentTotals.totalScoreWeighted += currentScore;
    segmentTotals.maxPossibleScore += item.score || 1;
    segmentTotals.maxPossibleScoreWeighted += maxScore;
    segmentTotals.totalFiles += fileCount;

    // Track unanswered required fields at segment level
    if (item.required && !isAnswered) {
      segmentTotals.unansweredRequiredFields.push({
        key,
        label: getLabel(item, window.lang || "en") || key,
        segment,
        section
      });
    }

    // Update global totals (same logic)
    globalTotals.allFields++;
    if (item.required) globalTotals.requiredFields++;
    if (isAnswered) globalTotals.answeredFields++;
    if (item.required && isAnswered) globalTotals.answeredRequiredFields++;
    globalTotals.totalScore += isAnswered ? item.score || 1 : 0;
    globalTotals.totalScoreWeighted += currentScore;
    globalTotals.maxPossibleScore += item.score || 1;
    globalTotals.maxPossibleScoreWeighted += maxScore;
    globalTotals.totalFiles += fileCount;

    // Track unanswered required fields at global level
    if (item.required && !isAnswered) {
      globalTotals.unansweredRequiredFields.push({
        key,
        label: getLabel(item, window.lang || "en") || key,
        segment,
        section
      });
    }

    // Rest of the function remains the same...
    // [Previous code for building result.segments]
  });

  return { result, summary };
}
