function addStyle() {
  if (document.getElementById('smartbar-styles')) return;
  const style = document.createElement('style');
  style.id = 'smartbar-styles';
  style.textContent = `
    html, body {
      margin: 0;
      padding: 0;
      height: 100%;
      animation: myanimation 30s infinite;
      transition: all 1.5s ease;
    }

    @keyframes myanimation {
      0% { background: linear-gradient(90deg, rgba(140, 224, 255, .2) 0%, rgba(108, 230, 157, .5) 50%, rgba(255, 255, 255, 1) 100%); }
      25% { background: linear-gradient(90deg, rgba(155, 151, 42, .5) 0%, rgba(188, 199, 87, .5) 50%, rgba(255, 255, 255, 1) 100%); }
      50% { background: linear-gradient(90deg, rgba(42, 123, 155, .2) 0%, rgba(87, 199, 133, .5) 50%, rgba(252, 249, 220, 1) 100%); }
      75% { background: linear-gradient(90deg, rgba(156, 165, 173, .2) 0%, rgba(87, 199, 133, .5) 50%, rgba(230, 230, 230, 1) 100%); }
      100% { background: linear-gradient(90deg, rgba(255, 107, 107, .2) 0%, rgba(247, 150, 64, .2) 50%, rgba(255, 254, 252, 1) 100%); }
    }

    #triggerIcon {
      position: absolute;
      width: 60px;
      height: 60px;
      border-radius: 50%;
      background-color: #444;
      background-image: url('https://i.pravatar.cc/60');
      background-size: cover;
      background-position: center;
      z-index: 10;
      cursor: grab;
      user-select: none;
      -webkit-user-select: none;
      -moz-user-select: none;
      -ms-user-select: none;
    }

    #triggerIcon.dragging {
      border: 1px solid rgba(190, 124, 28, 0.5);
      outline: 5px solid rgba(14, 181, 227, 0.2);
      box-sizing: border-box;
      cursor: grabbing;
    }

    #smartBar {
      position: absolute;
      width: 300px;
      height: 40px;
      border: 1px solid rgba(200, 200, 200, 0.5);
      border-radius: 20px;
      box-shadow: 0 4px 12px rgba(97, 97, 97, 0.55);
      z-index: 9;
      padding: 8px;
      display: flex;
      align-items: center;
      justify-content: space-between;
      overflow: hidden;
      box-sizing: border-box;
      background: rgba(0, 0, 0, 0.1);
      backdrop-filter: blur(20px);
    }

    #pillBar {
      position: relative;
      display: block;
      z-index: 8;
      background-color: transparent;
      box-shadow: none;
      padding: 0;
      margin: 0;
      width: 100%;
      height: 100%;
    }

    .other-group {
      display: flex;
      align-items: center;
      gap: 8px;
      overflow: hidden;
    }

    .bar-left {
      display: flex;
      align-items: center;
      gap: 0px;
      overflow: hidden;
    }

    .label {
      font-weight: bold;
      font-size: 14px;
      white-space: nowrap;
    }

    .username {
      font-weight: bold;
      font-size: 14px;
      white-space: nowrap;
    }

    .btn {
      font-size: 12px;
      padding: 4px 8px;
      border: none;
      background: #eee;
      cursor: pointer;
      border-radius: 4px;
    }

    .toggleBtn {
      border: solid 1px rgba(200, 200, 200, 0.5);
      width: 14px;
      height: 14px;
      min-width: 14px;
      min-height: 14px;
      border-radius: 50%;
      background: #2196f3;
      padding: 0;
      display: flex;
      align-items: center;
      justify-content: center;
      cursor: pointer;
      margin: 0;
      flex-shrink: 0;
    }

    .circle-btn {
      border: solid 1px rgba(200, 200, 200, 0.5);
      width: 14px;
      height: 14px;
      min-width: 14px;
      min-height: 14px;
      border-radius: 50%;
      padding: 0;
      display: flex;
      align-items: center;
      justify-content: center;
      cursor: pointer;
      margin: 0;
      flex-shrink: 0;
      font-size: 8px;
      color: white;
    }

    #langBtn {
      background: #4CAF50;
    }

    #exButton {
      background: #FFD700;
    }

    #logoutBtn {
      background: #f44336;
    }

    #clearBtn {
      background: #ff9800;
    }

    .bar-panel {
      position: fixed;
      background-color: rgba(155, 151, 152, .1);
      backdrop-filter: blur(10px);
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
      display: none;
      overflow: hidden;
      color: white;
      padding: 50px;
      box-sizing: border-box;
      height: 400px;
    }

    .menu-container {
      position: fixed;
      background-color: rgba(55, 51, 52, .1);
      backdrop-filter: blur(20px);
      border-radius: 8px;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.75);
      padding: 8px 0;
      z-index: 111;
      min-width: 180px;
      display: none;
    }

    .menu-group {
      margin: 8px 0;
    }

    .menu-group:not(:last-child) {
      border-bottom: 1px solid #eee;
    }

    .menu-item {
      padding: 10px 16px;
      cursor: pointer;
      color: #333;
      display: flex;
      align-items: center;
      gap: 8px;
      transition: background-color 0.3s ease, color 0.3s ease;
    }

    .menu-item:hover {
      background-color: #252553;
      color: #ffffff;
    }

    .menu-icon {
      width: 20px;
      text-align: center;
    }

    .menu-text {
      flex: 1;
    }

    .menu-overlay {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      z-index: 100;
      display: none;
    }

    .badge {
      background-color: #e31515;
      border-radius: 10px;
      padding: 2px 6px;
      font-size: 12px;
      color: #eedfdf;
    }

    .menuButton {
      font-size: 12px;
      padding: 4px 8px;
      border: none;
      cursor: pointer;
      border-radius: 4px;
    }
  `;
  document.head.appendChild(style);
}

function updateBadge(elementId, count) {
  const element = document.getElementById(elementId);
  if (!element) return;
  let badge = element.querySelector('.badge');
  if (!badge) {
    badge = document.createElement('span');
    badge.className = 'badge';
    element.appendChild(badge);
  }
  badge.textContent = count;
}

function updateTooltip(elementId, text, optionsOrPosition, color) {
  const element = document.getElementById(elementId);
  if (!element) return;
  element.title = text;
}

function createSmartBarElements() {
  const wrapper = document.createElement('div');
  wrapper.innerHTML = `
    <div id="pillBar">
      <div id="triggerIcon" title="Drag me!"></div>
      <div id="smartBar">        
        <div class="bar-left">
          <button class="toggleBtn" title="Expand/Collapse" id="toggleBarBtn"></button>    
          <div id="otherGroup" class="other-group">
            <div class="menuButton" id="menuButton">Menu</div>
            <button class="btn circle-btn" id="exButton"></button>
            <span class="label">[info]</span>
            <span class="username" id="username">User</span>
            <button class="btn circle-btn" id="langBtn"></button>
            <button class="btn circle-btn" id="logoutBtn"></button>
            <button class="btn circle-btn" id="clearBtn"></button>
          </div>
        </div>
      </div>
    </div>
    <div id="barPanel" class="bar-panel"></div>
    <div id="menuContainer" class="menu-container"></div>
    <div id="menuOverlay" class="menu-overlay"></div>
  `;
  document.body.appendChild(wrapper);
}


  addStyle();
  createSmartBarElements();

  const trigger = document.getElementById("triggerIcon");
  const smartBar = document.getElementById("smartBar");
  const barPanel = document.getElementById("barPanel");
  const toggleBarBtn = document.getElementById("toggleBarBtn");
  const otherGroup = document.getElementById("otherGroup");
  const menuContainer = document.getElementById("menuContainer");
  const menuOverlay = document.getElementById("menuOverlay");
  let barPanelVisible = false;
  let justDragged = false;
  let smartBarMode = true;

  const menuData = [
    {
      group: "View Options",
      items: [
        {
          item: "togglebar",
          icon: "🟦",
          text: "Assessment",
          onclick: () => {
            smartBarMode = true;
            menuContainer.style.display = "none";
            menuOverlay.style.display = "none";
            saveSmartBarState();
          },
        },
        {
          item: "togglelist",
          icon: "📋",
          text: "Profile",
          onclick: () => alert("Switched to list (simulated)"),
        },
      ],
    },
    {
      group: "User Actions",
      items: [
        {
          item: "profile",
          icon: "👤",
          text: "User",
          onclick: () => alert("Profile clicked"),
        },
        {
          item: "settings",
          icon: "⚙️",
          text: "Settings",
          onclick: () => alert("Settings clicked"),
        },
      ],
    },
    {
      group: "Support",
      items: [
        {
          item: "help",
          icon: "❓",
          text: "Help",
          onclick: () => alert("Help clicked"),
        },
        {
          item: "logout",
          icon: "🚪",
          text: "Logout",
          onclick: () => alert("Logged out"),
        },
      ],
    },
  ];

  function savePosition() {
    localStorage.setItem("triggerX", trigger.offsetLeft);
    localStorage.setItem("triggerY", trigger.offsetTop);
  }

  function loadPosition() {
    const x = parseInt(localStorage.getItem("triggerX"));
    const y = parseInt(localStorage.getItem("triggerY"));
    if (!isNaN(x) && !isNaN(y)) {
      const clamped = clampToViewport(x, y, trigger);
      trigger.style.left = clamped.x + "px";
      trigger.style.top = clamped.y + "px";
    } else {
      trigger.style.left = "20px";
      trigger.style.top = (window.innerHeight - trigger.offsetHeight - 20) + "px";
    }
  }

  function saveSmartBarState() {
    localStorage.setItem("smartBarWidth", parseInt(smartBar.style.width));
    localStorage.setItem("smartBarTop", smartBar.style.top);
    localStorage.setItem("smartBarLeft", smartBar.style.left);
    localStorage.setItem("smartBarMode", smartBarMode ? "1" : "0");
    localStorage.setItem("toggleBtnClass", toggleBarBtn.className);
    localStorage.setItem("toggleBtnLeft", toggleBarBtn.style.left || "");
    localStorage.setItem("toggleBtnRight", toggleBarBtn.style.right || "");
    localStorage.setItem("otherGroupMarginLeft", otherGroup.style.marginLeft || "");
    localStorage.setItem("otherGroupMarginRight", otherGroup.style.marginRight || "");
    localStorage.setItem("otherGroupPosition", otherGroup.style.position || "");
  }

  function loadSmartBarState() {
    const width = localStorage.getItem("smartBarWidth") || "300";
    const top = localStorage.getItem("smartBarTop");
    const left = localStorage.getItem("smartBarLeft");
    const otherMarginLeft = localStorage.getItem("otherGroupMarginLeft") || "";
    const otherMarginRight = localStorage.getItem("otherGroupMarginRight") || "";
    const otherPosition = localStorage.getItem("otherGroupPosition") || "";
    
    smartBar.style.width = width + "px";
    if (top && left) {
      smartBar.style.top = top;
      smartBar.style.left = left;
    } else {
      positionSmartBar();
    }
    const mode = localStorage.getItem("smartBarMode");
    smartBarMode = mode === "1";
    const toggleClass = localStorage.getItem("toggleBtnClass") || "toggleBtn";
    toggleBarBtn.className = toggleClass;
    toggleBarBtn.style.left = localStorage.getItem("toggleBtnLeft") || "";
    toggleBarBtn.style.right = localStorage.getItem("toggleBtnRight") || "";
    otherGroup.style.marginLeft = otherMarginLeft;
    otherGroup.style.marginRight = otherMarginRight;
    otherGroup.style.position = otherPosition;
    if (smartBarMode) {
      smartBar.style.display = "flex";
    } else {
      smartBar.style.display = "none";
    }
  }

  loadSmartBarState();
  loadPosition();
  smartBar.style.display = "flex";

  function getZone(rect) {
    const centerX = rect.left + rect.width / 2;
    const centerY = rect.top + rect.height / 2;
    const thirdW = window.innerWidth / 3;
    const thirdH = window.innerHeight / 3;
    const horiz = centerX < thirdW ? "L" : centerX < thirdW * 2 ? "C" : "R";
    const vert = centerY < thirdH ? "T" : centerY < thirdH * 2 ? "C" : "B";
    return vert + horiz;
  }

  function positionSmartBar() {
    const iconRect = trigger.getBoundingClientRect();
    const barWidth = smartBar.offsetWidth || 300;
    const barHeight = smartBar.offsetHeight || 40;
    const zone = getZone(iconRect);
    let top = iconRect.top + (iconRect.height - barHeight) / 2;
    let left;
    const SPACING = 15;

    if (["TL", "CL", "BL"].includes(zone)) {
      left = iconRect.right + SPACING;
      toggleBarBtn.style.left = "0px";
      toggleBarBtn.style.right = "";
      otherGroup.style.marginLeft = "";
      otherGroup.style.marginRight = "";
      otherGroup.style.position = "";
    } else if (["TR", "CR", "BR"].includes(zone)) {
      left = iconRect.left - barWidth - SPACING;
      toggleBarBtn.style.right = "0px";
      toggleBarBtn.style.left = "";
    } else {
      if (iconRect.right + barWidth + SPACING < window.innerWidth) {
        left = iconRect.right + SPACING;
        toggleBarBtn.style.left = "0px";
        toggleBarBtn.style.right = "";
        otherGroup.style.position = "";
        otherGroup.style.marginLeft = "";
      } else {
        left = iconRect.left - barWidth - SPACING;
        toggleBarBtn.style.right = "0px";
        toggleBarBtn.style.left = "";
      }
    }

    top = Math.max(10, Math.min(top, window.innerHeight - barHeight - 10));
    left = Math.max(10, Math.min(left, window.innerWidth - barWidth - 10));

    smartBar.style.top = `${top}px`;
    smartBar.style.left = `${left}px`;

    if (barPanelVisible) {
      positionBarPanel();
    }
  }

  function positionBarPanel() {
    const pillRect = smartBar.getBoundingClientRect();
    const pillBartop = pillRect.top;
    const panelHeight = barPanel.offsetHeight || 400;
    const spaceBelow = window.innerHeight - pillRect.bottom;
    const spaceRight = window.innerWidth - trigger.offsetLeft + trigger.offsetWidth;

    barPanel.style.width = (smartBar.offsetWidth + trigger.offsetWidth + 20) + 'px';

    if (spaceBelow >= panelHeight || spaceBelow >= pillRect.top) {
      barPanel.style.top = (pillBartop - 10) + 'px';
      barPanel.style.bottom = 'auto';
      barPanel.style.border = 'solid 10px rgba(200, 200, 200, 0.5)';
      barPanel.style.borderRadius = '30px 30px 10px 10px';
    } else {
      barPanel.style.top = 'auto';
      barPanel.style.bottom = (window.innerHeight - (pillRect.top + pillRect.height) - 10) + 'px';
      barPanel.style.border = 'solid 10px rgba(200, 200, 200, 0.5)';
      barPanel.style.borderRadius = '30px 30px 30px 30px';
    }
    if (spaceRight > trigger.offsetWidth) {
      barPanel.style.left = trigger.offsetLeft + 'px';
      barPanel.style.right = '10px';
    } else {
      barPanel.style.left = pillRect.left + 'px';
    }
    if (trigger.offsetLeft + trigger.offsetWidth + barPanel.offsetWidth > window.innerWidth) {
      barPanel.style.left = (trigger.offsetLeft + trigger.offsetWidth - barPanel.offsetWidth) + 'px';
      barPanel.style.borderRadius = '30px 30px 30px 30px';
    }
  }

  function positionMenuSmart(trigger) {
    const iconRect = trigger.getBoundingClientRect();
    const menu = menuContainer;
    const menuWidth = menu.offsetWidth || 180;
    let menuHeight = Array.from(menu.children).reduce((total, group) => {
      return total + group.offsetHeight +
             parseInt(getComputedStyle(group).marginTop) +
             parseInt(getComputedStyle(group).marginBottom);
    }, 0) || 200;

    const zone = getZone(iconRect);

    let top = 0, left = 0;
    switch (zone) {
      case "TL":
      case "TC":
        top = iconRect.bottom + 8;
        left = iconRect.left;
        break;
      case "TR":
        top = iconRect.bottom + 8;
        left = iconRect.right - menuWidth;
        break;
      case "CL":
        top = iconRect.bottom + 8;
        left = iconRect.left;
        if (iconRect.bottom + menuHeight + 10 > window.innerHeight)
          top = iconRect.top - menuHeight - 8;
        break;
      case "CR":
        top = iconRect.bottom + 8;
        left = iconRect.right - menuWidth;
        if (iconRect.bottom + menuHeight + 10 > window.innerHeight)
          top = iconRect.top - menuHeight - 8;
        break;
      case "CC":
        top = iconRect.bottom + 8;
        left = iconRect.left;
        break;
      case "BL":
      case "BC":
        top = iconRect.top - menuHeight - 8;
        left = iconRect.left;
        break;
      case "BR":
        top = iconRect.top - menuHeight - 8;
        left = iconRect.right - menuWidth;
        break;
    }

    top = Math.max(0, Math.min(top, window.innerHeight - menuHeight - 10));
    left = Math.max(0, Math.min(left, window.innerWidth - menuWidth - 10));

    menu.style.top = `${top}px`;
    menu.style.left = `${left}px`;
  }

  function renderMenu(data, container) {
    container.innerHTML = "";
    data.forEach((group, groupIndex) => {
      const groupDiv = document.createElement("div");
      groupDiv.className = "menu-group";
      group.items.forEach((entry, itemIndex) => {
        const item = document.createElement("div");
        item.className = "menu-item";
        item.id = `menuContainer-${entry.item}`;

        item.innerHTML = `
          <div class="menu-icon">${entry.icon}</div>
          <div class="menu-text">${entry.text}</div>
        `;

        item.onclick = () => {
          entry.onclick();
          if (container === menuContainer) {
            closeMenu();
          }
        };
        groupDiv.appendChild(item);
      });
      container.appendChild(groupDiv);
    });
  }

  function closeMenu() {
    menuContainer.style.display = 'none';
    menuOverlay.style.display = 'none';
  }

  function toggleBarPanel() {
    barPanelVisible = !barPanelVisible;
    if (barPanelVisible) {
      barPanel.style.display = 'block';
      positionBarPanel();
    } else {
      barPanel.style.display = 'none';
    }
  }

  let isDragging = false;
  let offsetX = 0, offsetY = 0;

  trigger.addEventListener("mousedown", (e) => {
    e.preventDefault();
    e.stopPropagation();
    const rect = trigger.getBoundingClientRect();
    offsetX = e.clientX - rect.left;
    offsetY = e.clientY - rect.top;
    isDragging = true;
    trigger.style.cursor = "grabbing";
    trigger.classList.add("dragging");
    justDragged = true;
  });

  document.addEventListener("mousemove", (e) => {
    if (!isDragging) return;
    e.preventDefault();
    const x = e.clientX - offsetX;
    const y = e.clientY - offsetY;
    const bounds = clampToViewport(x, y, trigger);
    trigger.style.left = bounds.x + "px";
    trigger.style.top = bounds.y + "px";
    positionSmartBar();
  });

  document.addEventListener("mouseup", (e) => {
    if (isDragging) {
      e.preventDefault();
      savePosition();
      positionSmartBar();
      saveSmartBarState();
      setTimeout(() => {
        justDragged = false;
      }, 100);
    }
    isDragging = false;
    trigger.style.cursor = "grab";
    trigger.classList.remove("dragging");
  });

  function clampToViewport(x, y, el) {
    const maxX = window.innerWidth - el.offsetWidth - 5;
    const maxY = window.innerHeight - el.offsetHeight - 5;
    return {
      x: Math.max(0, Math.min(x, maxX)),
      y: Math.max(0, Math.min(y, maxY)),
    };
  }

  const langBtn = document.getElementById("langBtn");
  const logoutBtn = document.getElementById("logoutBtn");
  const clearBtn = document.getElementById("clearBtn");
  const exButton = document.getElementById("exButton");
  const menuButton = document.getElementById("menuButton");

  toggleBarBtn.addEventListener("click", () => {
    const oldWidth = smartBar.offsetWidth || 300;
    const isExpanding = oldWidth <= 150;
    const newWidth = isExpanding ? 300 : 100;
    const iconRect = trigger.getBoundingClientRect();
    const zone = getZone(iconRect);
    const currentLeft = parseInt(smartBar.style.left || "0");

    if (["TR", "CR", "BR"].includes(zone)) {
      if (isExpanding) {
        smartBar.style.left = (currentLeft - (newWidth - oldWidth)) + "px";
        smartBar.style.width = newWidth + "px";
      } else {
        smartBar.style.width = newWidth + "px";
        smartBar.style.left = (currentLeft + (oldWidth - newWidth)) + "px";
      }
    } else {
      smartBar.style.width = newWidth + "px";
    }
    
    if (barPanelVisible) {
      positionBarPanel();
    }
    saveSmartBarState();
  });

  langBtn.addEventListener("click", () => {
    const current = langBtn.textContent.trim().toLowerCase();
    langBtn.textContent = current === "e" ? "t" : "e";
  });

  logoutBtn.addEventListener("click", () => {
    alert("Logged out");
  });

  clearBtn.addEventListener("click", () => {
    localStorage.clear();
    trigger.style.left = "20px";
    trigger.style.top = (window.innerHeight - trigger.offsetHeight - 20) + "px";
    smartBar.style.width = "300px";
    barPanel.style.display = "none";
    barPanelVisible = false;
    menuContainer.style.display = "none";
    menuOverlay.style.display = "none";
    positionSmartBar();
    saveSmartBarState();
    savePosition();
    alert("LocalStorage cleared. Reset to default position.");
  });

  exButton.addEventListener("click", (e) => {
    if (isDragging || justDragged) {
      justDragged = false;
      return;
    }
    toggleBarPanel();
  });

  menuButton.addEventListener("click", (e) => {
    const show = menuContainer.style.display !== 'block';
    menuContainer.style.display = show ? 'block' : 'none';
    menuOverlay.style.display = show ? 'block' : 'none';
    if (show) {
      renderMenu(menuData, menuContainer);
      updateBadge("menuContainer-profile", 2);
      updateBadge("menuContainer-settings", 1);
      updateTooltip("menuContainer-settings", "Tooltip on top", {
        position: 'bottom',
        color: 'red',
      });
      positionMenuSmart(e.target);
    }
  });

  menuOverlay.addEventListener('click', closeMenu);

  updateBadge("menuContainer-profile", 2);
  updateBadge("triggerIcon", 21);
  updateTooltip("triggerIcon", "Tooltip on top", "right", "purple");
  updateTooltip("logoutBtn", "Change language", {
    position: 'top',
    color: 'green',
  });
  updateTooltip("clearBtn", "Change language", {
    position: 'top',
    color: 'red',
  });
  updateTooltip("langBtn", "Change language", {
    position: 'bottom',
    color: 'red',
  });
  updateTooltip("username", "Drag me to reposition", {
    position: 'top',
    color: 'blue',
  });

  window.addEventListener('resize', () => {
    positionSmartBar();
    if (barPanelVisible) positionBarPanel();
    if (menuContainer.style.display === 'block') positionMenuSmart(menuButton);
  });
