async function itemsListPanel(url = 'https://i.kunok.com/api62.php?action=list_all&fields=id,name,flag,timestamp,board') {
    let items = [];

    function renderItems() {
        if (!items.length) return `<div class="error">No items found</div>`;

        return `
            <style>
                .item-entry {
                    background: #fff;
                    padding: 12px;
                    border-radius: 6px;
                    margin-bottom: 10px;
                    border: 1px solid #ccc;
                    font-size: 14px;
                    cursor: pointer;
                    transition: background 0.2s;
                }
                .item-entry:hover {
                    background: #f8f8f8;
                }
                .item-entry pre {
                    background: #f5f5f5;
                    padding: 6px;
                    border-radius: 4px;
                    font-size: 12px;
                    overflow-x: auto;
                    margin-top: 4px;
                }
                .item-meta {
                    font-size: 12px;
                    color: #666;
                    margin-bottom: 4px;
                }
                .error {
                    color: red;
                    text-align: center;
                    margin: 20px;
                }
                .progress-bar {
                    background: #f0f0f0;
                    height: 8px;
                    border-radius: 4px;
                    overflow: hidden;
                    margin-top: 4px;
                }
                .progress-fill {
                    height: 100%;
                    background: #4caf50;
                    transition: width 0.3s;
                }
            </style>
            <div id="itemsList">
                ${items.map(item => {
                    const board = item.boardObj || {};
                    const totalPercent = board.totalRequired && board.answeredRequired ? 
                        (board.answeredRequired / board.totalRequired * 100).toFixed(1) : '0';
                    const bucketPercent = board.bucket_totalRequired && board.bucket_answeredRequired ? 
                        (board.bucket_answeredRequired / board.bucket_totalRequired * 100).toFixed(1) : '0';
                    const cupPercent = board.cup_totalRequired && board.cup_answeredRequired ? 
                        (board.cup_answeredRequired / board.cup_totalRequired * 100).toFixed(1) : '0';
                    const flagFromBoard = board.userid ? 
                        (board.userid.startsWith('2025') ? 'new' : 'existing') : item.flag;

                    return `
                        <div class="item-entry" data-item-id="${item.id}">
                            <div><strong>ID:</strong> ${item.id}</div>
                            <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                            <div class="item-meta">Flag: ${flagFromBoard} | Timestamp: ${item.timestamp}</div>
                            <div><strong>Board:</strong></div>
                            <pre>${item.boardRaw}</pre>
                            <div><strong>Board User:</strong> ${item.boardUser || 'N/A'}</div>
                            <div><strong>User ID:</strong> ${board.userid || 'N/A'}</div>
                            <div><strong>Username:</strong> ${board.user_name || 'N/A'}</div>
                            <div><strong>Progress:</strong> ${totalPercent}%</div>
                            <div class="progress-bar"><div class="progress-fill" style="width: ${totalPercent}%"></div></div>
                            <div><strong>Bucket Progress:</strong> ${bucketPercent}%</div>
                            <div class="progress-bar"><div class="progress-fill" style="width: ${bucketPercent}%"></div></div>
                            <div><strong>Cup Progress:</strong> ${cupPercent}%</div>
                            <div class="progress-bar"><div class="progress-fill" style="width: ${cupPercent}%"></div></div>
                            <div><strong>Save Status:</strong> ${board.save || 'N/A'}</div>
                            <div><strong>Time:</strong> ${board.time || 'N/A'}</div>
                            <div><strong>First Time:</strong> ${board.firstTime || 'N/A'}</div>
                            <div><strong>Keeper:</strong> ${item.keeper || 'N/A'}</div>
                        </div>
                    `;
                }).join("")}
            </div>
        `;
    }

    async function refreshSingleItem(itemId) {
        try {
            const res = await fetch(`https://i.kunok.com/api62.php?id=${encodeURIComponent(itemId)}`, { mode: 'cors' });
            const json = await res.json();
            if (!json.data) return;

            const item = json.data;
            let boardObj = {};
            let boardUser = "N/A";
            let boardRaw = "N/A";

            if (item.board && typeof item.board === "string") {
                try {
                    boardObj = JSON.parse(item.board);
                    boardUser = boardObj.user_name || boardObj.user || "N/A";
                    boardRaw = JSON.stringify(boardObj, null, 2);
                } catch {
                    boardRaw = "Invalid JSON";
                }
            }

            const totalPercent = boardObj.totalRequired && boardObj.answeredRequired ? 
                (boardObj.answeredRequired / boardObj.totalRequired * 100).toFixed(1) : '0';
            const bucketPercent = boardObj.bucket_totalRequired && boardObj.bucket_answeredRequired ? 
                (boardObj.bucket_answeredRequired / boardObj.bucket_totalRequired * 100).toFixed(1) : '0';
            const cupPercent = boardObj.cup_totalRequired && boardObj.cup_answeredRequired ? 
                (boardObj.cup_answeredRequired / boardObj.cup_totalRequired * 100).toFixed(1) : '0';
            const flagFromBoard = boardObj.userid ? 
                (boardObj.userid.startsWith('2025') ? 'new' : 'existing') : item.flag;

            const box = document.querySelector(`.item-entry[data-item-id="${itemId}"]`);
            if (box) {
                box.innerHTML = `
                    <div><strong>ID:</strong> ${item.id}</div>
                    <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                    <div class="item-meta">Flag: ${flagFromBoard} | Timestamp: ${item.timestamp}</div>
                    <div><strong>Board:</strong></div>
                    <pre>${boardRaw}</pre>
                    <div><strong>Board User:</strong> ${boardUser || 'N/A'}</div>
                    <div><strong>User ID:</strong> ${boardObj.userid || 'N/A'}</div>
                    <div><strong>Username:</strong> ${boardObj.user_name || 'N/A'}</div>
                    <div><strong>Progress:</strong> ${totalPercent}%</div>
                    <div class="progress-bar"><div class="progress-fill" style="width: ${totalPercent}%"></div></div>
                    <div><strong>Bucket Progress:</strong> ${bucketPercent}%</div>
                    <div class="progress-bar"><div class="progress-fill" style="width: ${bucketPercent}%"></div></div>
                    <div><strong>Cup Progress:</strong> ${cupPercent}%</div>
                    <div class="progress-bar"><div class="progress-fill" style="width: ${cupPercent}%"></div></div>
                    <div><strong>Save Status:</strong> ${boardObj.save || 'N/A'}</div>
                    <div><strong>Time:</strong> ${boardObj.time || 'N/A'}</div>
                    <div><strong>First Time:</strong> ${boardObj.firstTime || 'N/A'}</div>
                    <div><strong>Keeper:</strong> ${item.keeper || 'N/A'}</div>
                `;

                box.style.transition = 'background-color 0.6s ease';
                box.style.backgroundColor = '#ffffcc';
                setTimeout(() => {
                    box.style.backgroundColor = '#ffffff';
                }, 600);
            }
        } catch (err) {
            console.error("Error refreshing single item:", err);
        }
    }

    function attachClickEvents() {
        const entries = document.querySelectorAll(".item-entry");
        entries.forEach(entry => {
            entry.addEventListener("click", () => {
                const itemId = entry.getAttribute("data-item-id");
                if (itemId && typeof updateItemPanelEX === "function") {
                    updateItemPanelEX(itemId, () => refreshSingleItem(itemId));
                } else {
                    alert("updateItemPanelEX(id) not available.");
                }
            });
        });
    }

    try {
        const response = await fetch(url, { mode: 'cors' });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const result = await response.json();

        if (!result.data) throw new Error("No data found");

        items = result.data.map(item => {
            let boardObj = {};
            let boardUser = "N/A";
            let boardRaw = "N/A";

            if (item.board && typeof item.board === 'string') {
                try {
                    boardObj = JSON.parse(item.board);
                    boardUser = boardObj.user_name || boardObj.user || "N/A";
                    boardRaw = JSON.stringify(boardObj, null, 2);
                } catch (e) {
                    console.error("Error parsing board JSON for item:", item, e);
                    boardRaw = "Invalid JSON";
                }
            }

            return {
                ...item,
                boardObj,
                boardUser,
                boardRaw
            };
        });

        const html = renderItems();

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        createPanel("itemsListPanel", "CC", "ctc", 600, 600, {
            bodyContent: html,
            noFooter: true,
            title: "Items List"
        });

        setTimeout(attachClickEvents, 100);
    } catch (err) {
        console.error("Error loading items:", err);
        createPanel("itemsListPanel", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Items List"
        });
    }
}
async function itemsListSort(url = 'https://i.kunok.com/api62.php?action=list_all&fields=id,name,flag,timestamp,board') {
    let items = [];
    let sortField = 'progress';
    let sortOrder = 'desc';

    function renderItems() {
        if (!items.length) return `<div class="error">No items found</div>`;

        // Sort items based on sortField and sortOrder
        const sortedItems = [...items].sort((a, b) => {
            const boardA = a.boardObj || {};
            const boardB = b.boardObj || {};
            let valueA, valueB;

            switch (sortField) {
                case 'id':
                    valueA = a.id || '';
                    valueB = b.id || '';
                    break;
                case 'name':
                    valueA = a.name || '';
                    valueB = b.name || '';
                    break;
                case 'boardUser':
                    valueA = a.boardUser || '';
                    valueB = b.boardUser || '';
                    break;
                case 'username':
                    valueA = boardA.user_name || '';
                    valueB = boardB.user_name || '';
                    break;
                case 'progress':
                    valueA = boardA.totalRequired && boardA.answeredRequired ? 
                        (boardA.answeredRequired / boardA.totalRequired * 100) : 0;
                    valueB = boardB.totalRequired && boardB.answeredRequired ? 
                        (boardB.answeredRequired / boardB.totalRequired * 100) : 0;
                    break;
            }

            if (sortField !== 'progress') {
                valueA = valueA.toString().toLowerCase();
                valueB = valueB.toString().toLowerCase();
                return sortOrder === 'asc' ? valueA.localeCompare(valueB) : valueB.localeCompare(valueA);
            } else {
                return sortOrder === 'asc' ? valueA - valueB : valueB - valueA;
            }
        });

        return `
            <style>
                .item-entry {
                    background: #fff;
                    padding: 12px;
                    border-radius: 6px;
                    margin-bottom: 10px;
                    border: 1px solid #ccc;
                    font-size: 14px;
                    cursor: pointer;
                    transition: background 0.2s;
                }
                .item-entry:hover {
                    background: #f8f8f8;
                }
                .error {
                    color: red;
                    text-align: center;
                    margin: 20px;
                }
                .progress-bar {
                    background: #f0f0f0;
                    height: 8px;
                    border-radius: 4px;
                    overflow: hidden;
                    margin-top: 4px;
                }
                .progress-fill {
                    height: 100%;
                    background: #4caf50;
                    transition: width 0.3s;
                }
                .sort-controls {
                    margin-bottom: 20px;
                    display: flex;
                    gap: 10px;
                    font-size: 14px;
                }
                .sort-controls select {
                    padding: 6px;
                    font-size: 14px;
                }
            </style>
            <div id="itemsList">
                <div class="sort-controls">
                    <label for="sortField">Sort by:</label>
                    <select id="sortField">
                        <option value="id" ${sortField === 'id' ? 'selected' : ''}>ID</option>
                        <option value="name" ${sortField === 'name' ? 'selected' : ''}>Name</option>
                        <option value="boardUser" ${sortField === 'boardUser' ? 'selected' : ''}>Board User</option>
                        <option value="username" ${sortField === 'username' ? 'selected' : ''}>Username</option>
                        <option value="progress" ${sortField === 'progress' ? 'selected' : ''}>Progress</option>
                    </select>
                    <label for="sortOrder">Order:</label>
                    <select id="sortOrder">
                        <option value="asc" ${sortOrder === 'asc' ? 'selected' : ''}>Ascending</option>
                        <option value="desc" ${sortOrder === 'desc' ? 'selected' : ''}>Descending</option>
                    </select>
                </div>
                ${sortedItems.map(item => {
                    const board = item.boardObj || {};
                    const totalPercent = board.totalRequired && board.answeredRequired ? 
                        (board.answeredRequired / board.totalRequired * 100).toFixed(1) : '0';

                    return `
                        <div class="item-entry" data-item-id="${item.id}">
                            <div><strong>ID:</strong> ${item.id}</div>
                            <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                            <div><strong>Board User:</strong> ${item.boardUser || 'N/A'}</div>
                            <div><strong>Username:</strong> ${board.user_name || 'N/A'}</div>
                            <div><strong>Progress:</strong> ${totalPercent}%</div>
                            <div class="progress-bar"><div class="progress-fill" style="width: ${totalPercent}%"></div></div>
                        </div>
                    `;
                }).join("")}
            </div>
        `;
    }

    async function refreshSingleItem(itemId) {
        try {
            const res = await fetch(`https://i.kunok.com/api62.php?id=${encodeURIComponent(itemId)}`, { mode: 'cors' });
            const json = await res.json();
            if (!json.data) return;

            const item = json.data;
            let boardObj = {};
            let boardUser = "N/A";

            if (item.board && typeof item.board === "string") {
                try {
                    boardObj = JSON.parse(item.board);
                    boardUser = boardObj.user_name || boardObj.user || "N/A";
                } catch {
                    boardUser = "Invalid JSON";
                }
            }

            const totalPercent = boardObj.totalRequired && boardObj.answeredRequired ? 
                (boardObj.answeredRequired / boardObj.totalRequired * 100).toFixed(1) : '0';

            const box = document.querySelector(`.item-entry[data-item-id="${itemId}"]`);
            if (box) {
                box.innerHTML = `
                    <div><strong>ID:</strong> ${item.id}</div>
                    <div><strong>Name:</strong> ${item.name || 'N/A'}</div>
                    <div><strong>Board User:</strong> ${boardUser || 'N/A'}</div>
                    <div><strong>Username:</strong> ${boardObj.user_name || 'N/A'}</div>
                    <div><strong>Progress:</strong> ${totalPercent}%</div>
                    <div class="progress-bar"><div class="progress-fill" style="width: ${totalPercent}%"></div></div>
                `;

                box.style.transition = 'background-color 0.6s ease';
                box.style.backgroundColor = '#ffffcc';
                setTimeout(() => {
                    box.style.backgroundColor = '#ffffff';
                }, 600);
            }
        } catch (err) {
            console.error("Error refreshing single item:", err);
        }
    }

    function attachClickEvents() {
        const entries = document.querySelectorAll(".item-entry");
        entries.forEach(entry => {
            entry.addEventListener("click", () => {
                const itemId = entry.getAttribute("data-item-id");
                if (itemId && typeof updateItemPanelEX === "function") {
                    updateItemPanelEX(itemId, () => refreshSingleItem(itemId));
                } else {
                    alert("updateItemPanelEX(id) not available.");
                }
            });
        });
    }

    function attachSortEvents() {
        const sortFieldSelect = document.getElementById("sortField");
        const sortOrderSelect = document.getElementById("sortOrder");
        if (sortFieldSelect && sortOrderSelect) {
            sortFieldSelect.addEventListener("change", () => {
                sortField = sortFieldSelect.value;
                const html = renderItems();
                const panelBody = document.querySelector("#itemsList");
                if (panelBody) {
                    panelBody.outerHTML = html;
                    attachClickEvents();
                    attachSortEvents();
                }
            });
            sortOrderSelect.addEventListener("change", () => {
                sortOrder = sortOrderSelect.value;
                const html = renderItems();
                const panelBody = document.querySelector("#itemsList");
                if (panelBody) {
                    panelBody.outerHTML = html;
                    attachClickEvents();
                    attachSortEvents();
                }
            });
        }
    }

    try {
        const response = await fetch(url, { mode: 'cors' });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const result = await response.json();

        if (!result.data) throw new Error("No data found");

        items = result.data.map(item => {
            let boardObj = {};
            let boardUser = "N/A";
            let boardRaw = "N/A";

            if (item.board && typeof item.board === 'string') {
                try {
                    boardObj = JSON.parse(item.board);
                    boardUser = boardObj.user_name || boardObj.user || "N/A";
                    boardRaw = JSON.stringify(boardObj, null, 2);
                } catch (e) {
                    console.error("Error parsing board JSON for item:", item, e);
                    boardRaw = "Invalid JSON";
                }
            }

            return {
                ...item,
                boardObj,
                boardUser,
                boardRaw
            };
        });

        const html = renderItems();

        if (typeof createPanel !== "function") {
            console.error("Missing createPanel() from micro2.js");
            return;
        }

        createPanel("itemsListSort", "CC", "ctc", 600, 600, {
            bodyContent: html,
            noFooter: true,
            title: "Items Sorted List"
        });

        setTimeout(() => {
            attachClickEvents();
            attachSortEvents();
        }, 100);
    } catch (err) {
        console.error("Error loading items:", err);
        createPanel("itemsListSort", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="error">${err.message}</div>`,
            noFooter: true,
            title: "Items Sorted List"
        });
    }
}

async function updateItemPanelEX(itemId = "", onUpdateDone = null) {
    const API = "https://i.kunok.com/api62.php";

    function renderForm(item) {
        let boardObj = {};
        if (item.board && typeof item.board === "string") {
            try {
                boardObj = JSON.parse(item.board);
            } catch {
                boardObj = {};
            }
        }

        const totalPercent = boardObj.totalRequired && boardObj.answeredRequired ? 
            (boardObj.answeredRequired / boardObj.totalRequired * 100).toFixed(1) : '0';
        const bucketPercent = boardObj.bucket_totalRequired && boardObj.bucket_answeredRequired ? 
            (boardObj.bucket_answeredRequired / boardObj.bucket_totalRequired * 100).toFixed(1) : '0';
        const cupPercent = boardObj.cup_totalRequired && boardObj.cup_answeredRequired ? 
            (boardObj.cup_answeredRequired / boardObj.cup_totalRequired * 100).toFixed(1) : '0';

        return `
            <style>
                .form-row { display: flex; gap: 10px; }
                .form-row > div { flex: 1; }
                textarea, input[type="text"], input[type="number"], select {
                    width: 100%;
                    font-family: monospace;
                    font-size: 13px;
                    padding: 6px;
                    box-sizing: border-box;
                }
                textarea {
                    min-height: 100px;
                    resize: vertical;
                }
                .actions {
                    margin-top: 10px;
                    display: flex;
                    gap: 10px;
                    flex-wrap: wrap;
                }
                .progress-bar {
                    background: #f0f0f0;
                    height: 8px;
                    border-radius: 4px;
                    overflow: hidden;
                    margin-top: 4px;
                }
                .progress-fill {
                    height: 100%;
                    background: #4caf50;
                    transition: width 0.3s;
                }
                .message { margin-top: 10px; font-size: 13px; }
            </style>
            <form id="updateItemFormEX">
                <div>
                    <label for="itemId">ID:</label>
                    <input type="text" id="itemId" value="${item.id}" style="width: 100%;">
                </div>
                <div>
                    <label for="itemName">Name:</label>
                    <input type="text" id="itemName" value="${item.name || ''}" required>
                </div>
                <div class="form-row">
                    <div>
                        <label for="itemFlag">Flag:</label>
                        <select id="itemFlag">
                            <option value="1" ${item.flag == 1 ? "selected" : ""}>1 (Active)</option>
                            <option value="0" ${item.flag == 0 ? "selected" : ""}>0 (Hidden)</option>
                            <option value="3" ${item.flag == 3 ? "selected" : ""}>3 (Deleted)</option>
                        </select>
                    </div>
                    <div>
                        <label for="itemKeeper">Keeper:</label>
                        <input type="text" id="itemKeeper" value="${item.keeper || ''}">
                    </div>
                </div>
                <div>
                    <label for="itemBoard">Board (JSON):</label>
                    <textarea id="itemBoard">${item.board || ''}</textarea>
                </div>
                <div class="form-row">
                    <div>
                        <label for="itemUserId">User ID:</label>
                        <input type="text" id="itemUserId" value="${boardObj.userid || ''}">
                    </div>
                    <div>
                        <label for="itemUserName">Username:</label>
                        <input type="text" id="itemUserName" value="${boardObj.user_name || ''}">
                    </div>
                </div>
                <div class="form-row">
                    <div>
                        <label for="itemILevel">ILevel:</label>
                        <input type="text" id="itemILevel" value="${boardObj.ILevel || ''}">
                    </div>
                    <div>
                        <label for="itemItype">Itype:</label>
                        <input type="text" id="itemItype" value="${boardObj.Itype || ''}">
                    </div>
                </div>
                <div>
                    <label for="itemIcode">Icode:</label>
                    <input type="text" id="itemIcode" value="${boardObj.Icode || ''}">
                </div>
                <div class="form-row">
                    <div>
                        <label for="itemTotalRequired">Total Required:</label>
                        <input type="number" id="itemTotalRequired" value="${boardObj.totalRequired || '0'}">
                    </div>
                    <div>
                        <label for="itemAnsweredRequired">Answered Required:</label>
                        <input type="number" id="itemAnsweredRequired" value="${boardObj.answeredRequired || '0'}">
                    </div>
                </div>
                <div>
                    <label>Progress: ${totalPercent}%</label>
                    <div class="progress-bar"><div class="progress-fill" style="width: ${totalPercent}%"></div></div>
                </div>
                <div class="form-row">
                    <div>
                        <label for="itemBucketTotal">Bucket Total:</label>
                        <input type="number" id="itemBucketTotal" value="${boardObj.bucket_totalRequired || '0'}">
                    </div>
                    <div>
                        <label for="itemBucketAnswered">Bucket Answered:</label>
                        <input type="number" id="itemBucketAnswered" value="${boardObj.bucket_answeredRequired || '0'}">
                    </div>
                </div>
                <div>
                    <label>Bucket Progress: ${bucketPercent}%</label>
                    <div class="progress-bar"><div class="progress-fill" style="width: ${bucketPercent}%"></div></div>
                </div>
                <div class="form-row">
                    <div>
                        <label for="itemCupTotal">Cup Total:</label>
                        <input type="number" id="itemCupTotal" value="${boardObj.cup_totalRequired || '0'}">
                    </div>
                    <div>
                        <label for="itemCupAnswered">Cup Answered:</label>
                        <input type="number" id="itemCupAnswered" value="${boardObj.cup_answeredRequired || '0'}">
                    </div>
                </div>
                <div>
                    <label>Cup Progress: ${cupPercent}%</label>
                    <div class="progress-bar"><div class="progress-fill" style="width: ${cupPercent}%"></div></div>
                </div>
                <div class="form-row">
                    <div>
                        <label for="itemSaveStatus">Save Status:</label>
                        <input type="text" id="itemSaveStatus" value="${boardObj.save || ''}">
                    </div>
                    <div>
                        <label for="itemTime">Time:</label>
                        <input type="text" id="itemTime" value="${boardObj.time || ''}">
                    </div>
                </div>
                <div>
                    <label for="itemFirstTime">First Time:</label>
                    <input type="text" id="itemFirstTime" value="${boardObj.firstTime || ''}">
                </div>
                <div class="actions">
                    <button type="submit">Update Item</button>
                    <button type="button" id="assignToPlateBtn">Assign to Plate</button>
                    <button type="button" id="assignFromPlateBtn">Assign from Plate</button>
                    <button type="button" id="openUserListBtn">Open User List</button>
                </div>
            </form>
            <div id="lengthSummary" style="margin-top:10px; font-size:13px; color:#333;">
                <strong>Length Info:</strong><br>
                bucket: ${(item.bucket || "").length} | 
                cup: ${(item.cup || "").length} | 
                board: ${(item.board || "").length}
            </div>
            <div id="itemMessageEX" class="message"></div>
        `;
    }

    function displayMessage(msg, isError = false) {
        const box = document.getElementById("itemMessageEX");
        if (box) {
            box.textContent = msg;
            box.style.color = isError ? "red" : "green";
        }
    }

    function updateLengthSummary() {
        const summaryBox = document.getElementById("lengthSummary");
        if (summaryBox) {
            summaryBox.innerHTML = `
                <strong>Length Info:</strong><br>
                bucket: ${window.bucket_plate ? window.bucket_plate.length : 0} | 
                cup: ${window.cup_plate ? window.cup_plate.length : 0} | 
                board: ${document.getElementById("itemBoard")?.value.length || 0}
            `;
        }
    }

    async function fetchItemData(id) {
        const res = await fetch(`${API}?id=${encodeURIComponent(id)}`, { mode: 'cors' });
        if (!res.ok) throw new Error(`HTTP error! Status: ${res.status}`);
        const json = await res.json();
        if (!json.data) throw new Error("Item not found");
        return json.data;
    }

    if (!itemId) {
        const urlParams = new URLSearchParams(window.location.search);
        itemId = urlParams.get("id");
    }

    if (!itemId) {
        createPanel("updateItemPanelEX", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="message error">No item ID provided</div>`,
            noFooter: true,
            title: "Update Item"
        });
        return;
    }

    try {
        const original = await fetchItemData(itemId);

        createPanel("updateItemPanelEX", "CC", "ctc", 500, 900, {
            bodyContent: renderForm(original),
            noFooter: true,
            title: "Update Item"
        });

        setTimeout(() => {
            const form = document.getElementById("updateItemFormEX");
            if (form) {
                form.addEventListener("submit", async (e) => {
                    e.preventDefault();

                    const boardObj = {
                        userid: document.getElementById("itemUserId").value.trim(),
                        user_name: document.getElementById("itemUserName").value.trim(),
                        ILevel: document.getElementById("itemILevel").value.trim(),
                        Itype: document.getElementById("itemItype").value.trim(),
                        Icode: document.getElementById("itemIcode").value.trim(),
                        totalRequired: parseInt(document.getElementById("itemTotalRequired").value) || 0,
                        answeredRequired: parseInt(document.getElementById("itemAnsweredRequired").value) || 0,
                        bucket_totalRequired: parseInt(document.getElementById("itemBucketTotal").value) || 0,
                        bucket_answeredRequired: parseInt(document.getElementById("itemBucketAnswered").value) || 0,
                        cup_totalRequired: parseInt(document.getElementById("itemCupTotal").value) || 0,
                        cup_answeredRequired: parseInt(document.getElementById("itemCupAnswered").value) || 0,
                        save: document.getElementById("itemSaveStatus").value.trim(),
                        time: document.getElementById("itemTime").value.trim(),
                        firstTime: document.getElementById("itemFirstTime").value.trim()
                    };

                    const payload = {
                        id: document.getElementById("itemId").value.trim(),
                        name: document.getElementById("itemName").value.trim(),
                        flag: parseInt(document.getElementById("itemFlag").value),
                        keeper: document.getElementById("itemKeeper").value.trim(),
                        board: JSON.stringify(boardObj),
                        bucket: original.bucket || "",
                        cup: original.cup || ""
                    };

                    try {
                        const res = await fetch(API, {
                            method: "POST",
                            headers: { "Content-Type": "application/json" },
                            body: JSON.stringify({ data: payload })
                        });

                        const result = await res.json();
                        if (!res.ok || result.error) throw new Error(result.error || "Unknown error");

                        displayMessage(result.message || "Item updated");
                        updateLengthSummary();

                        if (typeof onUpdateDone === "function") {
                            onUpdateDone();
                        }
                    } catch (err) {
                        displayMessage("Error: " + err.message, true);
                    }
                });
            }

            const toBtn = document.getElementById("assignToPlateBtn");
            if (toBtn) {
                toBtn.addEventListener("click", () => {
                    window.bucket_plate = original.bucket || "";
                    window.cup_plate = original.cup || "";
                    alert("Assigned to plate:\n\nbucket_plate = " + window.bucket_plate + "\ncup_plate = " + window.cup_plate);
                    updateLengthSummary();
                });
            }

            const fromBtn = document.getElementById("assignFromPlateBtn");
            if (fromBtn) {
                fromBtn.addEventListener("click", () => {
                    original.bucket = window.bucket_plate || "";
                    original.cup = window.cup_plate || "";
                    alert("Assigned from plate:\n\nbucket = " + original.bucket + "\ncup = " + original.cup);
                    updateLengthSummary();
                });
            }

            const openListBtn = document.getElementById("openUserListBtn");
            if (openListBtn) {
                openListBtn.addEventListener("click", () => {
                    if (typeof userListPanel === "function") {
                        userListPanel();
                    } else {
                        alert("userListPanel() not available");
                    }
                });
            }
        }, 100);
    } catch (err) {
        console.error("Failed to load item:", err);
        createPanel("updateItemPanelEX", "CC", "ctc", 400, 200, {
            bodyContent: `<div class="message error">${err.message}</div>`,
            noFooter: true,
            title: "Update Item"
        });
    }
}